/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.protocol.internal;

import com.datastax.oss.protocol.internal.Compressor;
import com.datastax.oss.protocol.internal.Frame;
import com.datastax.oss.protocol.internal.Message;
import com.datastax.oss.protocol.internal.NoopCompressor;
import com.datastax.oss.protocol.internal.PrimitiveCodec;
import com.datastax.oss.protocol.internal.PrimitiveSizes;
import com.datastax.oss.protocol.internal.ProtocolErrors;
import com.datastax.oss.protocol.internal.ProtocolV3ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV3ServerCodecs;
import com.datastax.oss.protocol.internal.ProtocolV4ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV4ServerCodecs;
import com.datastax.oss.protocol.internal.ProtocolV5ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV5ServerCodecs;
import com.datastax.oss.protocol.internal.util.IntIntMap;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class FrameCodec<B> {
    private final PrimitiveCodec<B> primitiveCodec;
    private final Compressor<B> compressor;
    private final IntIntMap<Message.Codec> encoders;
    private final IntIntMap<Message.Codec> decoders;

    public static <B> FrameCodec<B> defaultClient(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor) {
        return new FrameCodec<B>(primitiveCodec, compressor, new ProtocolV3ClientCodecs(), new ProtocolV4ClientCodecs(), new ProtocolV5ClientCodecs());
    }

    public static <B> FrameCodec<B> defaultServer(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor) {
        return new FrameCodec<B>(primitiveCodec, compressor, new ProtocolV3ServerCodecs(), new ProtocolV4ServerCodecs(), new ProtocolV5ServerCodecs());
    }

    public FrameCodec(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor, CodecGroup ... codecGroups) {
        ProtocolErrors.check(primitiveCodec != null, "primitiveCodec can't be null", new Object[0]);
        ProtocolErrors.check(compressor != null, "compressor can't be null, use Compressor.none()", new Object[0]);
        this.primitiveCodec = primitiveCodec;
        this.compressor = compressor;
        final IntIntMap.Builder encodersBuilder = IntIntMap.builder();
        final IntIntMap.Builder decodersBuilder = IntIntMap.builder();
        CodecGroup.Registry registry = new CodecGroup.Registry(){

            @Override
            public CodecGroup.Registry addCodec(Message.Codec codec) {
                this.addEncoder(codec);
                this.addDecoder(codec);
                return this;
            }

            @Override
            public CodecGroup.Registry addEncoder(Message.Codec codec) {
                encodersBuilder.put(codec.protocolVersion, codec.opcode, codec);
                return this;
            }

            @Override
            public CodecGroup.Registry addDecoder(Message.Codec codec) {
                decodersBuilder.put(codec.protocolVersion, codec.opcode, codec);
                return this;
            }
        };
        for (CodecGroup codecGroup : codecGroups) {
            codecGroup.registerCodecs(registry);
        }
        this.encoders = encodersBuilder.build();
        this.decoders = decodersBuilder.build();
    }

    public B encode(Frame frame) {
        int protocolVersion = frame.protocolVersion;
        Message request = frame.message;
        ProtocolErrors.check(protocolVersion >= 4 || frame.customPayload.isEmpty(), "Custom payload is not supported in protocol v%d", protocolVersion);
        ProtocolErrors.check(protocolVersion >= 4 || frame.warnings.isEmpty(), "Warnings are not supported in protocol v%d", protocolVersion);
        int opcode = request.opcode;
        Message.Codec encoder = this.encoders.get(protocolVersion, opcode);
        ProtocolErrors.check(encoder != null, "Unsupported opcode %s in protocol v%d", opcode, protocolVersion);
        EnumSet<Flag> flags = EnumSet.noneOf(Flag.class);
        if (!(this.compressor instanceof NoopCompressor) && opcode != 1) {
            flags.add(Flag.COMPRESSED);
        }
        if (frame.tracing || frame.tracingId != null) {
            flags.add(Flag.TRACING);
        }
        if (!frame.customPayload.isEmpty()) {
            flags.add(Flag.CUSTOM_PAYLOAD);
        }
        if (!frame.warnings.isEmpty()) {
            flags.add(Flag.WARNING);
        }
        if (protocolVersion == 5) {
            flags.add(Flag.USE_BETA);
        }
        int headerSize = FrameCodec.headerEncodedSize();
        if (!flags.contains((Object)Flag.COMPRESSED)) {
            int messageSize = encoder.encodedSize(request);
            if (frame.tracingId != null) {
                messageSize += 16;
            }
            if (!frame.customPayload.isEmpty()) {
                messageSize += PrimitiveSizes.sizeOfBytesMap(frame.customPayload);
            }
            if (!frame.warnings.isEmpty()) {
                messageSize += PrimitiveSizes.sizeOfStringList(frame.warnings);
            }
            B dest = this.primitiveCodec.allocate(headerSize + messageSize);
            this.encodeHeader(frame, flags, messageSize, dest);
            this.encodeTracingId(frame.tracingId, dest);
            this.encodeCustomPayload(frame.customPayload, dest);
            this.encodeWarnings(frame.warnings, dest);
            encoder.encode(dest, request, this.primitiveCodec);
            return dest;
        }
        int uncompressedMessageSize = encoder.encodedSize(request);
        if (frame.tracingId != null) {
            uncompressedMessageSize += 16;
        }
        if (!frame.customPayload.isEmpty()) {
            uncompressedMessageSize += PrimitiveSizes.sizeOfBytesMap(frame.customPayload);
        }
        if (!frame.warnings.isEmpty()) {
            uncompressedMessageSize += PrimitiveSizes.sizeOfStringList(frame.warnings);
        }
        B uncompressedMessage = this.primitiveCodec.allocate(uncompressedMessageSize);
        this.encodeTracingId(frame.tracingId, uncompressedMessage);
        this.encodeCustomPayload(frame.customPayload, uncompressedMessage);
        this.encodeWarnings(frame.warnings, uncompressedMessage);
        encoder.encode(uncompressedMessage, request, this.primitiveCodec);
        B compressedMessage = this.compressor.compress(uncompressedMessage);
        this.primitiveCodec.release(uncompressedMessage);
        int messageSize = this.primitiveCodec.sizeOf(compressedMessage);
        B header = this.primitiveCodec.allocate(headerSize);
        this.encodeHeader(frame, flags, messageSize, header);
        return this.primitiveCodec.concat(header, compressedMessage);
    }

    private static int headerEncodedSize() {
        return 9;
    }

    private void encodeHeader(Frame frame, EnumSet<Flag> flags, int messageSize, B dest) {
        int versionAndDirection = frame.protocolVersion;
        if (frame.message.isResponse) {
            versionAndDirection |= 0x80;
        }
        this.primitiveCodec.writeByte((byte)versionAndDirection, dest);
        Flag.encode(flags, dest, this.primitiveCodec, frame.protocolVersion);
        this.primitiveCodec.writeUnsignedShort(frame.streamId & 0xFFFF, dest);
        this.primitiveCodec.writeByte((byte)frame.message.opcode, dest);
        this.primitiveCodec.writeInt(messageSize, dest);
    }

    private void encodeTracingId(UUID tracingId, B dest) {
        if (tracingId != null) {
            this.primitiveCodec.writeUuid(tracingId, dest);
        }
    }

    private void encodeCustomPayload(Map<String, ByteBuffer> customPayload, B dest) {
        if (!customPayload.isEmpty()) {
            this.primitiveCodec.writeBytesMap(customPayload, dest);
        }
    }

    private void encodeWarnings(List<String> warnings, B dest) {
        if (!warnings.isEmpty()) {
            this.primitiveCodec.writeStringList(warnings, dest);
        }
    }

    public Frame decode(B source) {
        B newSource;
        byte directionAndVersion = this.primitiveCodec.readByte(source);
        boolean isResponse = (directionAndVersion & 0x80) == 128;
        int protocolVersion = directionAndVersion & 0x7F;
        EnumSet<Flag> flags = Flag.decode(source, this.primitiveCodec, protocolVersion);
        boolean beta = flags.contains((Object)Flag.USE_BETA);
        int streamId = this.readStreamId(source);
        byte opcode = this.primitiveCodec.readByte(source);
        int length = this.primitiveCodec.readInt(source);
        int actualLength = this.primitiveCodec.sizeOf(source);
        ProtocolErrors.check(length == actualLength, "Declared length in header (%d) does not match actual length (%d)", length, actualLength);
        boolean decompressed = false;
        if (flags.contains((Object)Flag.COMPRESSED) && (newSource = this.compressor.decompress(source)) != source) {
            decompressed = true;
            source = newSource;
        }
        boolean isTracing = flags.contains((Object)Flag.TRACING);
        UUID tracingId = isResponse && isTracing ? this.primitiveCodec.readUuid(source) : null;
        Map<String, ByteBuffer> customPayload = flags.contains((Object)Flag.CUSTOM_PAYLOAD) ? this.primitiveCodec.readBytesMap(source) : Collections.emptyMap();
        List<String> warnings = isResponse && flags.contains((Object)Flag.WARNING) ? this.primitiveCodec.readStringList(source) : Collections.emptyList();
        Message.Codec decoder = this.decoders.get(protocolVersion, opcode);
        ProtocolErrors.check(decoder != null, "Unsupported request opcode: %s in protocol %d", opcode, protocolVersion);
        Message response = decoder.decode(source, this.primitiveCodec);
        if (decompressed) {
            this.primitiveCodec.release(source);
        }
        return new Frame(protocolVersion, beta, streamId, isTracing, tracingId, customPayload, warnings, response);
    }

    private int readStreamId(B source) {
        int id = this.primitiveCodec.readUnsignedShort(source);
        return (short)id;
    }

    public static interface CodecGroup {
        public void registerCodecs(Registry var1);

        public static interface Registry {
            public Registry addCodec(Message.Codec var1);

            public Registry addEncoder(Message.Codec var1);

            public Registry addDecoder(Message.Codec var1);
        }
    }

    static enum Flag {
        COMPRESSED(1),
        TRACING(2),
        CUSTOM_PAYLOAD(4),
        WARNING(8),
        USE_BETA(16);

        private int mask;

        private Flag(int mask) {
            this.mask = mask;
        }

        static <B> EnumSet<Flag> decode(B source, PrimitiveCodec<B> decoder, int protocolVersion) {
            byte bits = decoder.readByte(source);
            EnumSet<Flag> set = EnumSet.noneOf(Flag.class);
            for (Flag flag : Flag.values()) {
                if ((bits & flag.mask) == 0) continue;
                set.add(flag);
            }
            return set;
        }

        static <B> void encode(EnumSet<Flag> flags, B dest, PrimitiveCodec<B> encoder, int protocolVersion) {
            int bits = 0;
            for (Flag flag : flags) {
                bits |= flag.mask;
            }
            encoder.writeByte((byte)bits, dest);
        }
    }
}

