/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.dse.graph;

import com.datastax.driver.dse.geometry.Geometry;
import com.datastax.driver.dse.geometry.LineString;
import com.datastax.driver.dse.geometry.Point;
import com.datastax.driver.dse.geometry.Polygon;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import java.io.IOException;

/**
 * Default deserializer used by the driver for geospatial types. It deserializes such types into
 * {@link Geometry} instances. The actual subclass depends on the type being deserialized.
 */
class DefaultGeometryDeserializer<T extends Geometry> extends StdDeserializer<T> {

  private final Class<T> geometryClass;

  DefaultGeometryDeserializer(Class<T> geometryClass) {
    super(geometryClass);
    this.geometryClass = geometryClass;
  }

  @Override
  public T deserialize(JsonParser parser, DeserializationContext ctx) throws IOException {
    String wkt = parser.readValueAs(String.class);
    Geometry geometry;
    if (wkt.startsWith("POINT")) geometry = Point.fromWellKnownText(wkt);
    else if (wkt.startsWith("LINESTRING")) geometry = LineString.fromWellKnownText(wkt);
    else if (wkt.startsWith("POLYGON")) geometry = Polygon.fromWellKnownText(wkt);
    else throw new JsonParseException(parser, "Unknown geometry type: " + wkt);
    return geometryClass.cast(geometry);
  }
}
