/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.dse;

import com.datastax.driver.core.Configuration;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.Policies;
import com.datastax.driver.dse.graph.GraphOptions;
import java.util.UUID;

/**
 * The configuration of a {@link DseCluster}.
 *
 * <p>This class extends the CQL driver's {@link Configuration} to add DSE-specific options.
 */
public class DseConfiguration extends Configuration {

  private final GraphOptions graphOptions;
  private final String applicationName;
  private final String applicationVersion;
  private final UUID clientId;

  DseConfiguration(
      Configuration toCopy,
      GraphOptions graphOptions,
      String applicationName,
      String applicationVersion,
      UUID clientId) {
    super(toCopy);
    this.graphOptions = graphOptions;
    this.applicationName = applicationName;
    this.applicationVersion = applicationVersion;
    this.clientId = clientId;
  }

  /**
   * Returns the default graph options to use for the cluster.
   *
   * @return the default graph options.
   */
  public GraphOptions getGraphOptions() {
    return graphOptions;
  }

  /** @return the application name to use for the cluster, or null if not set. */
  public String getApplicationName() {
    return applicationName;
  }

  /** @return the application version to use for the cluster, or null if not set. */
  public String getApplicationVersion() {
    return applicationVersion;
  }

  /**
   * @return the id assigned to the cluster, or null if it should be generated internally.
   * @deprecated Use {@link #getClientId()} instead. This method was intended to return a client id
   *     not cluster id from the beginning. The getClientId name is more appropriate.
   */
  @Deprecated
  public UUID getClusterId() {
    return getClientId();
  }

  public UUID getClientId() {
    return clientId;
  }

  /**
   * Builds an instance of the default load balancing policy used for DSE.
   *
   * <p>It is the core driver's default policy (as returned by {@link
   * Policies#defaultLoadBalancingPolicy()}), wrapped into a {@link DseLoadBalancingPolicy}.
   *
   * @return the new instance.
   */
  public static LoadBalancingPolicy defaultLoadBalancingPolicy() {
    return new DseLoadBalancingPolicy(Policies.defaultLoadBalancingPolicy());
  }
}
