/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.core.utils;

import com.datastax.driver.core.GuavaCompatibility;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;

/** Helpers to work with Guava's {@link ListenableFuture}. */
public class MoreFutures {
  /** An immediate successful {@code ListenableFuture<Void>}. */
  public static final ListenableFuture<Void> VOID_SUCCESS = Futures.immediateFuture(null);

  /** A {@link FutureCallback} that does nothing on failure. */
  public abstract static class SuccessCallback<V> implements FutureCallback<V> {
    @Override
    public void onFailure(Throwable t) {
      /* nothing */
    }
  }

  /** A {@link FutureCallback} that does nothing on success. */
  public abstract static class FailureCallback<V> implements FutureCallback<V> {
    @Override
    public void onSuccess(V result) {
      /* nothing */
    }
  }

  /**
   * Configures a {@link SettableFuture} to propagate the result of a future.
   *
   * @param settable future to be propagated to
   * @param future future to propagate
   * @param <T>
   */
  public static <T> void propagateFuture(
      final SettableFuture<T> settable, ListenableFuture<T> future) {
    GuavaCompatibility.INSTANCE.addCallback(
        future,
        new FutureCallback<T>() {
          @Override
          public void onSuccess(T result) {
            settable.set(result);
          }

          @Override
          public void onFailure(Throwable t) {
            settable.setException(t);
          }
        });
  }
}
