/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.core;

import com.datastax.driver.core.utils.MoreObjects;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.util.List;
import java.util.Map;
import java.util.Set;

class InsightsSchema {

  @JsonIgnoreProperties(ignoreUnknown = true)
  @JsonInclude(JsonInclude.Include.NON_EMPTY)
  static class Insight<T> {
    @JsonProperty("metadata")
    private final InsightsMetadata metadata;

    @JsonProperty("data")
    private final T insightsData;

    @JsonCreator
    Insight(@JsonProperty("metadata") InsightsMetadata metadata, @JsonProperty("data") T data) {
      this.metadata = metadata;
      this.insightsData = data;
    }

    InsightsMetadata getMetadata() {
      return metadata;
    }

    T getInsightsData() {
      return insightsData;
    }

    @Override
    public String toString() {
      return "Insight{" + "metadata=" + metadata + ", insightsData=" + insightsData + '}';
    }
  }

  static class InsightsMetadata {
    @JsonProperty("name")
    private final String name;

    @JsonProperty("timestamp")
    private final long timestamp;

    @JsonProperty("tags")
    private final Map<String, String> tags;

    @JsonProperty("insightType")
    private final InsightType insightType;

    @JsonProperty("insightMappingId")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private String insightMappingId;

    @JsonCreator
    InsightsMetadata(
        @JsonProperty("name") String name,
        @JsonProperty("timestamp") long timestamp,
        @JsonProperty("tags") Map<String, String> tags,
        @JsonProperty("insightType") InsightType insightType,
        @JsonProperty("insightMappingId") String insightMappingId) {
      Preconditions.checkArgument(!Strings.isNullOrEmpty(name), "name is required");

      this.name = name;
      this.timestamp = timestamp;
      this.tags = tags;
      this.insightType = insightType;
      this.insightMappingId = insightMappingId;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      InsightsMetadata that = (InsightsMetadata) other;
      return MoreObjects.equal(name, that.name)
          && MoreObjects.equal(timestamp, that.timestamp)
          && MoreObjects.equal(tags, that.tags)
          && MoreObjects.equal(insightType, that.insightType)
          && MoreObjects.equal(insightMappingId, that.insightMappingId);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(name, timestamp, tags, insightType, insightMappingId);
    }

    @Override
    public String toString() {
      return "InsightsMetadata{"
          + "name='"
          + name
          + '\''
          + ", timestamp="
          + timestamp
          + ", tags="
          + tags
          + ", insightType="
          + insightType
          + ", insightMappingId="
          + insightMappingId
          + '}';
    }

    String getName() {
      return name;
    }

    long getTimestamp() {
      return timestamp;
    }

    Map<String, String> getTags() {
      return tags;
    }

    InsightType getInsightType() {
      return insightType;
    }

    String getInsightMappingId() {
      return insightMappingId;
    }
  }

  static class AuthProviderType {
    @JsonProperty("type")
    private final String type;

    @JsonProperty("namespace")
    private final String namespace;

    @JsonCreator
    AuthProviderType(
        @JsonProperty("type") String type, @JsonProperty("namespace") String namespace) {
      this.type = type;
      this.namespace = namespace;
    }

    String getType() {
      return type;
    }

    public String getNamespace() {
      return namespace;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      AuthProviderType that = (AuthProviderType) other;
      return MoreObjects.equal(type, that.type) && MoreObjects.equal(namespace, that.namespace);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(type, namespace);
    }

    @Override
    public String toString() {
      return "AuthProviderType{"
          + "type='"
          + type
          + '\''
          + ", namespace='"
          + namespace
          + '\''
          + '}';
    }
  }

  static class PoolSizeByHostDistance {
    @JsonProperty("local")
    final int local;

    @JsonProperty("remote")
    final int remote;

    @JsonProperty("ignored")
    final int ignored;

    @JsonCreator
    PoolSizeByHostDistance(
        @JsonProperty("local") int local,
        @JsonProperty("remote") int remote,
        @JsonProperty("ignored") int ignored) {

      this.local = local;
      this.remote = remote;
      this.ignored = ignored;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      PoolSizeByHostDistance that = (PoolSizeByHostDistance) other;
      return MoreObjects.equal(local, that.local)
          && MoreObjects.equal(remote, that.remote)
          && MoreObjects.equal(ignored, that.ignored);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(local, remote, ignored);
    }
  }

  static class SpecificExecutionProfile {
    @JsonProperty("readTimeout")
    private final int readTimeout;

    @JsonProperty("loadBalancing")
    private final LoadBalancingInfo loadBalancing;

    @JsonProperty("speculativeExecution")
    private SpeculativeExecutionInfo speculativeExecution;

    @JsonProperty("consistency")
    private final String consistency;

    @JsonProperty("serialConsistency")
    private final String serialConsistency;

    @JsonProperty("graphOptions")
    private Map<String, Object> graphOptions;

    @JsonCreator
    SpecificExecutionProfile(
        @JsonProperty("readTimeout") int readTimeoutMillis,
        @JsonProperty("loadBalancing") LoadBalancingInfo loadBalancing,
        @JsonProperty("speculativeExecution") SpeculativeExecutionInfo speculativeExecutionInfo,
        @JsonProperty("consistency") String consistency,
        @JsonProperty("serialConsistency") String serialConsistency,
        @JsonProperty("graphOptions") Map<String, Object> graphOptions) {
      readTimeout = readTimeoutMillis;
      this.loadBalancing = loadBalancing;
      this.speculativeExecution = speculativeExecutionInfo;
      this.consistency = consistency;
      this.serialConsistency = serialConsistency;
      this.graphOptions = graphOptions;
    }

    int getReadTimeout() {
      return readTimeout;
    }

    LoadBalancingInfo getLoadBalancing() {
      return loadBalancing;
    }

    SpeculativeExecutionInfo getSpeculativeExecution() {
      return speculativeExecution;
    }

    String getConsistency() {
      return consistency;
    }

    String getSerialConsistency() {
      return serialConsistency;
    }

    Map<String, Object> getGraphOptions() {
      return graphOptions;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      SpecificExecutionProfile that = (SpecificExecutionProfile) other;
      return MoreObjects.equal(readTimeout, that.readTimeout)
          && MoreObjects.equal(loadBalancing, that.loadBalancing)
          && MoreObjects.equal(speculativeExecution, that.speculativeExecution)
          && MoreObjects.equal(consistency, that.consistency)
          && MoreObjects.equal(serialConsistency, that.serialConsistency)
          && MoreObjects.equal(graphOptions, that.graphOptions);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(
          readTimeout,
          loadBalancing,
          speculativeExecution,
          consistency,
          serialConsistency,
          graphOptions);
    }

    @Override
    public String toString() {
      return "SpecificExecutionProfile{"
          + "readTimeout="
          + readTimeout
          + ", loadBalancing="
          + loadBalancing
          + ", speculativeExecution="
          + speculativeExecution
          + ", consistency='"
          + consistency
          + '\''
          + ", serialConsistency='"
          + serialConsistency
          + '\''
          + ", graphOptions="
          + graphOptions
          + '}';
    }
  }

  static class LoadBalancingInfo {
    @JsonProperty("type")
    private final String type;

    @JsonProperty("options")
    private final Map<String, Object> options;

    @JsonProperty("namespace")
    private final String namespace;

    @JsonCreator
    LoadBalancingInfo(
        @JsonProperty("type") String type,
        @JsonProperty("namespace") String namespace,
        @JsonProperty("options") Map<String, Object> options) {
      this.type = type;
      this.namespace = namespace;
      this.options = options;
    }

    String getType() {
      return type;
    }

    Map<String, Object> getOptions() {
      return options;
    }

    public String getNamespace() {
      return namespace;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      LoadBalancingInfo that = (LoadBalancingInfo) other;
      return MoreObjects.equal(type, that.type)
          && MoreObjects.equal(options, that.options)
          && MoreObjects.equal(namespace, that.namespace);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(type, options, namespace);
    }

    @Override
    public String toString() {
      return "LoadBalancingInfo{"
          + "type='"
          + type
          + '\''
          + ", options="
          + options
          + ", namespace='"
          + namespace
          + '\''
          + '}';
    }
  }

  static class InsightsPlatformInfo {
    @JsonProperty("os")
    private final OS os;

    @JsonProperty("cpus")
    private CPUS cpus;

    /**
     * All dependencies in a map format grouped by the module: {"core" :
     * {"com.datastax.driver:core": {"runtimeVersion:" : "1.0.0", "compileVersion": "1.0.1"},...}},
     * "extras"" {...}
     */
    @JsonProperty("runtime")
    private Map<String, Map<String, RuntimeAndCompileTimeVersions>> runtime;

    @JsonCreator
    InsightsPlatformInfo(
        @JsonProperty("os") OS os,
        @JsonProperty("cpus") CPUS cpus,
        @JsonProperty("runtime") Map<String, Map<String, RuntimeAndCompileTimeVersions>> runtime) {
      this.os = os;
      this.cpus = cpus;
      this.runtime = runtime;
    }

    OS getOs() {
      return os;
    }

    CPUS getCpus() {
      return cpus;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      InsightsPlatformInfo that = (InsightsPlatformInfo) other;
      return MoreObjects.equal(os, that.os)
          && MoreObjects.equal(cpus, that.cpus)
          && MoreObjects.equal(runtime, that.runtime);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(os, cpus, runtime);
    }

    Map<String, Map<String, RuntimeAndCompileTimeVersions>> getRuntime() {
      return runtime;
    }

    static class OS {
      @JsonProperty("name")
      private final String name;

      @JsonProperty("version")
      private final String version;

      @JsonProperty("arch")
      private final String arch;

      @JsonCreator
      OS(
          @JsonProperty("name") String name,
          @JsonProperty("version") String version,
          @JsonProperty("arch") String arch) {
        this.name = name;
        this.version = version;
        this.arch = arch;
      }

      String getName() {
        return name;
      }

      String getVersion() {
        return version;
      }

      String getArch() {
        return arch;
      }

      @Override
      public boolean equals(Object other) {
        if (other == null || other.getClass() != this.getClass()) return false;

        OS that = (OS) other;
        return MoreObjects.equal(name, that.name)
            && MoreObjects.equal(version, that.version)
            && MoreObjects.equal(arch, that.arch);
      }

      @Override
      public int hashCode() {
        return MoreObjects.hashCode(name, version, arch);
      }
    }

    static class CPUS {
      @JsonProperty("length")
      private final int length;

      @JsonProperty("model")
      private final String model;

      @JsonCreator
      CPUS(@JsonProperty("length") int length, @JsonProperty("model") String model) {
        this.length = length;
        this.model = model;
      }

      int getLength() {
        return length;
      }

      String getModel() {
        return model;
      }

      @Override
      public boolean equals(Object other) {
        if (other == null || other.getClass() != this.getClass()) return false;

        CPUS that = (CPUS) other;
        return MoreObjects.equal(length, that.length) && MoreObjects.equal(model, that.model);
      }

      @Override
      public int hashCode() {
        return MoreObjects.hashCode(length, model);
      }
    }

    static class RuntimeAndCompileTimeVersions {
      @JsonProperty("runtimeVersion")
      private final String runtimeVersion;

      @JsonProperty("compileVersion")
      private final String compileVersion;

      @JsonProperty("optional")
      private final boolean optional;

      @JsonCreator
      RuntimeAndCompileTimeVersions(
          @JsonProperty("runtimeVersion") String runtimeVersion,
          @JsonProperty("compileVersion") String compileVersion,
          @JsonProperty("optional") boolean optional) {
        this.runtimeVersion = runtimeVersion;
        this.compileVersion = compileVersion;
        this.optional = optional;
      }

      String getRuntimeVersion() {
        return runtimeVersion;
      }

      String getCompileVersion() {
        return compileVersion;
      }

      public boolean isOptional() {
        return optional;
      }

      @Override
      public boolean equals(Object other) {
        if (other == null || other.getClass() != this.getClass()) return false;

        RuntimeAndCompileTimeVersions that = (RuntimeAndCompileTimeVersions) other;
        return MoreObjects.equal(runtimeVersion, that.runtimeVersion)
            && MoreObjects.equal(compileVersion, that.compileVersion)
            && MoreObjects.equal(optional, that.optional);
      }

      @Override
      public int hashCode() {
        return MoreObjects.hashCode(runtimeVersion, compileVersion, optional);
      }

      @Override
      public String toString() {
        return "RuntimeAndCompileTimeVersions{"
            + "runtimeVersion='"
            + runtimeVersion
            + '\''
            + ", compileVersion='"
            + compileVersion
            + '\''
            + ", optional="
            + optional
            + '}';
      }
    }
  }

  static class InsightsStartupData {
    @JsonProperty("clientId")
    private final String clientId;

    @JsonProperty("sessionId")
    private final String sessionId;

    @JsonProperty("applicationName")
    private final String applicationName;

    @JsonProperty("applicationVersion")
    private final String applicationVersion;

    @JsonProperty("contactPoints")
    private final Map<String, List<String>> contactPoints;

    @JsonProperty("initialControlConnection")
    private final String initialControlConnection;

    @JsonProperty("protocolVersion")
    private final int protocolVersion;

    @JsonProperty("localAddress")
    private final String localAddress;

    @JsonProperty("executionProfiles")
    private final Map<String, SpecificExecutionProfile> executionProfiles;

    @JsonProperty("poolSizeByHostDistance")
    private final PoolSizeByHostDistance poolSizeByHostDistance;

    @JsonProperty("heartbeatInterval")
    private final long heartbeatInterval;

    @JsonProperty("compression")
    private final ProtocolOptions.Compression compression;

    @JsonProperty("reconnectionPolicy")
    private final ReconnectionPolicyInfo reconnectionPolicy;

    @JsonProperty("ssl")
    private final SSL ssl;

    @JsonProperty("authProvider")
    private final AuthProviderType authProvider;

    @JsonProperty("otherOptions")
    private final Map<String, Object> otherOptions;

    @JsonProperty("configAntiPatterns")
    private final Map<String, String> configAntiPatterns;

    @JsonProperty("periodicStatusInterval")
    private final long periodicStatusInterval;

    @JsonProperty("platformInfo")
    private final InsightsPlatformInfo platformInfo;

    @JsonProperty("hostName")
    private final String hostName;

    @JsonProperty("driverName")
    private String driverName;

    @JsonProperty("applicationNameWasGenerated")
    private boolean applicationNameWasGenerated;

    @JsonProperty("driverVersion")
    private String driverVersion;

    @JsonProperty("dataCenters")
    private Set<String> dataCenters;

    @JsonCreator
    InsightsStartupData(
        @JsonProperty("clientId") String clientId,
        @JsonProperty("sessionId") String sessionId,
        @JsonProperty("applicationName") String applicationName,
        @JsonProperty("applicationVersion") String applicationVersion,
        @JsonProperty("contactPoints") Map<String, List<String>> contactPoints,
        @JsonProperty("initialControlConnection") String initialControlConnection,
        @JsonProperty("protocolVersion") int protocolVersion,
        @JsonProperty("localAddress") String localAddress,
        @JsonProperty("executionProfiles") Map<String, SpecificExecutionProfile> executionProfiles,
        @JsonProperty("poolSizeByHostDistance") PoolSizeByHostDistance poolSizeByHostDistance,
        @JsonProperty("heartbeatInterval") long heartbeatInterval,
        @JsonProperty("compression") ProtocolOptions.Compression compression,
        @JsonProperty("reconnectionPolicy") ReconnectionPolicyInfo reconnectionPolicy,
        @JsonProperty("ssl") SSL ssl,
        @JsonProperty("authProvider") AuthProviderType authProvider,
        @JsonProperty("otherOptions") Map<String, Object> otherOptions,
        @JsonProperty("configAntiPatterns") Map<String, String> configAntiPatterns,
        @JsonProperty("periodicStatusInterval") long periodicStatusInterval,
        @JsonProperty("platformInfo") InsightsPlatformInfo platformInfo,
        @JsonProperty("hostName") String hostName,
        @JsonProperty("driverName") String driverName,
        @JsonProperty("applicationNameWasGenerated") boolean applicationNameWasGenerated,
        @JsonProperty("driverVersion") String driverVersion,
        @JsonProperty("dataCenters") Set<String> dataCenters) {
      this.clientId = clientId;
      this.sessionId = sessionId;
      this.applicationName = applicationName;
      this.applicationVersion = applicationVersion;
      this.contactPoints = contactPoints;
      this.initialControlConnection = initialControlConnection;
      this.protocolVersion = protocolVersion;
      this.localAddress = localAddress;
      this.executionProfiles = executionProfiles;
      this.poolSizeByHostDistance = poolSizeByHostDistance;
      this.heartbeatInterval = heartbeatInterval;
      this.compression = compression;
      this.reconnectionPolicy = reconnectionPolicy;
      this.ssl = ssl;
      this.authProvider = authProvider;
      this.otherOptions = otherOptions;
      this.configAntiPatterns = configAntiPatterns;
      this.periodicStatusInterval = periodicStatusInterval;
      this.platformInfo = platformInfo;
      this.hostName = hostName;
      this.driverName = driverName;
      this.applicationNameWasGenerated = applicationNameWasGenerated;
      this.driverVersion = driverVersion;
      this.dataCenters = dataCenters;
    }

    String getClientId() {
      return clientId;
    }

    String getSessionId() {
      return sessionId;
    }

    String getApplicationName() {
      return applicationName;
    }

    String getApplicationVersion() {
      return applicationVersion;
    }

    Map<String, List<String>> getContactPoints() {
      return contactPoints;
    }

    String getInitialControlConnection() {
      return initialControlConnection;
    }

    int getProtocolVersion() {
      return protocolVersion;
    }

    String getLocalAddress() {
      return localAddress;
    }

    Map<String, SpecificExecutionProfile> getExecutionProfiles() {
      return executionProfiles;
    }

    PoolSizeByHostDistance getPoolSizeByHostDistance() {
      return poolSizeByHostDistance;
    }

    long getHeartbeatInterval() {
      return heartbeatInterval;
    }

    ProtocolOptions.Compression getCompression() {
      return compression;
    }

    ReconnectionPolicyInfo getReconnectionPolicy() {
      return reconnectionPolicy;
    }

    SSL getSsl() {
      return ssl;
    }

    AuthProviderType getAuthProvider() {
      return authProvider;
    }

    Map<String, Object> getOtherOptions() {
      return otherOptions;
    }

    Map<String, String> getConfigAntiPatterns() {
      return configAntiPatterns;
    }

    long getPeriodicStatusInterval() {
      return periodicStatusInterval;
    }

    InsightsPlatformInfo getPlatformInfo() {
      return platformInfo;
    }

    String getHostName() {
      return hostName;
    }

    String getDriverName() {
      return driverName;
    }

    boolean isApplicationNameWasGenerated() {
      return applicationNameWasGenerated;
    }

    String getDriverVersion() {
      return driverVersion;
    }

    Set<String> getDataCenters() {
      return dataCenters;
    }

    static InsightsStartupData.Builder builder() {
      return new InsightsStartupData.Builder();
    }

    static class Builder {
      private String clientId;
      private String sessionId;
      private String applicationName;
      private String applicationVersion;
      private Map<String, List<String>> contactPoints;
      private String initialControlConnection;
      private int protocolVersion;
      private String localAddress;
      private Map<String, SpecificExecutionProfile> executionProfiles;
      private PoolSizeByHostDistance poolSizeByHostDistance;
      private long heartbeatInterval;
      private ProtocolOptions.Compression compression;
      private ReconnectionPolicyInfo reconnectionPolicy;
      private SSL ssl;
      private AuthProviderType authProvider;
      private Map<String, Object> otherOptions;
      private Map<String, String> configAntiPatterns;
      private long periodicStatusInterval;
      private InsightsPlatformInfo platformInfo;
      private String hostName;
      private String driverName;
      private String driverVersion;
      private boolean applicationNameWasGenerated;
      private Set<String> dataCenters;

      InsightsStartupData build() {
        return new InsightsStartupData(
            clientId,
            sessionId,
            applicationName,
            applicationVersion,
            contactPoints,
            initialControlConnection,
            protocolVersion,
            localAddress,
            executionProfiles,
            poolSizeByHostDistance,
            heartbeatInterval,
            compression,
            reconnectionPolicy,
            ssl,
            authProvider,
            otherOptions,
            configAntiPatterns,
            periodicStatusInterval,
            platformInfo,
            hostName,
            driverName,
            applicationNameWasGenerated,
            driverVersion,
            dataCenters);
      }

      Builder withClientId(String clientId) {
        this.clientId = clientId;
        return this;
      }

      Builder withSessionId(String id) {
        this.sessionId = id;
        return this;
      }

      Builder withApplicationName(String applicationName) {
        this.applicationName = applicationName;
        return this;
      }

      Builder withApplicationVersion(String applicationVersion) {
        this.applicationVersion = applicationVersion;
        return this;
      }

      Builder withContactPoints(Map<String, List<String>> contactPoints) {
        this.contactPoints = contactPoints;
        return this;
      }

      Builder withInitialControlConnection(String inetSocketAddress) {
        this.initialControlConnection = inetSocketAddress;
        return this;
      }

      Builder withProtocolVersion(int protocolVersion) {
        this.protocolVersion = protocolVersion;
        return this;
      }

      Builder withLocalAddress(String localAddress) {
        this.localAddress = localAddress;
        return this;
      }

      Builder withExecutionProfiles(Map<String, SpecificExecutionProfile> executionProfiles) {
        this.executionProfiles = executionProfiles;
        return this;
      }

      Builder withPoolSizeByHostDistance(PoolSizeByHostDistance poolSizeByHostDistance) {
        this.poolSizeByHostDistance = poolSizeByHostDistance;
        return this;
      }

      Builder withHeartbeatInterval(long heartbeatInterval) {
        this.heartbeatInterval = heartbeatInterval;
        return this;
      }

      Builder withCompression(ProtocolOptions.Compression compression) {
        this.compression = compression;
        return this;
      }

      Builder withReconnectionPolicy(ReconnectionPolicyInfo reconnectionPolicy) {
        this.reconnectionPolicy = reconnectionPolicy;
        return this;
      }

      Builder withSsl(SSL ssl) {
        this.ssl = ssl;
        return this;
      }

      Builder withAuthProvider(AuthProviderType authProvider) {
        this.authProvider = authProvider;
        return this;
      }

      Builder withOtherOptions(Map<String, Object> otherOptions) {
        this.otherOptions = otherOptions;
        return this;
      }

      Builder withConfigAntiPatterns(Map<String, String> configAntiPatterns) {
        this.configAntiPatterns = configAntiPatterns;
        return this;
      }

      Builder withPeriodicStatusInterval(long periodicStatusInterval) {
        this.periodicStatusInterval = periodicStatusInterval;
        return this;
      }

      Builder withPlatformInfo(InsightsPlatformInfo insightsPlatformInfo) {
        this.platformInfo = insightsPlatformInfo;
        return this;
      }

      Builder withHostName(String hostName) {
        this.hostName = hostName;
        return this;
      }

      Builder withDriverName(String driverName) {
        this.driverName = driverName;
        return this;
      }

      Builder withDriverVersion(String driverVersion) {
        this.driverVersion = driverVersion;
        return this;
      }

      Builder withApplicationNameWasGenerated(boolean applicationNameWasGenerated) {
        this.applicationNameWasGenerated = applicationNameWasGenerated;
        return this;
      }

      Builder withDataCenters(Set<String> dataCenters) {
        this.dataCenters = dataCenters;
        return this;
      }
    }
  }

  static class InsightsStatusData {
    @JsonProperty("clientId")
    private final String clientId;

    @JsonProperty("sessionId")
    private final String sessionId;

    @JsonProperty("controlConnection")
    private final String controlConnection;

    @JsonProperty("connectedNodes")
    private final Map<String, SessionStateForNode> connectedNodes;

    @JsonCreator
    InsightsStatusData(
        @JsonProperty("clientId") String clientId,
        @JsonProperty("sessionId") String sessionId,
        @JsonProperty("controlConnection") String controlConnection,
        @JsonProperty("connectedNodes") Map<String, SessionStateForNode> connectedNodes) {
      this.clientId = clientId;
      this.sessionId = sessionId;
      this.controlConnection = controlConnection;
      this.connectedNodes = connectedNodes;
    }

    String getClientId() {
      return clientId;
    }

    String getSessionId() {
      return sessionId;
    }

    String getControlConnection() {
      return controlConnection;
    }

    Map<String, SessionStateForNode> getConnectedNodes() {
      return connectedNodes;
    }

    @Override
    public String toString() {
      return "InsightsStatusData{"
          + "clientId='"
          + clientId
          + '\''
          + ", sessionId='"
          + sessionId
          + '\''
          + ", controlConnection="
          + controlConnection
          + ", connectedNodes="
          + connectedNodes
          + '}';
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      InsightsStatusData that = (InsightsStatusData) other;
      return MoreObjects.equal(clientId, that.clientId)
          && MoreObjects.equal(sessionId, that.sessionId)
          && MoreObjects.equal(controlConnection, that.controlConnection)
          && MoreObjects.equal(connectedNodes, that.connectedNodes);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(clientId, sessionId, connectedNodes, connectedNodes);
    }

    static InsightsStatusData.Builder builder() {
      return new InsightsStatusData.Builder();
    }

    static class Builder {
      private String clientId;
      private String sessionId;
      private String controlConnection;
      private Map<String, SessionStateForNode> connectedNodes;

      Builder withClientId(String clientId) {
        this.clientId = clientId;
        return this;
      }

      Builder withSessionId(String id) {
        this.sessionId = id;
        return this;
      }

      Builder withControlConnection(String controlConnection) {
        this.controlConnection = controlConnection;
        return this;
      }

      Builder withConnectedNodes(Map<String, SessionStateForNode> connectedNodes) {
        this.connectedNodes = connectedNodes;
        return this;
      }

      InsightsStatusData build() {
        return new InsightsStatusData(clientId, sessionId, controlConnection, connectedNodes);
      }
    }
  }

  enum InsightType {
    EVENT,
    GAUGE,
    COUNTER,
    HISTOGRAM,
    TIMER,
    METER,
    LOG;
  }

  static class ReconnectionPolicyInfo {
    @JsonProperty("type")
    final String type;

    @JsonProperty("options")
    final Map<String, Object> options;

    @JsonProperty("namespace")
    private final String namespace;

    @JsonCreator
    ReconnectionPolicyInfo(
        @JsonProperty("type") String type,
        @JsonProperty("namespace") String namespace,
        @JsonProperty("options") Map<String, Object> options) {

      this.type = type;
      this.namespace = namespace;
      this.options = options;
    }

    String getType() {
      return type;
    }

    Map<String, Object> getOptions() {
      return options;
    }

    public String getNamespace() {
      return namespace;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      ReconnectionPolicyInfo that = (ReconnectionPolicyInfo) other;
      return MoreObjects.equal(type, that.type)
          && MoreObjects.equal(options, that.options)
          && MoreObjects.equal(namespace, that.namespace);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(type, options, namespace);
    }
  }

  static class SessionStateForNode {
    @JsonProperty("connections")
    private final Integer connections;

    @JsonProperty("inFlightQueries")
    private final Integer inFlightQueries;

    @JsonCreator
    SessionStateForNode(
        @JsonProperty("connections") Integer connections,
        @JsonProperty("inFlightQueries") Integer inFlightQueries) {
      this.connections = connections;
      this.inFlightQueries = inFlightQueries;
    }

    Integer getConnections() {
      return connections;
    }

    Integer getInFlightQueries() {
      return inFlightQueries;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      SessionStateForNode that = (SessionStateForNode) other;
      return MoreObjects.equal(connections, that.connections)
          && MoreObjects.equal(inFlightQueries, that.inFlightQueries);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(connections, inFlightQueries);
    }

    @Override
    public String toString() {
      return "SessionStateForNode{"
          + "connections="
          + connections
          + ", inFlightQueries="
          + inFlightQueries
          + '}';
    }
  }

  static class SpeculativeExecutionInfo {
    @JsonProperty("type")
    final String type;

    @JsonProperty("options")
    final Map<String, Object> options;

    @JsonProperty("namespace")
    final String namespace;

    @JsonCreator
    SpeculativeExecutionInfo(
        @JsonProperty("type") String type,
        @JsonProperty("namespace") String namespace,
        @JsonProperty("options") Map<String, Object> options) {
      this.type = type;
      this.namespace = namespace;
      this.options = options;
    }

    String getType() {
      return type;
    }

    Map<String, Object> getOptions() {
      return options;
    }

    public String getNamespace() {
      return namespace;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      SpeculativeExecutionInfo that = (SpeculativeExecutionInfo) other;
      return MoreObjects.equal(type, that.type)
          && MoreObjects.equal(options, that.options)
          && MoreObjects.equal(namespace, that.namespace);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(type, options, namespace);
    }

    @Override
    public String toString() {
      return "SpeculativeExecutionInfo{"
          + "type='"
          + type
          + '\''
          + ", options="
          + options
          + ", namespace='"
          + namespace
          + '\''
          + '}';
    }
  }

  static class SSL {
    @JsonProperty("enabled")
    final boolean enabled;

    @JsonCreator
    SSL(@JsonProperty("enabled") boolean enabled) {
      this.enabled = enabled;
    }

    boolean isEnabled() {
      return enabled;
    }

    @Override
    public boolean equals(Object other) {
      if (other == null || other.getClass() != this.getClass()) return false;

      SSL that = (SSL) other;
      return MoreObjects.equal(enabled, that.enabled);
    }

    @Override
    public int hashCode() {
      return MoreObjects.hashCode(enabled);
    }
  }
}
