/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.core;

import static com.datastax.driver.core.InsightsSchema.*;
import static com.datastax.driver.core.PackageUtil.getNamespace;

import com.datastax.driver.dse.DseCluster;
import com.datastax.driver.dse.graph.GraphOptions;
import java.util.HashMap;
import java.util.Map;

class ExecutionProfilesInfoFinder {
  private SpeculativeExecutionInfoFinder speculativeExecutionInfoFinder;
  private DataCentersFinder dataCentersFinder;

  ExecutionProfilesInfoFinder(
      SpeculativeExecutionInfoFinder speculativeExecutionInfoFinder,
      DataCentersFinder dataCentersFinder) {
    this.speculativeExecutionInfoFinder = speculativeExecutionInfoFinder;
    this.dataCentersFinder = dataCentersFinder;
  }

  Map<String, SpecificExecutionProfile> getExecutionProfilesInfo(Cluster cluster) {
    Map<String, SpecificExecutionProfile> executionProfiles =
        new HashMap<String, SpecificExecutionProfile>();
    Class<?> loadBalancingClass = cluster.getManager().loadBalancingPolicy().getClass();
    SpecificExecutionProfile defaultExecutionProfile =
        new SpecificExecutionProfile(
            cluster.getConfiguration().getSocketOptions().getReadTimeoutMillis(),
            new LoadBalancingInfo(
                cluster.getManager().loadBalancingPolicy().getClass().getSimpleName(),
                getNamespace(loadBalancingClass),
                maybeSetLocalDc(cluster)),
            speculativeExecutionInfoFinder.getSpeculativeExecutionInfo(
                cluster.getManager().speculativeExecutionPolicy()),
            cluster.getConfiguration().getQueryOptions().getConsistencyLevel().toString(),
            cluster.getConfiguration().getQueryOptions().getSerialConsistencyLevel().toString(),
            getGraphOptions(cluster));

    executionProfiles.put("default", defaultExecutionProfile);
    return executionProfiles;
  }

  private Map<String, Object> maybeSetLocalDc(Cluster cluster) {
    Map<String, Object> options = new HashMap<String, Object>();
    String localDataCenter = dataCentersFinder.getLocalDataCenter(cluster);
    if (localDataCenter != null) {
      options.put("localDataCenter", localDataCenter);
    }
    return options;
  }

  private Map<String, Object> getGraphOptions(Cluster cluster) {
    Map<String, Object> graphOptionsMap = new HashMap<String, Object>();
    if (cluster instanceof DseCluster) {
      DseCluster dseCluster = (DseCluster) cluster;
      GraphOptions graphOptions = dseCluster.getConfiguration().getGraphOptions();
      graphOptionsMap.put("language", graphOptions.getGraphLanguage());
      graphOptionsMap.put("source", graphOptions.getGraphSource());
    }
    return graphOptionsMap;
  }
}
