/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.driver.core;

import com.datastax.driver.core.policies.ChainableLoadBalancingPolicy;
import com.datastax.driver.core.policies.DCAwareRoundRobinPolicy;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.RetryPolicy;
import com.google.common.annotations.VisibleForTesting;
import java.util.HashMap;
import java.util.Map;

class ConfigAntiPatternsFinder {
  Map<String, String> findAntiPatterns(Cluster cluster) {
    Map<String, String> antiPatterns = new HashMap<String, String>();
    findLoadBalancingPolicyAntiPatterns(
        antiPatterns, cluster.getConfiguration().getPolicies().getLoadBalancingPolicy());
    findRetryPolicyAntiPatterns(
        antiPatterns, cluster.getConfiguration().getPolicies().getRetryPolicy());
    return antiPatterns;
  }

  @SuppressWarnings("deprecation")
  private void findRetryPolicyAntiPatterns(
      Map<String, String> antiPatterns, RetryPolicy retryPolicy) {
    if (retryPolicy
        instanceof com.datastax.driver.core.policies.DowngradingConsistencyRetryPolicy) {
      antiPatterns.put("downgradingConsistency", "Downgrading consistency retry policy in use");
    }
  }

  @VisibleForTesting
  void findLoadBalancingPolicyAntiPatterns(
      Map<String, String> antiPatterns, LoadBalancingPolicy loadBalancingPolicy) {
    if (loadBalancingPolicy instanceof ChainableLoadBalancingPolicy) {
      findLoadBalancingPolicyAntiPatterns(
          antiPatterns, ((ChainableLoadBalancingPolicy) loadBalancingPolicy).getChildPolicy());
    }
    if (loadBalancingPolicy instanceof DCAwareRoundRobinPolicy) {
      DCAwareRoundRobinPolicy dcAwareRoundRobinPolicy =
          (DCAwareRoundRobinPolicy) loadBalancingPolicy;
      if (dcAwareRoundRobinPolicy.isContactPointsMultipleDCs()) {
        antiPatterns.put(
            "contactPointsMultipleDCs", "Contact points contain hosts from multiple data centers");
      }

      if (dcAwareRoundRobinPolicy.isUseRemoteHostsForFailover()) {
        antiPatterns.put("useRemoteHosts", "Using remote hosts for failover");
      }
    }
  }
}
