/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.SessionManager;
import com.datastax.driver.core.SetKeyspaceException;
import com.datastax.driver.core.UnsupportedProtocolVersionException;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.utils.MoreFutures;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.FutureFallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HostConnectionPool
implements Connection.Owner {
    private static final Logger logger = LoggerFactory.getLogger(HostConnectionPool.class);
    private static final int MAX_SIMULTANEOUS_CREATION = 1;
    final Host host;
    volatile HostDistance hostDistance;
    protected final SessionManager manager;
    final List<Connection> connections;
    private final AtomicInteger open;
    final AtomicInteger totalInFlight = new AtomicInteger();
    private final AtomicInteger maxTotalInFlight = new AtomicInteger();
    @VisibleForTesting
    final Set<Connection> trash = new CopyOnWriteArraySet<Connection>();
    private volatile int waiter = 0;
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();
    private final Runnable newConnectionTask;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    protected final AtomicReference<CloseFuture> closeFuture = new AtomicReference();
    protected final AtomicReference<Phase> phase = new AtomicReference<Phase>(Phase.INITIALIZING);
    private final int minAllowedStreams;

    public HostConnectionPool(Host host, HostDistance hostDistance, SessionManager manager) {
        assert (hostDistance != HostDistance.IGNORED);
        this.host = host;
        this.hostDistance = hostDistance;
        this.manager = manager;
        this.newConnectionTask = new Runnable(){

            @Override
            public void run() {
                HostConnectionPool.this.addConnectionIfUnderMaximum();
                HostConnectionPool.this.scheduledForCreation.decrementAndGet();
            }
        };
        this.connections = new CopyOnWriteArrayList<Connection>();
        this.open = new AtomicInteger();
        this.minAllowedStreams = this.options().getMaxRequestsPerConnection(hostDistance) * 3 / 4;
    }

    ListenableFuture<Void> initAsync(Connection reusedConnection) {
        String keyspace = this.manager.poolsState.keyspace;
        Executor initExecutor = this.manager.cluster.manager.configuration.getPoolingOptions().getInitializationExecutor();
        final int coreSize = this.options().getCoreConnectionsPerHost(this.hostDistance);
        final ArrayList connections = Lists.newArrayListWithCapacity((int)coreSize);
        ArrayList connectionFutures = Lists.newArrayListWithCapacity((int)coreSize);
        for (int i = 0; i < coreSize; ++i) {
            ListenableFuture<Void> connectionFuture;
            Connection connection;
            if (reusedConnection != null && reusedConnection.setOwner(this)) {
                connection = reusedConnection;
                connectionFuture = MoreFutures.VOID_SUCCESS;
            } else {
                connection = this.manager.connectionFactory().newConnection(this);
                connectionFuture = connection.initAsync();
            }
            reusedConnection = null;
            connections.add(connection);
            connectionFutures.add(this.handleErrors(this.setKeyspaceAsync(connectionFuture, connection, keyspace), initExecutor));
        }
        ListenableFuture allConnectionsFuture = Futures.allAsList((Iterable)connectionFutures);
        final SettableFuture initFuture = SettableFuture.create();
        Futures.addCallback((ListenableFuture)allConnectionsFuture, (FutureCallback)new FutureCallback<List<Void>>(){

            public void onSuccess(List<Void> l) {
                ListIterator it = connections.listIterator();
                while (it.hasNext()) {
                    if (!((Connection)it.next()).isClosed()) continue;
                    it.remove();
                }
                HostConnectionPool.this.connections.addAll(connections);
                HostConnectionPool.this.open.set(connections.size());
                if (HostConnectionPool.this.isClosed()) {
                    initFuture.setException((Throwable)new ConnectionException(HostConnectionPool.this.host.getSocketAddress(), "Pool was closed during initialization"));
                    HostConnectionPool.this.forceClose(connections);
                } else {
                    logger.debug("Created connection pool to host {} ({} connections needed, {} successfully opened)", new Object[]{HostConnectionPool.this.host, coreSize, connections.size()});
                    HostConnectionPool.this.phase.compareAndSet(Phase.INITIALIZING, Phase.READY);
                    initFuture.set(null);
                }
            }

            public void onFailure(Throwable t) {
                HostConnectionPool.this.phase.compareAndSet(Phase.INITIALIZING, Phase.INIT_FAILED);
                HostConnectionPool.this.forceClose(connections);
                initFuture.setException(t);
            }
        }, (Executor)initExecutor);
        return initFuture;
    }

    private ListenableFuture<Void> handleErrors(ListenableFuture<Void> connectionInitFuture, Executor executor) {
        return Futures.withFallback(connectionInitFuture, (FutureFallback)new FutureFallback<Void>(){

            public ListenableFuture<Void> create(Throwable t) throws Exception {
                Throwables.propagateIfInstanceOf((Throwable)t, ClusterNameMismatchException.class);
                Throwables.propagateIfInstanceOf((Throwable)t, UnsupportedProtocolVersionException.class);
                Throwables.propagateIfInstanceOf((Throwable)t, SetKeyspaceException.class);
                Throwables.propagateIfInstanceOf((Throwable)t, Error.class);
                return MoreFutures.VOID_SUCCESS;
            }
        }, (Executor)executor);
    }

    private ListenableFuture<Void> setKeyspaceAsync(ListenableFuture<Void> initFuture, final Connection connection, final String keyspace) {
        return keyspace == null ? initFuture : Futures.transform(initFuture, (AsyncFunction)new AsyncFunction<Void, Void>(){

            public ListenableFuture<Void> apply(Void input) throws Exception {
                return connection.setKeyspaceAsync(keyspace);
            }
        });
    }

    private void forceClose(List<Connection> connections) {
        for (Connection connection : connections) {
            connection.closeAsync().force();
        }
    }

    private PoolingOptions options() {
        return this.manager.configuration().getPoolingOptions();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        int currentCapacity;
        int oldMax;
        Phase phase = this.phase.get();
        if (phase != Phase.READY) {
            throw new ConnectionException(this.host.getSocketAddress(), "Pool is " + (Object)((Object)phase));
        }
        if (this.connections.isEmpty()) {
            if (!this.host.convictionPolicy.canReconnectNow()) {
                throw new TimeoutException("Connection pool is empty, currently trying to reestablish connections");
            }
            int coreSize = this.options().getCoreConnectionsPerHost(this.hostDistance);
            if (coreSize == 0) {
                this.maybeSpawnNewConnection();
            } else {
                for (int i = 0; i < coreSize; ++i) {
                    this.scheduledForCreation.incrementAndGet();
                    this.manager.blockingExecutor().submit(this.newConnectionTask);
                }
            }
            Connection c = this.waitForConnection(timeout, unit);
            this.totalInFlight.incrementAndGet();
            c.setKeyspace(this.manager.poolsState.keyspace);
            return c;
        }
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight = connection.inFlight.get();
            if (inFlight >= minInFlight) continue;
            minInFlight = inFlight;
            leastBusy = connection;
        }
        if (leastBusy == null) {
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown");
            }
            leastBusy = this.waitForConnection(timeout, unit);
        } else {
            int inFlight;
            do {
                if ((inFlight = leastBusy.inFlight.get()) < Math.min(leastBusy.maxAvailableStreams(), this.options().getMaxRequestsPerConnection(this.hostDistance))) continue;
                leastBusy = this.waitForConnection(timeout, unit);
                break;
            } while (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1));
        }
        int totalInFlightCount = this.totalInFlight.incrementAndGet();
        while (totalInFlightCount > (oldMax = this.maxTotalInFlight.get()) && !this.maxTotalInFlight.compareAndSet(oldMax, totalInFlightCount)) {
        }
        int connectionCount = this.open.get() + this.scheduledForCreation.get();
        if (connectionCount < this.options().getCoreConnectionsPerHost(this.hostDistance)) {
            this.maybeSpawnNewConnection();
        } else if (connectionCount < this.options().getMaxConnectionsPerHost(this.hostDistance) && totalInFlightCount > (currentCapacity = (connectionCount - 1) * this.options().getMaxRequestsPerConnection(this.hostDistance) + this.options().getNewConnectionThreshold(this.hostDistance))) {
            this.maybeSpawnNewConnection();
        }
        leastBusy.setKeyspace(this.manager.poolsState.keyspace);
        return leastBusy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        this.waitLock.lock();
        ++this.waiter;
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            --this.waiter;
            this.waitLock.unlock();
        }
    }

    private void signalAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private void signalAllAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        if (timeout == 0L) {
            throw new TimeoutException("All connections are busy and pool timeout is 0");
        }
        long start = System.nanoTime();
        long remaining = timeout;
        do {
            int inFlight;
            try {
                this.awaitAvailableConnection(remaining, unit);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                timeout = 0L;
            }
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown");
            }
            int minInFlight = Integer.MAX_VALUE;
            Connection leastBusy = null;
            for (Connection connection : this.connections) {
                int inFlight2 = connection.inFlight.get();
                if (inFlight2 >= minInFlight) continue;
                minInFlight = inFlight2;
                leastBusy = connection;
            }
            if (leastBusy == null) continue;
            while ((inFlight = leastBusy.inFlight.get()) < Math.min(leastBusy.maxAvailableStreams(), this.options().getMaxRequestsPerConnection(this.hostDistance))) {
                if (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1)) continue;
                return leastBusy;
            }
        } while ((remaining = timeout - Cluster.timeSince(start, unit)) > 0L);
        throw new TimeoutException("All connections are busy");
    }

    public void returnConnection(Connection connection) {
        connection.inFlight.decrementAndGet();
        this.totalInFlight.decrementAndGet();
        if (this.isClosed()) {
            this.close(connection);
            return;
        }
        if (connection.isDefunct()) {
            return;
        }
        if (connection.state.get() != Connection.State.TRASHED) {
            if (connection.maxAvailableStreams() < this.minAllowedStreams) {
                this.replaceConnection(connection);
            } else {
                this.signalAvailableConnection();
            }
        }
    }

    private void replaceConnection(Connection connection) {
        if (!connection.state.compareAndSet(Connection.State.OPEN, Connection.State.TRASHED)) {
            return;
        }
        this.open.decrementAndGet();
        this.maybeSpawnNewConnection();
        connection.maxIdleTime = Long.MIN_VALUE;
        this.doTrashConnection(connection);
    }

    private boolean trashConnection(Connection connection) {
        int opened;
        if (!connection.state.compareAndSet(Connection.State.OPEN, Connection.State.TRASHED)) {
            return true;
        }
        do {
            if ((opened = this.open.get()) > this.options().getCoreConnectionsPerHost(this.hostDistance)) continue;
            connection.state.set(Connection.State.OPEN);
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        logger.trace("Trashing {}", (Object)connection);
        connection.maxIdleTime = System.currentTimeMillis() + (long)(this.options().getIdleTimeoutSeconds() * 1000);
        this.doTrashConnection(connection);
        return true;
    }

    private void doTrashConnection(Connection connection) {
        this.connections.remove(connection);
        this.trash.add(connection);
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.options().getMaxConnectionsPerHost(this.hostDistance)) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.phase.get() != Phase.READY) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            Connection newConnection = this.tryResurrectFromTrash();
            if (newConnection == null) {
                if (!this.host.convictionPolicy.canReconnectNow()) {
                    this.open.decrementAndGet();
                    return false;
                }
                logger.debug("Creating new connection on busy pool to {}", (Object)this.host);
                newConnection = this.manager.connectionFactory().open(this);
                newConnection.setKeyspace(this.manager.poolsState.keyspace);
            }
            this.connections.add(newConnection);
            newConnection.state.compareAndSet(Connection.State.RESURRECTING, Connection.State.OPEN);
            if (this.isClosed() && !newConnection.isClosed()) {
                this.close(newConnection);
                this.open.decrementAndGet();
                return false;
            }
            this.signalAvailableConnection();
            return true;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.open.decrementAndGet();
            return false;
        }
        catch (ConnectionException e) {
            this.open.decrementAndGet();
            logger.debug("Connection error to {} while creating additional connection", (Object)this.host);
            return false;
        }
        catch (AuthenticationException e) {
            this.open.decrementAndGet();
            logger.error("Authentication error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (UnsupportedProtocolVersionException e) {
            this.open.decrementAndGet();
            logger.error("UnsupportedProtocolVersionException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (ClusterNameMismatchException e) {
            this.open.decrementAndGet();
            logger.error("ClusterNameMismatchException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
    }

    private Connection tryResurrectFromTrash() {
        long highestMaxIdleTime = System.currentTimeMillis();
        Connection chosen = null;
        do {
            for (Connection connection : this.trash) {
                if (connection.maxIdleTime <= highestMaxIdleTime || connection.maxAvailableStreams() <= this.minAllowedStreams) continue;
                chosen = connection;
                highestMaxIdleTime = connection.maxIdleTime;
            }
            if (chosen != null) continue;
            return null;
        } while (!chosen.state.compareAndSet(Connection.State.TRASHED, Connection.State.RESURRECTING));
        logger.trace("Resurrecting {}", chosen);
        this.trash.remove(chosen);
        return chosen;
    }

    private void maybeSpawnNewConnection() {
        int inCreation;
        if (!this.host.convictionPolicy.canReconnectNow()) {
            return;
        }
        do {
            if ((inCreation = this.scheduledForCreation.get()) < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.manager.blockingExecutor().submit(this.newConnectionTask);
    }

    @Override
    public void onConnectionDefunct(Connection connection) {
        if (connection.state.compareAndSet(Connection.State.OPEN, Connection.State.GONE)) {
            this.open.decrementAndGet();
        }
        this.connections.remove(connection);
    }

    void cleanupIdleConnections(long now) {
        if (this.isClosed()) {
            return;
        }
        this.shrinkIfBelowCapacity();
        this.cleanupTrash(now);
    }

    private void shrinkIfBelowCapacity() {
        int currentLoad = this.maxTotalInFlight.getAndSet(this.totalInFlight.get());
        int maxRequestsPerConnection = this.options().getMaxRequestsPerConnection(this.hostDistance);
        int needed = currentLoad / maxRequestsPerConnection + 1;
        if (currentLoad % maxRequestsPerConnection > this.options().getNewConnectionThreshold(this.hostDistance)) {
            ++needed;
        }
        needed = Math.max(needed, this.options().getCoreConnectionsPerHost(this.hostDistance));
        int actual = this.open.get();
        int toTrash = Math.max(0, actual - needed);
        logger.trace("Current inFlight = {}, {} connections needed, {} connections available, trashing {}", new Object[]{currentLoad, needed, actual, toTrash});
        if (toTrash <= 0) {
            return;
        }
        for (Connection connection : this.connections) {
            if (!this.trashConnection(connection) || --toTrash != 0) continue;
            return;
        }
    }

    private void cleanupTrash(long now) {
        for (Connection connection : this.trash) {
            if (connection.maxIdleTime >= now || !connection.state.compareAndSet(Connection.State.TRASHED, Connection.State.GONE)) continue;
            if (connection.inFlight.get() == 0) {
                logger.trace("Cleaning up {}", (Object)connection);
                this.trash.remove(connection);
                this.close(connection);
                continue;
            }
            connection.state.set(Connection.State.TRASHED);
        }
    }

    private void close(Connection connection) {
        connection.closeAsync();
    }

    public final boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public final CloseFuture closeAsync() {
        CloseFuture future = this.closeFuture.get();
        if (future != null) {
            return future;
        }
        this.phase.set(Phase.CLOSING);
        this.signalAllAvailableConnection();
        future = new CloseFuture.Forwarding(this.discardAvailableConnections());
        return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
    }

    public int opened() {
        return this.open.get();
    }

    int trashed() {
        return this.trash.size();
    }

    private List<CloseFuture> discardAvailableConnections() {
        ArrayList<CloseFuture> futures = new ArrayList<CloseFuture>(this.connections.size() + this.trash.size());
        for (final Connection connection : this.connections) {
            CloseFuture future = connection.closeAsync();
            future.addListener(new Runnable(){

                @Override
                public void run() {
                    if (connection.state.compareAndSet(Connection.State.OPEN, Connection.State.GONE)) {
                        HostConnectionPool.this.open.decrementAndGet();
                    }
                }
            }, (Executor)MoreExecutors.sameThreadExecutor());
            futures.add(future);
        }
        for (final Connection connection : this.trash) {
            futures.add(connection.closeAsync());
        }
        return futures;
    }

    public void ensureCoreConnections() {
        int opened;
        if (this.isClosed()) {
            return;
        }
        if (!this.host.convictionPolicy.canReconnectNow()) {
            return;
        }
        for (int i = opened = this.open.get(); i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
            this.scheduledForCreation.incrementAndGet();
            this.manager.blockingExecutor().submit(this.newConnectionTask);
        }
    }

    static class PoolState {
        volatile String keyspace;

        PoolState(String keyspace) {
            this.keyspace = keyspace;
        }

        void setKeyspace(String keyspace) {
            this.keyspace = keyspace;
        }
    }

    private static enum Phase {
        INITIALIZING,
        READY,
        INIT_FAILED,
        CLOSING;

    }
}

