/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Connection;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.exceptions.AuthenticationException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HostConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(HostConnectionPool.class);
    private static final int MAX_SIMULTANEOUS_CREATION = 1;
    public final Host host;
    public volatile HostDistance hostDistance;
    private final Session.Manager manager;
    private final List<Connection> connections;
    private final AtomicInteger open;
    private final AtomicBoolean isShutdown = new AtomicBoolean();
    private final Set<Connection> trash = new CopyOnWriteArraySet<Connection>();
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();
    private final Runnable newConnectionTask;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();

    public HostConnectionPool(Host host, HostDistance hostDistance, Session.Manager manager) throws ConnectionException {
        this.host = host;
        this.hostDistance = hostDistance;
        this.manager = manager;
        this.newConnectionTask = new Runnable(){

            @Override
            public void run() {
                HostConnectionPool.this.addConnectionIfUnderMaximum();
                HostConnectionPool.this.scheduledForCreation.decrementAndGet();
            }
        };
        ArrayList<Connection> l = new ArrayList<Connection>(this.options().getCoreConnectionsPerHost(hostDistance));
        try {
            for (int i = 0; i < this.options().getCoreConnectionsPerHost(hostDistance); ++i) {
                l.add(manager.connectionFactory().open(host));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.connections = new CopyOnWriteArrayList<Connection>(l);
        this.open = new AtomicInteger(this.connections.size());
        logger.trace("Created connection pool to host {}", (Object)host);
    }

    private PoolingOptions options() {
        return this.manager.configuration().getPoolingOptions();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        int inFlight;
        if (this.isShutdown.get()) {
            throw new ConnectionException(this.host.getAddress(), "Pool is shutdown");
        }
        if (this.connections.isEmpty()) {
            for (int i = 0; i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
                this.scheduledForCreation.incrementAndGet();
                this.manager.executor().submit(this.newConnectionTask);
            }
            Connection c = this.waitForConnection(timeout, unit);
            c.setKeyspace(this.manager.poolsState.keyspace);
            return c;
        }
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight2 = connection.inFlight.get();
            if (inFlight2 >= minInFlight) continue;
            minInFlight = inFlight2;
            leastBusy = connection;
        }
        if (minInFlight >= this.options().getMaxSimultaneousRequestsPerConnectionTreshold(this.hostDistance) && this.connections.size() < this.options().getMaxConnectionPerHost(this.hostDistance)) {
            this.maybeSpawnNewConnection();
        }
        do {
            if ((inFlight = leastBusy.inFlight.get()) < 128) continue;
            leastBusy = this.waitForConnection(timeout, unit);
            break;
        } while (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1));
        leastBusy.setKeyspace(this.manager.poolsState.keyspace);
        return leastBusy;
    }

    private static long elapsed(long start, TimeUnit unit) {
        return unit.convert(System.currentTimeMillis() - start, TimeUnit.MILLISECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            this.waitLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signalAvailableConnection() {
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signalAllAvailableConnection() {
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        long start = System.currentTimeMillis();
        long remaining = timeout;
        do {
            int inFlight;
            try {
                this.awaitAvailableConnection(remaining, unit);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                timeout = 0L;
            }
            if (this.isShutdown()) {
                throw new ConnectionException(this.host.getAddress(), "Pool is shutdown");
            }
            int minInFlight = Integer.MAX_VALUE;
            Connection leastBusy = null;
            for (Connection connection : this.connections) {
                int inFlight2 = connection.inFlight.get();
                if (inFlight2 >= minInFlight) continue;
                minInFlight = inFlight2;
                leastBusy = connection;
            }
            while ((inFlight = leastBusy.inFlight.get()) < 128) {
                if (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1)) continue;
                return leastBusy;
            }
        } while ((remaining = timeout - HostConnectionPool.elapsed(start, unit)) > 0L);
        throw new TimeoutException();
    }

    public void returnConnection(Connection connection) {
        int inFlight = connection.inFlight.decrementAndGet();
        if (connection.isDefunct()) {
            if (this.host.getMonitor().signalConnectionFailure(connection.lastException())) {
                this.shutdown();
            } else {
                this.replace(connection);
            }
        } else {
            if (this.trash.contains((Object)connection) && inFlight == 0) {
                if (this.trash.remove((Object)connection)) {
                    this.close(connection);
                }
                return;
            }
            if (this.connections.size() > this.options().getCoreConnectionsPerHost(this.hostDistance) && inFlight <= this.options().getMinSimultaneousRequestsPerConnectionTreshold(this.hostDistance)) {
                this.trashConnection(connection);
            } else {
                this.signalAvailableConnection();
            }
        }
    }

    private boolean trashConnection(Connection connection) {
        int opened;
        do {
            if ((opened = this.open.get()) > this.options().getCoreConnectionsPerHost(this.hostDistance)) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        this.trash.add(connection);
        this.connections.remove((Object)connection);
        if (connection.inFlight.get() == 0 && this.trash.remove((Object)connection)) {
            this.close(connection);
        }
        return true;
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.options().getMaxConnectionPerHost(this.hostDistance)) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.isShutdown()) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            this.connections.add(this.manager.connectionFactory().open(this.host));
            this.signalAvailableConnection();
            return true;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.open.decrementAndGet();
            return false;
        }
        catch (ConnectionException e) {
            this.open.decrementAndGet();
            logger.debug("Connection error to {} while creating additional connection", (Object)this.host);
            if (this.host.getMonitor().signalConnectionFailure(e)) {
                this.shutdown();
            }
            return false;
        }
        catch (AuthenticationException e) {
            this.open.decrementAndGet();
            logger.error("Authentication error while creating additional connection (error is: {})", (Object)e.getMessage());
            this.shutdown();
            return false;
        }
    }

    private void maybeSpawnNewConnection() {
        int inCreation;
        do {
            if ((inCreation = this.scheduledForCreation.get()) < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        logger.debug("Creating new connection on busy pool to {}", (Object)this.host);
        this.manager.executor().submit(this.newConnectionTask);
    }

    private void replace(final Connection connection) {
        this.connections.remove((Object)connection);
        this.manager.executor().submit(new Runnable(){

            @Override
            public void run() {
                connection.close();
                HostConnectionPool.this.addConnectionIfUnderMaximum();
            }
        });
    }

    private void close(final Connection connection) {
        this.manager.executor().submit(new Runnable(){

            @Override
            public void run() {
                connection.close();
            }
        });
    }

    public boolean isShutdown() {
        return this.isShutdown.get();
    }

    public void shutdown() {
        if (!this.isShutdown.compareAndSet(false, true)) {
            return;
        }
        logger.debug("Shutting down pool");
        this.signalAllAvailableConnection();
        this.discardAvailableConnections();
    }

    public int opened() {
        return this.open.get();
    }

    private void discardAvailableConnections() {
        for (Connection connection : this.connections) {
            connection.close();
            this.open.decrementAndGet();
        }
    }

    public void ensureCoreConnections() {
        int opened;
        if (this.isShutdown()) {
            return;
        }
        for (int i = opened = this.open.get(); i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
            this.scheduledForCreation.incrementAndGet();
            this.manager.executor().submit(this.newConnectionTask);
        }
    }

    static class PoolState {
        volatile String keyspace;

        PoolState() {
        }

        public void setKeyspace(String keyspace) {
            this.keyspace = keyspace;
        }
    }
}

