/*
 * Decompiled with CFR 0.152.
 */
package datadog.trace.util;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class ClassNameTrie {
    private static final char LEAF_MARKER = '\u8000';
    private static final char BUD_MARKER = '\u4000';
    private static final char GLOB_MARKER = '\u2000';
    private static final char MAX_NODE_VALUE = '\u1fff';
    private static final char LONG_JUMP_MARKER = '\u8000';
    private static final int BRANCH_CONTROL_CHARS = 3;
    private static final int NO_END_JUMP = 1;
    private static final int FILE_MAGIC = -586583533;
    private final char[] trieData;
    private final int[] longJumps;

    public int apply(String key) {
        return ClassNameTrie.apply(this.trieData, this.longJumps, key);
    }

    public int apply(String key, int fromIndex) {
        return ClassNameTrie.apply(this.trieData, this.longJumps, key, fromIndex);
    }

    public static int apply(char[] data, int[] longJumps, String key) {
        return ClassNameTrie.apply(data, longJumps, key, 0);
    }

    public static int apply(char[] data, int[] longJumps, String key, int fromIndex) {
        int keyLength = key.length();
        int keyIndex = fromIndex;
        int dataIndex = 0;
        int result = -1;
        while (keyIndex < keyLength) {
            char c;
            char branchCount;
            int branchIndex;
            if ((branchIndex = Arrays.binarySearch(data, dataIndex, dataIndex + (branchCount = data[dataIndex++]), (c = key.charAt(keyIndex++)) == '/' ? (char)'.' : (char)c)) < 0) {
                return result;
            }
            int valueIndex = branchIndex + branchCount;
            char value = data[valueIndex];
            char segmentLength = '\u0000';
            if ((value & 0xC000) != 0) {
                if (keyIndex == keyLength || (value & 0x2000) != 0) {
                    result = value & 0x1FFF;
                }
                if (keyIndex == keyLength || (value & 0x8000) != 0) {
                    return result;
                }
            } else {
                segmentLength = value;
            }
            if (branchIndex > dataIndex) {
                int branchJump = data[valueIndex + branchCount - 1];
                if ((branchJump & 0x8000) != 0) {
                    branchJump = longJumps[branchJump & 0xFFFF7FFF];
                }
                dataIndex += branchJump;
            }
            dataIndex += branchCount * 3 - 1;
            if (segmentLength <= '\u0000') continue;
            if (keyLength - keyIndex < segmentLength) {
                return result;
            }
            int segmentEnd = dataIndex + segmentLength;
            while (dataIndex < segmentEnd) {
                if (((c = key.charAt(keyIndex++)) == '/' ? (char)'.' : (char)c) == data[dataIndex++]) continue;
                return result;
            }
            value = data[dataIndex];
            if ((value & 0x8000) == 0) continue;
            if (keyIndex == keyLength || (value & 0x2000) != 0) {
                result = value & 0x1FFF;
            }
            return result;
        }
        return result;
    }

    public static ClassNameTrie readFrom(DataInput in) throws IOException {
        int[] longJumps;
        int magic = in.readInt();
        if (magic != -586583533) {
            throw new IOException("Unexpected file magic " + magic);
        }
        int trieLength = in.readInt();
        char[] trieData = new char[trieLength];
        for (int i = 0; i < trieLength; ++i) {
            byte b = in.readByte();
            char c = (b & 0x80) == 0 ? (char)b : ((b & 0xE0) == 224 ? (char)((b & 0xF) << 12 | (in.readByte() & 0x3F) << 6 | in.readByte() & 0x3F) : (char)((b & 0x1F) << 6 | in.readByte() & 0x3F));
            trieData[i] = c;
        }
        int longJumpCount = in.readInt();
        if (longJumpCount > 0) {
            longJumps = new int[longJumpCount];
            for (int i = 0; i < longJumpCount; ++i) {
                longJumps[i] = in.readInt();
            }
        } else {
            longJumps = null;
        }
        return new ClassNameTrie(trieData, longJumps);
    }

    ClassNameTrie(char[] trieData, int[] longJumps) {
        this.trieData = trieData;
        this.longJumps = longJumps;
    }

    public static class JavaGenerator {
        public static void main(String[] args) throws IOException {
            if (args.length < 2) {
                throw new IllegalArgumentException("Expected: trie-dir java-dir [file.trie ...]");
            }
            Path trieDir = Paths.get(args[0], new String[0]).toAbsolutePath().normalize();
            if (!Files.isDirectory(trieDir, new LinkOption[0])) {
                throw new IllegalArgumentException("Bad trie directory: " + trieDir);
            }
            Path javaDir = Paths.get(args[1], new String[0]).toAbsolutePath().normalize();
            if (!Files.isDirectory(javaDir, new LinkOption[0])) {
                throw new IllegalArgumentException("Bad java directory: " + javaDir);
            }
            for (int i = 2; i < args.length; ++i) {
                Path triePath = trieDir.resolve(args[i]).normalize();
                String className = JavaGenerator.toClassName(triePath.getFileName().toString());
                Path pkgPath = trieDir.relativize(triePath.getParent());
                String pkgName = pkgPath.toString().replace(File.separatorChar, '.');
                Path javaPath = javaDir.resolve(pkgPath).resolve(className + ".java");
                JavaGenerator.generateJavaFile(triePath, javaPath, pkgName, className);
            }
        }

        private static String toClassName(String trieName) {
            StringBuilder className = new StringBuilder();
            boolean upperNext = true;
            for (int i = 0; i < trieName.length(); ++i) {
                char c = trieName.charAt(i);
                if (c == '_' | c == '.') {
                    upperNext = true;
                    continue;
                }
                className.append(upperNext ? Character.toUpperCase(c) : c);
                upperNext = false;
            }
            return className.toString();
        }

        private static void generateJavaFile(Path triePath, Path javaPath, String pkgName, String className) throws IOException {
            Builder trie = new Builder();
            trie.readClassNameMapping(triePath);
            ArrayList<String> lines = new ArrayList<String>();
            if (!pkgName.isEmpty()) {
                lines.add("package " + pkgName + ';');
            }
            lines.add("");
            lines.add("import datadog.trace.util.ClassNameTrie;");
            lines.add("");
            lines.add("// Generated from '" + triePath.getFileName() + "' - DO NOT EDIT!");
            lines.add("public final class " + className + " {");
            lines.add("");
            boolean hasLongJumps = JavaGenerator.generateJavaTrie(lines, "", trie);
            lines.add("");
            lines.add("  public static int apply(String key) {");
            if (hasLongJumps) {
                lines.add("    return ClassNameTrie.apply(TRIE_DATA, LONG_JUMPS, key);");
            } else {
                lines.add("    return ClassNameTrie.apply(TRIE_DATA, null, key);");
            }
            lines.add("  }");
            lines.add("");
            lines.add("  public static int apply(String key, int fromIndex) {");
            if (hasLongJumps) {
                lines.add("    return ClassNameTrie.apply(TRIE_DATA, LONG_JUMPS, key, fromIndex);");
            } else {
                lines.add("    return ClassNameTrie.apply(TRIE_DATA, null, key, fromIndex);");
            }
            lines.add("  }");
            lines.add("");
            lines.add("  private " + className + "() {}");
            lines.add("}");
            Files.write(javaPath, lines, StandardCharsets.UTF_8, new OpenOption[0]);
        }

        public static boolean generateJavaTrie(List<String> lines, String prefix, Builder trie) {
            int i;
            boolean hasLongJumps = trie.longJumpCount > 0;
            int firstLineNumber = lines.size();
            int chunk = 1;
            lines.add("  private static final String " + prefix + "TRIE_TEXT_" + chunk + " =");
            int chunkSize = 0;
            StringBuilder buf = new StringBuilder();
            buf.append("      \"");
            for (i = 0; i < trie.trieLength; ++i) {
                char c = trie.trieData[i];
                if (++chunkSize > 10000) {
                    ++chunk;
                    chunkSize = 0;
                    lines.add(buf + "\";");
                    lines.add("");
                    lines.add("  private static final String " + prefix + "TRIE_TEXT_" + chunk + " =");
                    buf.setLength(0);
                    buf.append("      \"");
                } else if (buf.length() > 120) {
                    lines.add(buf + "\"");
                    buf.setLength(0);
                    buf.append("          + \"");
                }
                if (c <= '\u00ff') {
                    buf.append(String.format("\\%03o", c));
                    continue;
                }
                buf.append(String.format("\\u%04x", c));
            }
            lines.add(buf + "\";");
            lines.add("");
            if (chunk > 1) {
                int n;
                lines.add("  private static final char[] " + prefix + "TRIE_DATA;");
                lines.add("  static {");
                lines.add("    int dataLength = 0;");
                for (n = 1; n <= chunk; ++n) {
                    lines.add("    dataLength += " + prefix + "TRIE_TEXT_" + n + ".length();");
                }
                lines.add("    " + prefix + "TRIE_DATA = new char[dataLength];");
                lines.add("    int dataIndex = 0;");
                lines.add("    String chunk;");
                for (n = 1; n <= chunk; ++n) {
                    lines.add("    chunk = " + prefix + "TRIE_TEXT_" + n + ";");
                    lines.add("    chunk.getChars(0, chunk.length(), " + prefix + "TRIE_DATA, dataIndex);");
                    lines.add("    dataIndex += chunk.length();");
                }
                lines.add("  }");
            } else {
                lines.set(firstLineNumber, "  private static final String " + prefix + "TRIE_TEXT =");
                lines.add("  private static final char[] " + prefix + "TRIE_DATA = " + prefix + "TRIE_TEXT.toCharArray();");
            }
            if (hasLongJumps) {
                lines.add("");
                lines.add("  private static final int[] " + prefix + "LONG_JUMPS = {");
                buf.setLength(0);
                buf.append("   ");
                for (i = 0; i < trie.longJumpCount; ++i) {
                    int jump = trie.longJumps[i];
                    if (buf.length() > 90) {
                        lines.add(buf.toString());
                        buf.setLength(0);
                        buf.append("   ");
                    }
                    buf.append(' ').append(String.format("0x%06X", jump)).append(',');
                }
                lines.add(buf.toString());
                lines.add("  };");
            }
            return hasLongJumps;
        }
    }

    public static class Builder {
        public static final ClassNameTrie EMPTY_TRIE = new ClassNameTrie(new char[]{'\u0000'}, null);
        private static final Pattern MAPPING_LINE = Pattern.compile("^\\s*(?:([0-9]+)\\s+)?([^\\s#]+)");
        private char[] trieData;
        private int trieLength;
        private int[] longJumps;
        private int longJumpCount;

        public Builder() {
        }

        public Builder(ClassNameTrie trie) {
            this.trieData = trie.trieData;
            this.trieLength = this.trieData.length;
            this.longJumps = trie.longJumps;
            this.longJumpCount = null != this.longJumps ? this.longJumps.length : 0;
        }

        public boolean isEmpty() {
            return this.trieLength == 0;
        }

        public int apply(String key) {
            return this.trieLength > 0 ? ClassNameTrie.apply(this.trieData, this.longJumps, key) : -1;
        }

        public ClassNameTrie buildTrie() {
            if (this.trieLength == 0) {
                return EMPTY_TRIE;
            }
            if (this.trieData.length > this.trieLength) {
                this.trieData = Arrays.copyOfRange(this.trieData, 0, this.trieLength);
            }
            if (null != this.longJumps && this.longJumps.length > this.longJumpCount) {
                this.longJumps = Arrays.copyOfRange(this.longJumps, 0, this.longJumpCount);
            }
            return new ClassNameTrie(this.trieData, this.longJumps);
        }

        public void writeTo(DataOutput out) throws IOException {
            int i;
            out.writeInt(-586583533);
            out.writeInt(this.trieLength);
            for (i = 0; i < this.trieLength; ++i) {
                char c = this.trieData[i];
                if (c >= '\u0001' && c <= '\u007f') {
                    out.writeByte(c);
                    continue;
                }
                if (c > '\u07ff') {
                    out.writeByte(0xE0 | c >> 12 & 0xF);
                    out.writeByte(0x80 | c >> 6 & 0x3F);
                    out.writeByte(0x80 | c & 0x3F);
                    continue;
                }
                out.writeByte(0xC0 | c >> 6 & 0x1F);
                out.writeByte(0x80 | c & 0x3F);
            }
            out.writeInt(this.longJumpCount);
            for (i = 0; i < this.longJumpCount; ++i) {
                out.writeInt(this.longJumps[i]);
            }
        }

        public void readClassNameMapping(Path triePath) throws IOException {
            for (String l : Files.readAllLines(triePath, StandardCharsets.UTF_8)) {
                Matcher m = MAPPING_LINE.matcher(l);
                if (!m.find()) continue;
                this.put(m.group(2), m.group(1) != null ? Integer.parseInt(m.group(1)) : 1);
            }
        }

        public void put(String className, int number) {
            char value;
            String key;
            if (null == className || className.isEmpty()) {
                throw new IllegalArgumentException("Null or empty class name");
            }
            if (number < 0) {
                throw new IllegalArgumentException("Number for " + className + " is negative: " + number);
            }
            if (number > 8191) {
                throw new IllegalArgumentException("Number for " + className + " is too big: " + number);
            }
            if (className.charAt(className.length() - 1) == '*') {
                key = className.substring(0, className.length() - 1);
                value = (char)(number | 0x2000);
            } else {
                key = className;
                value = (char)number;
            }
            if (this.trieLength == 0) {
                int keyLength = key.length();
                this.trieLength = (keyLength > 1 ? 3 : 2) + keyLength;
                this.trieData = new char[8192];
                this.trieData[0] = '\u0001';
                this.trieData[1] = key.charAt(0);
                if (keyLength > 1) {
                    this.trieData[2] = (char)(keyLength - 1);
                    key.getChars(1, keyLength, this.trieData, 3);
                }
                this.trieData[this.trieLength - 1] = (char)(value | 0x8000);
            } else {
                this.insertMapping(key, value);
            }
        }

        private void makeHole(int start, int length) {
            char[] oldData = this.trieData;
            if (this.trieLength + length > oldData.length) {
                this.trieData = new char[Math.max(this.trieLength + length, oldData.length + (oldData.length >> 1))];
                System.arraycopy(oldData, 0, this.trieData, 0, start);
            }
            System.arraycopy(oldData, start, this.trieData, start + length, this.trieLength - start);
            this.trieLength += length;
        }

        private char setJump(int jump) {
            if (jump < 32768) {
                return (char)jump;
            }
            if (this.longJumpCount == 0) {
                this.longJumps = new int[16];
            } else if (this.longJumpCount == this.longJumps.length) {
                int[] oldJumps = this.longJumps;
                this.longJumps = new int[this.longJumpCount + (this.longJumpCount >> 1)];
                System.arraycopy(oldJumps, 0, this.longJumps, 0, this.longJumpCount);
            }
            this.longJumps[this.longJumpCount] = jump;
            return (char)(this.longJumpCount++ | 0x8000);
        }

        private int getJump(char jump) {
            return (jump & 0x8000) == 0 ? jump : this.longJumps[jump & 0xFFFF7FFF];
        }

        private char updateJump(char jump, int offset) {
            if (jump < '\u8000') {
                return this.setJump(jump + offset);
            }
            int n = jump & 0xFFFF7FFF;
            this.longJumps[n] = this.longJumps[n] + offset;
            return jump;
        }

        private void insertMapping(String key, char valueToInsert) {
            BitSet jumpsToOffset = new BitSet();
            int keyLength = key.length();
            int keyIndex = 0;
            int dataIndex = 0;
            int subTrieEnd = this.trieLength;
            int jumpOffset = 0;
            while (keyIndex < keyLength) {
                char c;
                char branchCount;
                int branchIndex;
                if ((branchIndex = Arrays.binarySearch(this.trieData, dataIndex, dataIndex + (branchCount = this.trieData[dataIndex++]), c = key.charAt(keyIndex++))) < 0) {
                    jumpOffset = this.insertBranch(dataIndex, key, keyIndex - 1, valueToInsert, branchCount, ~branchIndex - dataIndex, subTrieEnd);
                    break;
                }
                int valueIndex = branchIndex + branchCount;
                char value = this.trieData[valueIndex];
                if ((value & 0xC000) != 0 && keyIndex == keyLength) {
                    this.trieData[valueIndex] = (char)(value & 0xFFFFE000 | valueToInsert);
                    return;
                }
                int branch = branchIndex - dataIndex;
                if (branch < branchCount - '\u0001') {
                    int nextJumpIndex = valueIndex + branchCount;
                    int nextBranchJump = this.getJump(this.trieData[nextJumpIndex]);
                    subTrieEnd = dataIndex + branchCount * 3 - 1 + nextBranchJump;
                    for (int b = branch + 1; b < branchCount; ++b) {
                        jumpsToOffset.set(nextJumpIndex++);
                    }
                }
                if (branch > 0) {
                    dataIndex += this.getJump(this.trieData[valueIndex + branchCount - 1]);
                }
                dataIndex += branchCount * 3 - 1;
                if ((value & 0x8000) != 0) {
                    this.trieData[valueIndex] = (char)(value & 0xFFFF7FFF | 0x4000);
                    jumpOffset = this.appendLeaf(dataIndex, key, keyIndex, valueToInsert);
                    break;
                }
                if ((value & 0x4000) != 0) continue;
                if (keyIndex == keyLength) {
                    this.trieData[valueIndex] = (char)(valueToInsert | 0x4000);
                    if (value <= '\u0000') break;
                    jumpOffset = this.prependNode(dataIndex, value - '\u0001');
                    break;
                }
                if (value <= '\u0000') continue;
                char segmentLength = value;
                int segmentEnd = dataIndex + segmentLength;
                while (keyIndex < keyLength && dataIndex < segmentEnd && (c = key.charAt(keyIndex)) == this.trieData[dataIndex]) {
                    ++keyIndex;
                    ++dataIndex;
                }
                if (dataIndex < segmentEnd) {
                    if (keyIndex == keyLength) {
                        int n = valueIndex;
                        this.trieData[n] = (char)(this.trieData[n] - (segmentEnd - (dataIndex - 1)));
                        jumpOffset = this.insertBud(dataIndex - 1, valueToInsert, segmentEnd);
                        break;
                    }
                    int n = valueIndex;
                    this.trieData[n] = (char)(this.trieData[n] - (segmentEnd - dataIndex));
                    if (c < this.trieData[dataIndex]) {
                        jumpOffset = this.insertLeafLeft(dataIndex, key, keyIndex, valueToInsert, segmentEnd);
                        break;
                    }
                    jumpOffset = this.insertLeafRight(dataIndex, key, keyIndex, valueToInsert, segmentEnd, subTrieEnd);
                    break;
                }
                value = this.trieData[dataIndex];
                if ((value & 0x8000) != 0) {
                    if (keyIndex < keyLength) {
                        int n = valueIndex;
                        this.trieData[n] = (char)(this.trieData[n] - '\u0001');
                        jumpOffset = this.appendLeaf(dataIndex, key, keyIndex, valueToInsert);
                        break;
                    }
                    this.trieData[dataIndex] = (char)(value & 0xFFFFE000 | valueToInsert);
                    return;
                }
                if (keyIndex != keyLength) continue;
                int n = valueIndex;
                this.trieData[n] = (char)(this.trieData[n] - '\u0001');
                jumpOffset = this.prependNode(dataIndex - 1, valueToInsert | 0x4000);
                break;
            }
            if (jumpOffset > 0) {
                int i = jumpsToOffset.nextSetBit(0);
                while (i >= 0) {
                    this.trieData[i] = this.updateJump(this.trieData[i], jumpOffset);
                    i = jumpsToOffset.nextSetBit(i + 1);
                }
            }
        }

        private int insertBranch(int dataIndex, String key, int keyIndex, int value, int branchCount, int newBranch, int subTrieEnd) {
            int precedingJump;
            boolean collapseRight;
            int remainingKeyLength = key.length() - keyIndex;
            int insertedCharacters = 3 + remainingKeyLength;
            boolean bl = collapseRight = remainingKeyLength == 1;
            if (collapseRight) {
                remainingKeyLength = 0;
                insertedCharacters = 3;
            }
            int i = dataIndex + newBranch;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            this.trieData[dataIndex - 1] = (char)(branchCount + 1);
            this.trieData[i++] = key.charAt(keyIndex);
            System.arraycopy(this.trieData, j, this.trieData, i, branchCount);
            i += branchCount;
            j += branchCount;
            this.trieData[i++] = (char)(collapseRight ? value | 0x8000 : remainingKeyLength - 1);
            int subTrieStart = dataIndex + branchCount * 3 - 1;
            if (newBranch < branchCount) {
                System.arraycopy(this.trieData, j, this.trieData, i, branchCount);
                j += branchCount;
                precedingJump = newBranch > 0 ? this.getJump(this.trieData[(i += branchCount) - 1]) : 0;
                this.trieData[i++] = this.setJump(precedingJump + remainingKeyLength);
                for (int b = newBranch + 1; b < branchCount; ++b) {
                    this.trieData[i++] = this.updateJump(this.trieData[j++], remainingKeyLength);
                }
            } else {
                System.arraycopy(this.trieData, j, this.trieData, i, branchCount - 1);
                i += branchCount - 1;
                j += branchCount - 1;
                precedingJump = subTrieEnd - subTrieStart;
                this.trieData[i++] = this.setJump(precedingJump);
            }
            System.arraycopy(this.trieData, subTrieStart + insertedCharacters, this.trieData, i, precedingJump);
            i += precedingJump;
            if (!collapseRight) {
                key.getChars(keyIndex + 1, key.length(), this.trieData, i);
                i += remainingKeyLength - 1;
                this.trieData[i++] = (char)(value | 0x8000);
            }
            return insertedCharacters;
        }

        private int prependNode(int dataIndex, int value) {
            boolean collapseRight;
            int insertedCharacters = 2;
            boolean bl = collapseRight = value == 0 && (this.trieData[dataIndex + 1] & 0x8000) != 0;
            if (collapseRight) {
                --insertedCharacters;
            }
            int i = dataIndex;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            this.trieData[i++] = '\u0001';
            this.trieData[i++] = this.trieData[j];
            if (!collapseRight) {
                this.trieData[i++] = (char)value;
            }
            return insertedCharacters;
        }

        private int insertBud(int dataIndex, int value, int segmentEnd) {
            boolean collapseRight;
            int insertedCharacters = 4;
            int pivot = dataIndex + 2;
            boolean bl = collapseRight = pivot == segmentEnd && (this.trieData[segmentEnd] & 0x8000) != 0;
            if (collapseRight) {
                insertedCharacters = 3;
            }
            int i = dataIndex;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            this.trieData[i++] = '\u0001';
            this.trieData[i++] = this.trieData[j];
            this.trieData[i++] = (char)(value | 0x4000);
            this.trieData[i++] = '\u0001';
            this.trieData[i++] = this.trieData[j + 1];
            if (!collapseRight) {
                this.trieData[i++] = (char)(segmentEnd - pivot);
            }
            return insertedCharacters;
        }

        private int insertLeafLeft(int dataIndex, String key, int keyIndex, int value, int segmentEnd) {
            boolean collapseRight;
            boolean collapseLeft;
            int remainingKeyLength = key.length() - keyIndex;
            int insertedCharacters = 5 + remainingKeyLength;
            int pivot = dataIndex + 1;
            boolean bl = collapseLeft = remainingKeyLength == 1;
            if (collapseLeft) {
                --insertedCharacters;
            }
            boolean bl2 = collapseRight = pivot == segmentEnd && (this.trieData[segmentEnd] & 0x8000) != 0;
            if (collapseRight) {
                --insertedCharacters;
                ++pivot;
            }
            int i = dataIndex;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            this.trieData[i++] = 2;
            this.trieData[i++] = key.charAt(keyIndex);
            this.trieData[i++] = this.trieData[j];
            this.trieData[i++] = (char)(collapseLeft ? value | 0x8000 : remainingKeyLength - 1);
            this.trieData[i++] = (char)(collapseRight ? this.trieData[j + 1] : segmentEnd - pivot);
            if (!collapseLeft) {
                this.trieData[i++] = (char)remainingKeyLength;
                key.getChars(keyIndex + 1, key.length(), this.trieData, i);
                i += remainingKeyLength - 1;
                this.trieData[i++] = (char)(value | 0x8000);
            } else {
                this.trieData[i++] = '\u0000';
            }
            return insertedCharacters;
        }

        private int insertLeafRight(int dataIndex, String key, int keyIndex, int value, int segmentEnd, int subTrieEnd) {
            boolean collapseRight;
            boolean collapseLeft;
            int remainingKeyLength = key.length() - keyIndex;
            int insertedCharacters = 5 + remainingKeyLength;
            int pivot = dataIndex + 1;
            boolean bl = collapseLeft = pivot == segmentEnd && (this.trieData[segmentEnd] & 0x8000) != 0;
            if (collapseLeft) {
                --insertedCharacters;
                ++pivot;
            }
            boolean bl2 = collapseRight = remainingKeyLength == 1;
            if (collapseRight) {
                --insertedCharacters;
            }
            int i = dataIndex;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            this.trieData[i++] = 2;
            this.trieData[i++] = this.trieData[j];
            this.trieData[i++] = key.charAt(keyIndex);
            this.trieData[i++] = (char)(collapseLeft ? this.trieData[j + 1] : segmentEnd - pivot);
            this.trieData[i++] = (char)(collapseRight ? value | 0x8000 : remainingKeyLength - 1);
            this.trieData[i++] = this.setJump(subTrieEnd - pivot);
            System.arraycopy(this.trieData, pivot + insertedCharacters, this.trieData, i, subTrieEnd - pivot);
            i += subTrieEnd - pivot;
            if (!collapseRight) {
                key.getChars(keyIndex + 1, key.length(), this.trieData, i);
                i += remainingKeyLength - 1;
                this.trieData[i++] = (char)(value | 0x8000);
            }
            return insertedCharacters;
        }

        private int appendLeaf(int dataIndex, String key, int keyIndex, int value) {
            boolean collapseRight;
            boolean insertBud;
            int remainingKeyLength = key.length() - keyIndex;
            int insertedCharacters = 3 + remainingKeyLength;
            boolean bl = insertBud = dataIndex < this.trieLength && (this.trieData[dataIndex] & 0x8000) != 0;
            if (insertBud) {
                ++insertedCharacters;
            }
            boolean bl2 = collapseRight = remainingKeyLength == 1;
            if (collapseRight) {
                --insertedCharacters;
            }
            int i = dataIndex;
            int j = i + insertedCharacters;
            this.makeHole(i, insertedCharacters);
            if (insertBud) {
                char c = this.trieData[i - 1];
                this.trieData[i - 1] = '\u0001';
                this.trieData[i++] = c;
                this.trieData[i++] = (char)(this.trieData[j] & 0xFFFF7FFF | 0x4000);
            }
            this.trieData[i++] = '\u0001';
            this.trieData[i++] = key.charAt(keyIndex);
            if (!collapseRight) {
                this.trieData[i++] = (char)(remainingKeyLength - 1);
                key.getChars(keyIndex + 1, key.length(), this.trieData, i);
                i += remainingKeyLength - 1;
            }
            this.trieData[i++] = (char)(value | 0x8000);
            return insertedCharacters;
        }
    }
}

