package com.databricks.jdbc.common.util;

import com.databricks.jdbc.api.internal.IDatabricksConnectionContext;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.sdk.core.UserAgent;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;

public class UserAgentManager {
  private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(UserAgentManager.class);
  private static final String SDK_USER_AGENT = "databricks-sdk-java";
  private static final String JDBC_HTTP_USER_AGENT = "databricks-jdbc-http";
  private static final String DEFAULT_USER_AGENT = "DatabricksJDBCDriverOSS";
  private static final String CLIENT_USER_AGENT_PREFIX = "Java";
  public static final String USER_AGENT_SEA_CLIENT = "SQLExecHttpClient";
  public static final String USER_AGENT_THRIFT_CLIENT = "THttpClient";
  private static final String VERSION_FILLER = "version";

  /**
   * Set the user agent for the Databricks JDBC driver.
   *
   * @param connectionContext The connection context.
   */
  public static void setUserAgent(IDatabricksConnectionContext connectionContext) {
    // Set the base product and client info
    UserAgent.withProduct(DEFAULT_USER_AGENT, DriverUtil.getDriverVersion());
    UserAgent.withOtherInfo(CLIENT_USER_AGENT_PREFIX, connectionContext.getClientUserAgent());
    if (connectionContext.getCustomerUserAgent() == null) {
      return;
    }
    try {
      String decodedUA =
          URLDecoder.decode(
              connectionContext.getCustomerUserAgent(),
              StandardCharsets.UTF_8); // This is for encoded userAgentString
      int i = decodedUA.indexOf('/');
      String customerName = (i < 0) ? decodedUA : decodedUA.substring(0, i);
      String customerVersion = (i < 0) ? VERSION_FILLER : decodedUA.substring(i + 1);
      UserAgent.withOtherInfo(customerName, UserAgent.sanitize(customerVersion));
    } catch (Exception e) {
      LOGGER.debug(
          "Failed to set user agent for customer userAgent entry {}, Error {}",
          connectionContext.getCustomerUserAgent(),
          e);
    }
  }

  /** Gets the user agent string for Databricks Driver HTTP Client. */
  public static String getUserAgentString() {
    String sdkUserAgent = UserAgent.asString();
    // Split the string into parts
    String[] parts = sdkUserAgent.split("\\s+");
    // User Agent is in format:
    // product/product-version databricks-sdk-java/sdk-version jvm/jvm-version other-info
    // Remove the SDK part from user agent
    StringBuilder mergedString = new StringBuilder();
    for (int i = 0; i < parts.length; i++) {
      if (parts[i].startsWith(SDK_USER_AGENT)) {
        mergedString.append(JDBC_HTTP_USER_AGENT);
      } else {
        mergedString.append(parts[i]);
      }
      if (i != parts.length - 1) {
        mergedString.append(" "); // Add space between parts
      }
    }
    return mergedString.toString();
  }
}
