/*
 * Decompiled with CFR 0.152.
 */
package com.danikula.videocache;

import android.content.Context;
import android.net.Uri;
import com.danikula.videocache.CacheListener;
import com.danikula.videocache.Config;
import com.danikula.videocache.GetRequest;
import com.danikula.videocache.HttpProxyCacheServerClients;
import com.danikula.videocache.Pinger;
import com.danikula.videocache.Preconditions;
import com.danikula.videocache.ProxyCacheException;
import com.danikula.videocache.ProxyCacheUtils;
import com.danikula.videocache.StorageUtils;
import com.danikula.videocache.file.DiskUsage;
import com.danikula.videocache.file.FileNameGenerator;
import com.danikula.videocache.file.Md5FileNameGenerator;
import com.danikula.videocache.file.TotalCountLruDiskUsage;
import com.danikula.videocache.file.TotalSizeLruDiskUsage;
import com.danikula.videocache.sourcestorage.SourceInfoStorage;
import com.danikula.videocache.sourcestorage.SourceInfoStorageFactory;
import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpProxyCacheServer {
    private static final Logger LOG = LoggerFactory.getLogger((String)"HttpProxyCacheServer");
    private static final String PROXY_HOST = "127.0.0.1";
    private final Object clientsLock = new Object();
    private final ExecutorService socketProcessor = Executors.newFixedThreadPool(8);
    private final Map<String, HttpProxyCacheServerClients> clientsMap = new ConcurrentHashMap<String, HttpProxyCacheServerClients>();
    private final ServerSocket serverSocket;
    private final int port;
    private final Thread waitConnectionThread;
    private final Config config;
    private final Pinger pinger;

    public HttpProxyCacheServer(Context context) {
        this(new Builder(context).buildConfig());
    }

    private HttpProxyCacheServer(Config config) {
        this.config = Preconditions.checkNotNull(config);
        try {
            InetAddress inetAddress = InetAddress.getByName(PROXY_HOST);
            this.serverSocket = new ServerSocket(0, 8, inetAddress);
            this.port = this.serverSocket.getLocalPort();
            CountDownLatch startSignal = new CountDownLatch(1);
            this.waitConnectionThread = new Thread(new WaitRequestsRunnable(startSignal));
            this.waitConnectionThread.start();
            startSignal.await();
            this.pinger = new Pinger(PROXY_HOST, this.port);
            LOG.info("Proxy cache server started. Is it alive? " + this.isAlive());
        }
        catch (IOException | InterruptedException e) {
            this.socketProcessor.shutdown();
            throw new IllegalStateException("Error starting local proxy server", e);
        }
    }

    public String getProxyUrl(String url) {
        return this.getProxyUrl(url, true);
    }

    public String getProxyUrl(String url, boolean allowCachedFileUri) {
        if (allowCachedFileUri && this.isCached(url)) {
            File cacheFile = this.getCacheFile(url);
            this.touchFileSafely(cacheFile);
            return Uri.fromFile((File)cacheFile).toString();
        }
        return this.isAlive() ? this.appendToProxyUrl(url) : url;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerCacheListener(CacheListener cacheListener, String url) {
        Preconditions.checkAllNotNull(cacheListener, url);
        Object object = this.clientsLock;
        synchronized (object) {
            try {
                this.getClients(url).registerCacheListener(cacheListener);
            }
            catch (ProxyCacheException e) {
                LOG.warn("Error registering cache listener", (Throwable)e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unregisterCacheListener(CacheListener cacheListener, String url) {
        Preconditions.checkAllNotNull(cacheListener, url);
        Object object = this.clientsLock;
        synchronized (object) {
            try {
                this.getClients(url).unregisterCacheListener(cacheListener);
            }
            catch (ProxyCacheException e) {
                LOG.warn("Error registering cache listener", (Throwable)e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unregisterCacheListener(CacheListener cacheListener) {
        Preconditions.checkNotNull(cacheListener);
        Object object = this.clientsLock;
        synchronized (object) {
            for (HttpProxyCacheServerClients clients : this.clientsMap.values()) {
                clients.unregisterCacheListener(cacheListener);
            }
        }
    }

    public boolean isCached(String url) {
        Preconditions.checkNotNull(url, "Url can't be null!");
        return this.getCacheFile(url).exists();
    }

    public void shutdown() {
        LOG.info("Shutdown proxy server");
        this.shutdownClients();
        this.config.sourceInfoStorage.release();
        this.waitConnectionThread.interrupt();
        try {
            if (!this.serverSocket.isClosed()) {
                this.serverSocket.close();
            }
        }
        catch (IOException e) {
            this.onError(new ProxyCacheException("Error shutting down proxy server", e));
        }
    }

    private boolean isAlive() {
        return this.pinger.ping(3, 70);
    }

    private String appendToProxyUrl(String url) {
        return String.format(Locale.US, "http://%s:%d/%s", PROXY_HOST, this.port, ProxyCacheUtils.encode(url));
    }

    private File getCacheFile(String url) {
        File cacheDir = this.config.cacheRoot;
        String fileName = this.config.fileNameGenerator.generate(url);
        return new File(cacheDir, fileName);
    }

    private void touchFileSafely(File cacheFile) {
        try {
            this.config.diskUsage.touch(cacheFile);
        }
        catch (IOException e) {
            LOG.error("Error touching file " + cacheFile, (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void shutdownClients() {
        Object object = this.clientsLock;
        synchronized (object) {
            for (HttpProxyCacheServerClients clients : this.clientsMap.values()) {
                clients.shutdown();
            }
            this.clientsMap.clear();
        }
    }

    private void waitForRequest() {
        try {
            while (!Thread.currentThread().isInterrupted()) {
                Socket socket = this.serverSocket.accept();
                LOG.debug("Accept new socket " + socket);
                this.socketProcessor.submit(new SocketProcessorRunnable(socket));
            }
        }
        catch (IOException e) {
            this.onError(new ProxyCacheException("Error during waiting connection", e));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processSocket(Socket socket) {
        try {
            GetRequest request = GetRequest.read(socket.getInputStream());
            LOG.debug("Request to cache proxy:" + request);
            String url = ProxyCacheUtils.decode(request.uri);
            if (this.pinger.isPingRequest(url)) {
                this.pinger.responseToPing(socket);
            } else {
                HttpProxyCacheServerClients clients = this.getClients(url);
                clients.processRequest(request, socket);
            }
        }
        catch (SocketException e) {
            LOG.debug("Closing socket\u2026 Socket is closed by client.");
        }
        catch (ProxyCacheException | IOException e) {
            this.onError(new ProxyCacheException("Error processing request", e));
        }
        finally {
            this.releaseSocket(socket);
            LOG.debug("Opened connections: " + this.getClientsCount());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private HttpProxyCacheServerClients getClients(String url) throws ProxyCacheException {
        Object object = this.clientsLock;
        synchronized (object) {
            HttpProxyCacheServerClients clients = this.clientsMap.get(url);
            if (clients == null) {
                clients = new HttpProxyCacheServerClients(url, this.config);
                this.clientsMap.put(url, clients);
            }
            return clients;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int getClientsCount() {
        Object object = this.clientsLock;
        synchronized (object) {
            int count = 0;
            for (HttpProxyCacheServerClients clients : this.clientsMap.values()) {
                count += clients.getClientsCount();
            }
            return count;
        }
    }

    private void releaseSocket(Socket socket) {
        this.closeSocketInput(socket);
        this.closeSocketOutput(socket);
        this.closeSocket(socket);
    }

    private void closeSocketInput(Socket socket) {
        try {
            if (!socket.isInputShutdown()) {
                socket.shutdownInput();
            }
        }
        catch (SocketException e) {
            LOG.debug("Releasing input stream\u2026 Socket is closed by client.");
        }
        catch (IOException e) {
            this.onError(new ProxyCacheException("Error closing socket input stream", e));
        }
    }

    private void closeSocketOutput(Socket socket) {
        try {
            if (!socket.isOutputShutdown()) {
                socket.shutdownOutput();
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to close socket on proxy side: {}. It seems client have already closed connection.", (Object)e.getMessage());
        }
    }

    private void closeSocket(Socket socket) {
        try {
            if (!socket.isClosed()) {
                socket.close();
            }
        }
        catch (IOException e) {
            this.onError(new ProxyCacheException("Error closing socket", e));
        }
    }

    private void onError(Throwable e) {
        LOG.error("HttpProxyCacheServer error", e);
    }

    public static final class Builder {
        private static final long DEFAULT_MAX_SIZE = 0x20000000L;
        private File cacheRoot;
        private FileNameGenerator fileNameGenerator;
        private DiskUsage diskUsage;
        private SourceInfoStorage sourceInfoStorage;

        public Builder(Context context) {
            this.sourceInfoStorage = SourceInfoStorageFactory.newSourceInfoStorage(context);
            this.cacheRoot = StorageUtils.getIndividualCacheDirectory(context);
            this.diskUsage = new TotalSizeLruDiskUsage(0x20000000L);
            this.fileNameGenerator = new Md5FileNameGenerator();
        }

        public Builder cacheDirectory(File file) {
            this.cacheRoot = Preconditions.checkNotNull(file);
            return this;
        }

        public Builder fileNameGenerator(FileNameGenerator fileNameGenerator) {
            this.fileNameGenerator = Preconditions.checkNotNull(fileNameGenerator);
            return this;
        }

        public Builder maxCacheSize(long maxSize) {
            this.diskUsage = new TotalSizeLruDiskUsage(maxSize);
            return this;
        }

        public Builder maxCacheFilesCount(int count) {
            this.diskUsage = new TotalCountLruDiskUsage(count);
            return this;
        }

        public HttpProxyCacheServer build() {
            Config config = this.buildConfig();
            return new HttpProxyCacheServer(config);
        }

        private Config buildConfig() {
            return new Config(this.cacheRoot, this.fileNameGenerator, this.diskUsage, this.sourceInfoStorage);
        }
    }

    private final class SocketProcessorRunnable
    implements Runnable {
        private final Socket socket;

        public SocketProcessorRunnable(Socket socket) {
            this.socket = socket;
        }

        @Override
        public void run() {
            HttpProxyCacheServer.this.processSocket(this.socket);
        }
    }

    private final class WaitRequestsRunnable
    implements Runnable {
        private final CountDownLatch startSignal;

        public WaitRequestsRunnable(CountDownLatch startSignal) {
            this.startSignal = startSignal;
        }

        @Override
        public void run() {
            this.startSignal.countDown();
            HttpProxyCacheServer.this.waitForRequest();
        }
    }
}

