package com.dada.smart.user.visitor;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.AbsListView;
import android.widget.ListView;

import com.dada.smart.common.ReflectUtils;
import com.dada.smart.common.ResourceIds;
import com.dada.smart.common.Utils;
import com.dada.smart.common.WindowMagician;
import com.dada.smart.user.R;
import com.dada.smart.user.event.Event;
import com.dada.smart.user.event.Events;
import com.dada.smart.user.event.ViewEvent;
import com.dada.smart.user.event.ViewEventListener;
import com.dada.smart.user.event.ViewTreeChild;
import com.orhanobut.logger.Logger;

import java.util.ArrayList;
import java.util.List;

/**
 * 处理view的遍历以及需要遍历的条件管理
 * Created by tomkeyzhang on 21/5/18.
 */
public class ViewVisitor implements ViewEventListener,
        ViewTreeObserver.OnGlobalLayoutListener,
        ViewTreeObserver.OnWindowFocusChangeListener,
        ViewTreeObserver.OnWindowAttachListener {
    private ViewEventListener listener;
    //    private Map<View, AccessibilityDelegate> delegateMap;
    private VisitorExecutor executor;

    //本界面绑定的控件事件
    private Events viewEvents;

    //主Window的根View
    private View decorView;
    //最近被访问的ListView
    private ListView recentListView;
    //最近被访问的RecyclerView
    private RecyclerView recentRecyclerView;

    private WindowManager windowManager;

    private Activity activity;
    private List<ViewTreeObserver> treeObservers;
    private ViewMatcher viewMatcher;
    private State state;

    long createTime;

    public enum State {
        INIT, CREATE, /**
         * RESUME, PAUSE,
         **/DESTROY
    }

    public ViewVisitor(@NonNull Activity activity,
                       @NonNull ResourceIds resourceIds,
                       @NonNull Events viewEvents,
                       @NonNull ViewEventListener listener) {
        this.activity = activity;
        this.viewEvents = viewEvents;
        this.listener = listener;
        this.treeObservers = new ArrayList<>();
        this.windowManager = activity.getWindowManager();
        this.decorView = activity.getWindow().getDecorView();
        this.executor = VisitorExecutor.getInstance();
        this.viewMatcher = new ViewMatcher(resourceIds);
        this.state = State.INIT;
        this.createTime = System.currentTimeMillis();
    }


    public void onCreate() {
        createViewTreeObservers(decorView.findViewById(android.R.id.content));
        state = State.CREATE;
    }

    public void onDestroy() {
        state = State.DESTROY;
        removeViewTreeObservers();
        activity = null;
        windowManager = null;
        decorView = null;
    }

    @Override
    public void onEvent(ViewEvent event) {
        if (listener != null)
            listener.onEvent(event);
    }

    @Override
    public void onGlobalLayout() {
        Logger.e("onGlobalLayout-" + activity);
        bindEventAsync();
    }

    @Override
    public void onWindowAttached() {
    }

    @Override
    public void onWindowDetached() {
        reAddObserverForNewWindow(true);
        Logger.e("onWindowDetached-" + activity);
        bindEventAsync();
    }

    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        if (!hasFocus) {//todo 为啥是丢失焦点去执行重新绑定
            reAddObserverForNewWindow(false);
            Logger.e("onWindowFocusChanged-" + activity);
            bindEventAsync();
        }
    }

    /**
     * 对新产生的window添加observer
     *
     * @param fromDetached 是否来自窗口detached，窗口的detached时
     */
    private void reAddObserverForNewWindow(boolean fromDetached) {
        if (treeObservers.size() == 0) {
            return;
        }
        if (state == State.DESTROY)
            return;
        List<WindowMagician.ViewRootData> rootViews = WindowMagician.getFocusWindowRootView(activity.getWindowManager(), fromDetached);
        ViewTreeObserver viewTreeObserver;
        for (WindowMagician.ViewRootData viewRootData : rootViews) {
            viewTreeObserver = viewRootData.getView().getViewTreeObserver();
//            if (isDebug)
//                WindowMagician.printViewTree(viewRootData.getView(), 0);
            if (!treeObservers.contains(viewTreeObserver)) {
                viewTreeObserver.addOnWindowAttachListener(this);
                viewTreeObserver.addOnWindowFocusChangeListener(this);
                viewTreeObserver.addOnGlobalLayoutListener(this);
                treeObservers.add(viewTreeObserver);
            }
        }
    }

    /**
     * 异步的执行activity中全部view的绑定事件操作
     */
    private void bindEventAsync() {
        executor.execute(this, new VisitorExecutor.Task() {
            @Override
            public void run(VisitorExecutor.TaskInfo taskInfo) {
                Utils.sleep(300);
                if (state != State.DESTROY) {
                    refreshViewTree(viewEvents);
                    bindEventsReal(taskInfo);
                }
            }
        });
    }


    void refreshViewTree(Events events) {
        List<Event> copyEvents = events.getEvents();
        for (Event event : copyEvents) {
            if (!event.isIdMatcher()) {
                for (ViewTreeChild child : event.getViewTreeChildren()) {
                    child.setPassVerify(false);
                }
            }
        }
    }

    /**
     * 异步的执行指定view及其child的绑定事件操作
     *
     * @param root
     */
    private void bindEventAsync(final View root) {
        executor.execute(this, new VisitorExecutor.Task() {
            @Override
            public void run(VisitorExecutor.TaskInfo taskInfo) {
                if (state != State.DESTROY) {
                    refreshViewTree(viewEvents);
                    bindEventToTargetView(root, taskInfo);
                }
            }
        });
    }

    private void bindEventsReal(VisitorExecutor.TaskInfo taskInfo) {
        try {
            if (viewEvents.isInMainWindow()) {
                //todo 初始状态取R.id.content 是否会有遗漏view
                decorView.setTag(R.id.tag_tree_index, 0);
                bindEventToTargetView(decorView, taskInfo);
            } else {
                List<WindowMagician.ViewRootData> dataList = WindowMagician.getFocusWindowRootView(windowManager, false);
                for (WindowMagician.ViewRootData viewRootData : dataList) {
                    View decorView = viewRootData.getView();
                    decorView.setTag(R.id.tag_tree_index, 0);
                    bindEventToTargetView(decorView, taskInfo);
                }
            }
        } catch (NullPointerException ignore) {
        }

    }

    /**
     * @param root     待匹配的view
     * @param taskInfo 当前异步任务描述
     */
    private void bindEventToTargetView(View root, VisitorExecutor.TaskInfo taskInfo) {
        //        Logger.v("bindEventToTargetView:" + root);
        if (root == null || root.getVisibility() != View.VISIBLE || state == State.DESTROY || taskInfo.isCancelled) {
            return;
        }
        for (Event event : viewEvents.getEvents()) {
            ViewEvent viewEvent = viewMatcher.matches(root, event, recentListView, recentRecyclerView);
            if (viewEvent != null) {
                Logger.v("find view=" + root + " event" + event);
                if (event.getTypeId() == Event.TYPE_CLICK) {
                    addEventToAccessibilityDelegate(root, viewEvent);
                } else if (event.getTypeId() == Event.TYPE_VIEW_SHOW) {
                    onEvent(viewEvent);
                }
            }
        }
        if (root instanceof ViewGroup) {
            if (root instanceof ListView) {
                recentListView = (ListView) root;
                setOnScrollListenerIfNeed(recentListView);
            } else if (root instanceof RecyclerView) {
                recentRecyclerView = (RecyclerView) root;
                addOnScrollListenerIfNeed(recentRecyclerView);
            }

            ViewGroup parent = (ViewGroup) root;
            try {
                for (int i = 0; i < parent.getChildCount(); i++) {
                    if (!taskInfo.isCancelled) {
                        View child = parent.getChildAt(i);
                        child.setTag(R.id.tag_tree_index, (int) parent.getTag(R.id.tag_tree_index) + 1);
                        bindEventToTargetView(child, taskInfo);
                    }
                }
            } catch (NullPointerException ignore) {
            }


            if (root instanceof ListView) {
                recentListView = null;
            } else if (root instanceof RecyclerView) {
                recentRecyclerView = null;
            }
        }
    }

    /**
     * 给每个ListView最多设置一次ScrollListener事件
     *
     * @param listView
     */
    private void setOnScrollListenerIfNeed(@NonNull ListView listView) {
        try {
            AbsListView.OnScrollListener originListener = ReflectUtils.reflect(listView).field("mOnScrollListener").get();
            if (!(originListener instanceof ScrollListenerDelegate))
                recentListView.setOnScrollListener(new ScrollListenerDelegate(originListener));
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        }
    }

    /**
     * 给每个RecyclerView最多设置一次ScrollListener事件
     *
     * @param recyclerView
     */
    private void addOnScrollListenerIfNeed(@NonNull RecyclerView recyclerView) {
        if (recyclerView.getTag(R.id.tag_listener_recycler) == null) {
            RecyclerView.OnScrollListener listener = new RecyclerView.OnScrollListener() {
                @Override
                public void onScrollStateChanged(RecyclerView recyclerView, int newState) {
                    if (newState == RecyclerView.SCROLL_STATE_IDLE) {
                        Logger.v("RecyclerView.OnScrollListener SCROLL_STATE_IDLE");
                        bindEventAsync(recyclerView);
                    }
                }
            };
            recyclerView.addOnScrollListener(listener);
            recyclerView.setTag(R.id.tag_listener_recycler, listener);
        }
    }

    /**
     * 添加根view树改变时的监听
     *
     * @param root
     */
    private void createViewTreeObservers(View root) {
        ViewTreeObserver viewTreeObserver = root.getViewTreeObserver();
        viewTreeObserver.addOnWindowFocusChangeListener(this);
        viewTreeObserver.addOnGlobalLayoutListener(this);
        treeObservers.add(viewTreeObserver);
    }

    /**
     * 移除所有window中根view树的监听器
     */
    private void removeViewTreeObservers() {
        for (ViewTreeObserver observer : treeObservers) {
            if (observer != null && observer.isAlive()) {
                observer.removeOnWindowFocusChangeListener(this);
                observer.removeOnGlobalLayoutListener(this);
                observer.removeOnWindowAttachListener(this);
            }
        }
        treeObservers.clear();
    }

    /**
     * 给view添加辅助功能代理事件
     *
     * @param root
     * @param event
     */
    private void addEventToAccessibilityDelegate(View root, ViewEvent event) {
        AccessibilityDelegate delegate = (AccessibilityDelegate) root.getTag(R.id.tag_accessibility);
        if (delegate == null) {
            delegate = new AccessibilityDelegate(this);
            root.setAccessibilityDelegate(delegate);
            root.setTag(R.id.tag_accessibility, delegate);
        }
        delegate.addViewEvent(event);
    }

    /**
     * ListView的OnScrollListener的代理类，用于全部业务的OnScrollListener被正确执行
     */
    public class ScrollListenerDelegate implements AbsListView.OnScrollListener {

        private AbsListView.OnScrollListener delegate;

        public ScrollListenerDelegate(AbsListView.OnScrollListener delegate) {
            this.delegate = delegate;
        }

        @Override
        public void onScrollStateChanged(AbsListView view, int scrollState) {
            if (delegate != null)
                delegate.onScrollStateChanged(view, scrollState);
            if (scrollState == AbsListView.OnScrollListener.SCROLL_STATE_IDLE) {
                bindEventAsync(view);
                Logger.v("AbsListView.OnScrollListener SCROLL_STATE_IDLE");
            }
        }

        @Override
        public void onScroll(AbsListView view, int firstVisibleItem, int visibleItemCount, int totalItemCount) {
            if (delegate != null)
                delegate.onScroll(view, firstVisibleItem, visibleItemCount, totalItemCount);
        }
    }
}
