package com.dada.smart.user.visitor;

import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.util.SparseBooleanArray;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.ListView;

import com.dada.smart.common.ResourceIds;
import com.dada.smart.common.Utils;
import com.dada.smart.user.R;
import com.dada.smart.user.event.Event;
import com.dada.smart.user.event.ViewEvent;
import com.dada.smart.user.event.ViewTreeChild;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 按规则进行view的匹配
 */
public class ViewMatcher {

    //一个recyclerView中出现的次数
    private Map<String, Integer> idCountMap;
    //用于记录view是否被展示过
    private SparseBooleanArray viewShowArray;
    private List<View> visitViews;
    private ResourceIds resourceIds;

    public ViewMatcher(ResourceIds resourceIds) {
        this.resourceIds = resourceIds;
        this.viewShowArray = new SparseBooleanArray();
    }

    /**
     * @param root
     * @param event
     * @return
     */
    public ViewEvent matches(View root, Event event, ListView recentListView, RecyclerView recentRecyclerView) {
        if (root instanceof ListView || root instanceof RecyclerView)
            clearIdCountMap();

        ViewEvent viewEvent;
        if (event.isIdMatcher()) {
            viewEvent = idMatches(root, event, recentListView, recentRecyclerView);
        } else {
            viewEvent = pathMatches(root, event);
        }
        if (viewEvent != null && event.getTypeId() == Event.TYPE_VIEW_SHOW) {
            int eventId = (int) event.getId();
            boolean isShow = viewShowArray.get(eventId);
            //view首次显示事件，确保只触发一次
            if (!isShow) {
                viewShowArray.append(eventId, true);
            } else {
                viewEvent = null;
            }
        }
        return viewEvent;
    }

    /**
     * 路径的方式匹配view,暂不考虑listview和recyclerview中的position问题
     *
     * @param root
     * @param event
     * @return
     */
    private ViewEvent pathMatches(View root, Event event) {
        List<ViewTreeChild> list = event.getViewTreeChildren();
        int viewTreeLevel = (int) root.getTag(R.id.tag_tree_index);
        ViewParent parent = root.getParent();
        if (parent != null) {
            if (parent instanceof ViewGroup) {
                if (viewTreeLevel < list.size()) {
                    ViewTreeChild child = list.get(viewTreeLevel);
                    int childCount = ((ViewGroup) parent).getChildCount();
                    int index = ((ViewGroup) parent).indexOfChild(root);
                    if (TextUtils.equals(child.getClassName(), Utils.simpleName(root)) && childCount == child.getBrothersCount() && index == child.getIndex()) {
                        child.setPassVerify(true);
                    }
                }
            } else {
                //parent不是ViewGroup的情况只有ViewRootImpl一种，也就是DecorView的parent是ViewRootImpl
                list.get(0).setPassVerify(true);
            }
        }
        return event.checkAllTreeViewVerified() ? new ViewEvent(event, root) : null;
    }

    /**
     * id的方式匹配view
     *
     * @param root
     * @param event
     * @return
     */
    private ViewEvent idMatches(final View root, Event event, ListView recentListView, RecyclerView recentRecyclerView) {
        String viewIdName = event.getViewIdName();
        String viewClassName = event.getViewClassName();
        if (viewClassName == null || viewIdName == null) {
            return null;
        }
        boolean isMatch = viewIdName.equals(resourceIds.nameForId(root.getId()))
                && viewClassName.equals(Utils.canonicalName(root));
        if (isMatch) {
            ViewEvent viewEvent = new ViewEvent(event, root);
            if (event.hasPos(R.id.tag_list_pos_index) && recentListView != null) {
                //表达式中需要获取view在ListView中的位置，同时在view树中存在ListView
                int pos = calculatePosition(recentListView.getFirstVisiblePosition(), countKey(recentListView.getId(), event.getId()));
                com.orhanobut.logger.Logger.v("ListView pos=" + pos);
                root.setTag(R.id.tag_list_pos_index, pos);
            } else if (event.hasPos(R.id.tag_recycler_pos_index) && recentRecyclerView != null) {
                //表达式中需要获取view在RecyclerView中的位置，同时在view树中存在RecyclerView
                RecyclerView.LayoutManager layoutManager = recentRecyclerView.getLayoutManager();
                if (layoutManager instanceof LinearLayoutManager) {
                    int pos = calculatePosition(((LinearLayoutManager) layoutManager).findFirstVisibleItemPosition(), countKey(recentRecyclerView.getId(), event.getId()));
                    com.orhanobut.logger.Logger.v("RecyclerView pos=" + pos);
                    root.setTag(R.id.tag_recycler_pos_index, pos);
                }
            } else if (event.hasPos(R.id.tag_pos_index)) {
                //表达式中需要获取view在LinearLayout等不存在复用情况的ViewGroup中的位置
                int pos = getLinearPos(root);
                com.orhanobut.logger.Logger.v("LinearLayout.. pos=" + pos);
                root.setTag(R.id.tag_pos_index, pos);
            }
            return viewEvent;
        } else {
            return null;
        }
    }

    private int getLinearPos(View root) {
        if (visitViews == null)
            visitViews = new ArrayList<>(2);
        if (visitViews.contains(root)) {
            visitViews.clear();
        }
        int index = 0;
        for (View view : visitViews) {
            if (view.getId() == root.getId()) {
                index++;
            }
        }
        visitViews.add(root);
        return index;
    }

    private String countKey(int viewId, long eventId) {
        return viewId + "." + eventId;
    }

    /**
     * 计算事件发生在列表中的位置
     *
     * @param firstVisiblePosition
     * @param countKey
     * @return
     */
    private int calculatePosition(int firstVisiblePosition, String countKey) {
        if (idCountMap == null)
            idCountMap = new HashMap<>(2);
        int count = 0;
        if (idCountMap.containsKey(countKey)) {
            count = idCountMap.get(countKey) + 1;
        }
        idCountMap.put(countKey, count);
        return firstVisiblePosition + count;
    }

    private void clearIdCountMap() {
        if (idCountMap != null)
            idCountMap.clear();
    }
}
