package com.dada.smart.user;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;

import com.dada.smart.common.ActivityMonitor;
import com.dada.smart.common.Utils;
import com.dada.smart.common.expression.Expressions;
import com.dada.smart.common.expression.Result;
import com.dada.smart.user.event.Event;
import com.dada.smart.user.event.EventLogRepository;
import com.dada.smart.user.event.ViewEvent;
import com.dada.smart.user.event.ViewEventListener;
import com.dada.smart.user.http.CacheHttpClient;
import com.dada.smart.user.http.Client;
import com.dada.smart.user.log.LogDatabase;
import com.dada.smart.user.visitor.ActivityVisitor;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by tomkeyzhang on 16/5/18.
 */
public class ViewMonitor extends ActivityMonitor {

    private Map<Activity, ActivityVisitor> activityVisitorMap;
    private EventLogRepository eventLogRepository;
    private ViewEventListener listener;
    private long appId;
    private SharedPreferences preferences;

    public ViewMonitor(Context context, long appId, String server, Client client, EventLogRepository.RequestParam requestParam, boolean debug) {
        super(context, debug);
        this.appId = appId;
        this.preferences = context.getSharedPreferences("app_start.pref", Context.MODE_PRIVATE);
        this.eventLogRepository = new EventLogRepository.Builder(server).logDao(LogDatabase.getDatabase(context).logDao()).client(client).requestParam(requestParam).build();
        this.listener = new EventListener();
        this.activityVisitorMap = new HashMap<>();
    }

    public ViewMonitor(Context context, long appId, String server, EventLogRepository.RequestParam requestParam, boolean debug) {
        this(context, appId, server, new CacheHttpClient(context.getSharedPreferences("manager.pref", Context.MODE_PRIVATE)), requestParam, debug);
    }

    public void update() {
        eventLogRepository.fetchConfig(appId);
        eventLogRepository.sendAppStartEventIfNeed(preferences, System.currentTimeMillis(), appId);
    }

    @Override
    public void onActivityCreated(final Activity activity, Bundle savedInstanceState) {
        super.onActivityCreated(activity, savedInstanceState);
        String activityName = Utils.canonicalName(activity);
        if (eventLogRepository.hasEvent(activityName)) {
            ActivityVisitor activityVisitor = new ActivityVisitor(activity,
                    eventLogRepository.getEvents(activityName),
                    eventLogRepository.getPvEvents(activityName),
                    listener, getResourceIds());
            activityVisitor.onCreate();
            activityVisitorMap.put(activity, activityVisitor);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        super.onActivityStopped(activity);
        //只有当前activity有配置过事件，在销毁的时候才进行一次日志发送
        if (eventLogRepository.hasEvent(Utils.canonicalName(activity)))
            eventLogRepository.sendEvents(appId);
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        super.onActivityDestroyed(activity);
        if (activityVisitorMap.containsKey(activity)) {
            activityVisitorMap.get(activity).onDestroy();
            activityVisitorMap.remove(activity);
        }
    }

    private class EventListener implements ViewEventListener {

        @Override
        public void onEvent(ViewEvent event) {
            ViewMonitor.this.onEvent(event);
        }
    }

    public void onEvent(ViewEvent viewEvent) {
        Event event = viewEvent.getEvent();
        List<Result> results = Expressions.fetchResult(uiThreadObject().get(), event.getExpressions(), viewEvent);
        JSONObject object = new JSONObject();
        for (Result result : results) {
            if (result.isTypeEqual()) {
                if (!result.isLeftEqualsValue())
                    return;//在相等的表达式类型下，只有当右侧表达式的值等于左侧才会触发事件
            } else if (result.isTypeAssign()) {
                try {
                    object.put(result.getLeft(), result.getValue());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }
        eventLogRepository.onEvent(event.getId(), event.getTypeId(), getLastPageClass(), object.toString());
    }

    //    @VisibleForTesting
    //    public void setListener(ViewVisitor.OnEventListener listener) {
    //        this.listener = listener;
    //    }
    //
    //    @VisibleForTesting
    //    public void setPreferences(SharedPreferences preferences) {
    //        this.preferences = preferences;
    //    }


}
