/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.d3x.core.repo;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

public class RepoMaster {

    private final Map<Class<?>,Repo> repoMap = new HashMap<>();

    /**
     * Constructor
     * @param repos     the collection of repos
     */
    public RepoMaster(Collection<Repo<?>> repos) {
        repos.forEach(repo -> {
            var type = repo.getType();
            if (repoMap.containsKey(type)) {
                throw new RuntimeException("Duplicate repos for type: " + type);
            } else {
                this.repoMap.put(type, repo);
            }
        });
    }

    /**
     * Registers a repo if one is not already registered
     * @param repo  the repo to register
     * @param <T>   the repo type
     */
    public synchronized <T> void registerIfAbsent(Repo<T> repo) {
        var type = repo.getType();
        var existing = repoMap.get(type);
        if (existing == null) {
            this.repoMap.put(type, repo);
        }
    }

    /**
     * Returns the repo for the type specified
     * @param type      the data type
     * @param <T>       the repo type
     * @return          the repo match
     */
    @SuppressWarnings("unchecked")
    public <T> Repo<T> repo(Class<T> type) {
        var repo = (Repo<T>)repoMap.get(type);
        if (repo != null) {
            return repo;
        } else {
            throw new RuntimeException("No Repo register for type: " + type);
        }
    }
}
