/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.d3x.core.repo;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * An interface to a repository of objects that supports CRUD and query operations
 *
 * @param <T>   the type for repo
 *
 * @author Xavier Witdouck
 */
public interface Repo<T> {

    /**
     * Returns the type for this repo
     * @return  the type for repo
     */
    Class<T> getType();

    /**
     * Stores a single value into this repo
     * @param value the value to store
     */
    default void put(T value) {
        this.putAll(Collections.singleton(value));
    }

    /**
     * Removes a single value from this repo
     * @param value the value to remove
     */
    default void remove(T value) {
        this.removeAll(Collections.singleton(value));
    }

    /**
     * Stores one or more values in this repo
     * @param values    the values to store
     */
    void putAll(Collection<T> values);

    /**
     * Removes one or more values from this repo
     * @param values    the values to remove
     */
    void removeAll(Collection<T> values);

    /**
     * Returns a list of objects for the request
     * @param request   the request descriptor
     * @return          the list of objects
     */
    <R extends RepoRequest> List<T> select(R request);

}
