/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.d3x.core.util.IO;

/**
 * Utilities class for the database abstraction layer
 *
 * @author Xavier Witdouck
 */
@lombok.extern.slf4j.Slf4j
public class DatabaseUtils {

    private static final Map<String,String> sqlMap = new HashMap<>();

    /**
     * Returns the SQL loaded from a classpath resource if arg starts with "/"
     * @param resource  the classpath resource to load SQL from
     * @return          the SQL expression
     */
    public static String loadSql(String resource) {
        try {
            if (!resource.startsWith("/")) {
                return resource;
            } else if (sqlMap.containsKey(resource)) {
                var sql = sqlMap.get(resource);
                return sql != null ? sql : resource;
            } else {
                log.debug("Loading SQL expression from " + resource);
                var url = Database.class.getResource(resource);
                if (url == null) {
                    throw new RuntimeException("No classpath resource located for path: " + resource);
                } else {
                    var sql = IO.readText(url.openStream());
                    sqlMap.put(resource, sql);
                    return sql;
                }
            }
        } catch (IOException ex) {
            throw new DatabaseException("Failed to load SQL from resource: " + resource, ex);
        }
    }

}
