/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.util.concurrent.Callable;

/**
 * A class to capture timing and record counts for database operations
 *
 * @author Xavier Witdouck
 */
public class DatabaseTiming {

    private long connectMillis;
    private long queryMillis;
    private long resultSetMillis;


    /**
     * Resets this timing entity
     * @return  this timing entity
     */
    public DatabaseTiming reset() {
        this.connectMillis = 0;
        this.queryMillis = 0;
        this.resultSetMillis = 0;
        return this;
    }


    /**
     * Times a callable and tags the resulting time as connect time
     * @param callable  the callable to run
     * @param <T>       the type for callable
     * @return          the result of callable
     */
    <T> T timeConnect(Callable<T> callable) {
        try {
            var t1 = System.currentTimeMillis();
            var result = callable.call();
            var t2 = System.currentTimeMillis();
            this.connectMillis = t2 - t1;
            return result;
        } catch (Exception ex) {
            throw new RuntimeException("Timing operation failed: " + ex.getMessage(), ex);
        }
    }


    /**
     * Times a callable and tags the resulting time as query time
     * @param callable  the callable to run
     * @param <T>       the type for callable
     * @return          the result of callable
     */
    <T> T timeQuery(Callable<T> callable) {
        try {
            var t1 = System.currentTimeMillis();
            var result = callable.call();
            var t2 = System.currentTimeMillis();
            this.queryMillis = t2 - t1;
            return result;
        } catch (Exception ex) {
            throw new RuntimeException("Timing operation failed: " + ex.getMessage(), ex);
        }
    }


    /**
     * Times a callable and tags the resulting time as result set time
     * @param callable  the callable to run
     * @param <T>       the type for callable
     * @return          the result of callable
     */
    <T> T timeResultSet(Callable<T> callable) {
        try {
            var t1 = System.currentTimeMillis();
            var result = callable.call();
            var t2 = System.currentTimeMillis();
            this.resultSetMillis = t2 - t1;
            return result;
        } catch (Exception ex) {
            throw new RuntimeException("Timing operation failed: " + ex.getMessage(), ex);
        }
    }



    @Override()
    public String toString() {
        var template = "DB Timing, connect: %s millis, query: %s millis, resultSet: %s millis";
        return String.format(template, connectMillis, queryMillis, resultSetMillis);
    }


}
