/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.lang.reflect.Type;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import com.d3x.core.util.Option;

/**
 * A convenience container that provides a generic mapping class to a row in a database table.
 *
 * It is not advisable to use this generic container for large volume I/O against a database.
 *
 * @author Xavier Witdouck
 */
public class DatabaseRecord {

    @lombok.Getter()
    private Map<String,Object> values;

    /**
     * Constructor
     * @param values    the key-value pairs for this record
     */
    private DatabaseRecord(Map<String,Object> values) {
        this.values = values;
    }

    /**
     * Returns a newly created record based on what the consumer puts in map
     * @param consumer  the consumer to populate the field key-value pairs
     * @return          the newly created record
     */
    public static DatabaseRecord of(Consumer<Map<String,Object>> consumer) {
        var values = new LinkedHashMap<String,Object>();
        consumer.accept(values);
        return new DatabaseRecord(values);
    }

    /**
     * Returns the number of entries for this record
     * @return  the number of entries
     */
    public int size() {
        return values.size();
    }


    /**
     * Returns the set of field names for record
     * @return      the field names
     */
    public Set<String> names() {
        return Collections.unmodifiableSet(values.keySet());
    }


    /**
     * Returns the field values for this record
     * @return  the field values for record
     */
    public Collection<Object> values() {
        return Collections.unmodifiableCollection(values.values());
    }

    /**
     * Returns an option on the value with the name
     * @param name  the field name
     * @param <T>   the value type
     * @return      the option on value
     */
    @SuppressWarnings("unchecked")
    public <T> Option<T> getValue(String name) {
        return Option.of((T)values.get(name));
    }


    /**
     * Returns an option on the value with the name
     * @param name  the field name
     * @param <T>   the value type
     * @return      the option on value
     */
    @SuppressWarnings("unchecked")
    public <T> T getValueOrFail(String name) {
        var value = (T)values.get(name);
        if (value != null) {
            return value;
        } else {
            throw new IllegalArgumentException("No field entry in record for: " + name);
        }
    }


    /**
     * A DatabaseMapping for the DatabaseRecord class
     */
    static class Mapping implements DatabaseMapping<DatabaseRecord> {

        @Override
        public Type type() {
            return DatabaseRecord.class;
        }

        @Override
        public Mapper<DatabaseRecord> select() {
            var types = new int[100];
            var names = new String[100];
            var count = new AtomicInteger();
            return (rs) -> {
                if (count.get() == 0) {
                    var metaData = rs.getMetaData();
                    var colCount = metaData.getColumnCount();
                    count.set(metaData.getColumnCount());
                    for (int i=0; i<colCount; ++i) {
                        types[i] = metaData.getColumnType(i+1);
                        names[i] = metaData.getColumnName(i+1);
                    }
                }
                var colCount = count.get();
                var values = new LinkedHashMap<String,Object>(colCount);
                for (int i=0; i<colCount; ++i) {
                    var typeCode = types[i];
                    var name = names[i];
                    switch (typeCode) {
                        case Types.BOOLEAN:     values.put(name, DatabaseMapping.readBoolean(rs, i+1).orNull());         break;
                        case Types.INTEGER:     values.put(name, DatabaseMapping.readInt(rs, i+1).orNull());             break;
                        case Types.FLOAT:       values.put(name, DatabaseMapping.readFloat(rs, i+1).orNull());           break;
                        case Types.BIGINT:      values.put(name, DatabaseMapping.readLong(rs, i+1).orNull());            break;
                        case Types.DOUBLE:      values.put(name, DatabaseMapping.readDouble(rs, i+1).orNull());          break;
                        case Types.DATE:        values.put(name, DatabaseMapping.readLocalDate(rs, i+1).orNull());       break;
                        case Types.TIMESTAMP:   values.put(name, DatabaseMapping.readLocalDateTime(rs, i+1).orNull());   break;
                        case Types.VARCHAR:     values.put(name, DatabaseMapping.readString(rs, i+1).orNull());          break;
                        default:                values.put(name, rs.getObject(i+1));                 break;
                    }
                }
                return new DatabaseRecord(values);
            };
        }

        @Override
        public Binder<DatabaseRecord> insert() {
            return (record, stmt) -> DatabaseMapping.bindArgs(stmt, new ArrayList<>(record.values()));
        }
    }

}
