/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.d3x.core.util.IO;

/**
 * An iterator over records in a database that must be closed after use
 * @param <T>   the type for iterator
 */
public interface DatabaseIterator<T> extends Iterator<T>, Closeable {


    /**
     * Extracts all values from this iterator into a list
     * @return  the list of all values extracted from iterator
     */
    default List<T> toList() {
        try {
            final List<T> list = new ArrayList<>();
            while (hasNext()) list.add(next());
            return list;
        } finally {
            IO.close(this);
        }
    }


    /**
     * Extracts all values from this iterator into a set
     * @return  the set of values extracted from set
     */
    default Set<T> toSet() {
        try {
            final Set<T> list = new HashSet<>();
            while (hasNext()) list.add(next());
            return list;
        } finally {
            IO.close(this);
        }
    }


    /**
     * Returns an empty database iterator
     * @param <T>   the iterator type
     * @return      the iterator
     */
    static <T> DatabaseIterator<T> empty() {
        return new DatabaseIterator<T>() {
            @Override
            public void close()  { }

            @Override
            public boolean hasNext() {
                return false;
            }

            @Override
            public T next() {
                return null;
            }
        };
    }

}
