/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.Consumer;

import com.d3x.core.util.IO;
import com.d3x.core.util.Option;

/**
 * A class used to setup and execute a database select
 *
 * @author Xavier Witdouck
 */
public class DatabaseSelect<T> {

    private Database db;
    private Class<T> type;
    private List<Object> args;
    private Option<String> sql;
    private Option<Integer> limit;
    private Option<Duration> timeout;
    private Option<Integer> fetchSize;
    private DatabaseTiming timing;

    /**
     * Constructor
     * @param db        the database reference
     * @param type      the data type for this operation
     */
    DatabaseSelect(Database db, Class<T> type) {
        this.db = db;
        this.type = type;
        this.sql = Option.empty();
        this.limit = Option.empty();
        this.timeout = Option.empty();
        this.args = new ArrayList<>();
        this.timing = new DatabaseTiming();
        this.fetchSize = db.getConfig().getFetchSize();
    }


    /**
     * Binds a custom SQL statement to this select
     * @param sql   the SQL statement or classpath resource to SQL statement
     * @return      this select object
     */
    public DatabaseSelect<T> sql(String sql) {
        this.sql = Option.of(sql);
        return this;
    }


    /**
     * Binds arguments to this select
     * @param args  the arguments for SQL string
     * @return      this select object
     */
    public DatabaseSelect<T> args(Object... args) {
        this.args = Arrays.asList(args);
        return this;
    }


    /**
     * Binds arguments to this select
     * @param args  the arguments for SQL string
     * @return      this select object
     */
    public DatabaseSelect<T> args(Collection<?> args) {
        this.args = new ArrayList<>(args);
        return this;
    }


    /**
     * Sets the query timeout for this select
     * @param duration  the timeout duration
     * @return          this select object
     */
    public DatabaseSelect<T> timeout(Duration duration) {
        this.timeout = Option.of(duration);
        return this;
    }


    /**
     * Sets the statement fetch size for this operation
     * @param fetchSize     the fetch size
     * @see java.sql.ResultSet#setFetchSize(int)
     * @return          this select object
     */
    public DatabaseSelect<T> fetchSize(int fetchSize) {
        this.fetchSize = fetchSize > 0 ? Option.of(fetchSize) : Option.empty();
        return this;
    }


    /**
     * Sets the max number of records to include
     * @param limit    the max number of records, must be > 0
     * @return          this select object
     */
    public DatabaseSelect<T> limit(int limit) {
        this.limit = limit > 0 ? Option.of(limit) : Option.empty();
        return this;
    }


    /**
     * Returns the default mapper for this select
     * @return  the default mapper for this
     */
    private DatabaseMapping.Mapper<T> mapper() {
        return db.mapping(type).select();
    }


    /**
     * Returns the first object matched by this select
     * @return          the optional first object
     */
    public Option<T> first() {
        return first(mapper());
    }


    /**
     * Returns the set of objects that match this select
     * @return          the set of resulting objects
     */
    public Set<T> set() {
        return new HashSet<>(list());
    }


    /**
     * Returns the list of objects that match this select
     * @return          the list of resulting objects
     */
    public List<T> list() {
        return list(mapper());
    }


    /**
     * Returns a iterator of objects that match this select
     * @return          the iterator of resulting objects
     */
    public DatabaseIterator<T> iterator() {
        return iterator(mapper());
    }


    /**
     * Returns the timing instance for this operation
     * @return      the timing instance
     */
    public DatabaseTiming getTiming() {
        return timing;
    }

    /**
     * Returns the SQL expression for this select
     * @return  the SQL expression for select
     */
    private String resolveSql() {
        return sql.map(DatabaseUtils::loadSql).orElse(() -> {
            final DatabaseMapping<T> mapping = db.mapping(type);
            return DatabaseMapping.getSelectSql(mapping);
        });
    }


    /**
     * Returns the first object matched by this select
     * @param mapper    the mapper used to generate objects from ResultSet
     * @return          the optional first object
     */
    public Option<T> first(DatabaseMapping.Mapper<T> mapper) {
        Connection conn = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        final String sql = this.resolveSql();
        try {
            conn = db.getConnection();
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            rs = stmt.executeQuery();
            if (rs.next()) {
                final T result = mapper.map(rs);
                return Option.of(result);
            } else {
                return Option.empty();
            }
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to load first record for sql: " + sql, ex);
        } finally {
            IO.close(rs, stmt, conn);
        }
    }


    /**
     * Returns the record count that match this select
     * @return  the record count
     */
    public int count() {
        Connection conn = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        var sql = this.resolveSql();
        try {
            var count = 0;
            conn = db.getConnection();
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setMaxRows(limit.orElse(0));
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            rs = stmt.executeQuery();
            while (rs.next()) count++;
            return count;
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to load records for sql: " + sql, ex);
        } finally {
            IO.close(rs, stmt, conn);
        }
    }


    /**
     * Iterates over all records that match this query applying each record to the consumer
     * @param consumer  the consumer to accept each record that matches this query
     */
    public void forEach(Consumer<T> consumer) {
        Connection conn = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        var sql = this.resolveSql();
        var limitValue = limit.orElse(Integer.MAX_VALUE);
        var mapper = this.mapper();
        try {
            int count = 0;
            conn = db.getConnection();
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setMaxRows(limit.orElse(0));
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            rs = stmt.executeQuery();
            while (rs.next()) {
                final T record = mapper.map(rs);
                consumer.accept(record);
                if (++count >= limitValue) {
                    break;
                }
            }
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to load records for sql: " + sql, ex);
        } finally {
            IO.close(rs, stmt, conn);
        }
    }


    /**
     * Returns the list of objects that match this select
     * @param mapper    the mapper to generate objects from ResultSet
     * @return          the list of resulting objects
     */
    public List<T> list(DatabaseMapping.Mapper<T> mapper) {
        Connection conn = null;
        PreparedStatement stmt = null;
        var sql = this.resolveSql();
        var limitValue = limit.orElse(Integer.MAX_VALUE);
        try {
            conn = timing.timeConnect(db::getConnection);
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setMaxRows(limit.orElse(0));
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            var rs = timing.timeQuery(stmt::executeQuery);
            return timing.timeResultSet(() -> {
                int count = 0;
                var results = new ArrayList<T>();
                while (rs.next()) {
                    var record = mapper.map(rs);
                    results.add(record);
                    if (++count >= limitValue) {
                        break;
                    }
                }
                return results;
            });
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to load records for sql: " + sql, ex);
        } finally {
            IO.close(stmt, conn);
        }
    }


    /**
     * Returns a iterator over the objects that match this select
     * @param mapper    the mapper to generate objects from ResultSet
     * @return          the iterator of resulting objects
     */
    public DatabaseIterator<T> iterator(DatabaseMapping.Mapper<T> mapper) {
        Connection conn = null;
        PreparedStatement stmt = null;
        var sql = this.resolveSql();
        try {
            conn = timing.timeConnect(db::getConnection);
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setMaxRows(limit.orElse(0));
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            var rs = timing.timeQuery(stmt::executeQuery);
            var limitValue = limit.orElse(Integer.MAX_VALUE);
            return iterator(rs, limitValue, mapper, rs, stmt, conn);
        } catch (Exception ex) {
            IO.close(stmt, conn);
            throw new DatabaseException("Failed to load records for sql: " + sql, ex);
        }
    }


    /**
     * Returns an object generated from a ResultSet according to handler
     * @param handler   the handler that extracts data from result set to create an object
     * @return          the object generated from ResultSet
     */
    public T apply(Handler<T> handler) {
        Connection conn = null;
        PreparedStatement stmt = null;
        var sql = this.resolveSql();
        try {
            conn = timing.timeConnect(() -> db.getConnection());
            stmt = DatabaseMapping.bindArgs(conn.prepareStatement(sql), args);
            stmt.setMaxRows(limit.orElse(0));
            stmt.setQueryTimeout((int)timeout.orElse(Duration.ofSeconds(30)).toSeconds());
            stmt.setFetchSize(fetchSize.orElse(0));
            var rs = timing.timeQuery(stmt::executeQuery);
            return timing.timeResultSet(() -> handler.accept(rs));
        } catch (Exception ex) {
            throw new DatabaseException("Failed to load records for sql: " + sql, ex);
        } finally {
            IO.close(stmt, conn);
        }
    }


    /**
     * Returns a DatabaseIterator over a database ResultSet
     * @param rs        the result set to iterate over
     * @param limit     the max number of records to include
     * @param mapper    the mapper to map objects
     * @return          the newly created iterator
     */
    private DatabaseIterator<T> iterator(ResultSet rs, int limit, DatabaseMapping.Mapper<T> mapper, AutoCloseable... closeables) {
        return new DatabaseIterator<>() {

            private int count;
            private Boolean hasNext;

            @Override
            public void close() {
                IO.close(rs);
                IO.close(closeables);
            }

            @Override
            public boolean hasNext() {
                try {
                    if (hasNext != null) {
                        return hasNext;
                    } else if (count >= limit) {
                        this.close();
                        return false;
                    } else {
                        this.hasNext = rs.next();
                        if (!hasNext) {
                            this.close();
                            return false;
                        } else {
                            return true;
                        }
                    }
                } catch (Throwable t) {
                    this.close();
                    throw new DatabaseException(t.getMessage(), t);
                }
            }

            @Override
            public T next() {
                try {
                    if (!hasNext()) {
                        throw new NoSuchElementException("Database Iterator has been exhausted");
                    } else {
                        final T record = mapper.map(rs);
                        this.hasNext = null;
                        this.count++;
                        return record;
                    }
                } catch (Throwable t) {
                    this.close();
                    throw new DatabaseException(t.getMessage(), t);
                }
            }
        };
    }


    /**
     * An interface to a component that can generate an Object from the contents of a ResultSet
     * @param <T>   the object type
     */
    public interface Handler<T> {

        /**
         * Returns an object generated from the ResultSet
         * @param rs    the result set to create object from
         * @return      the newly created object
         * @throws SQLException if SQL exception raised
         */
        T accept(ResultSet rs) throws SQLException;
    }
}
