/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.d3x.core.util.Option;
import lombok.AccessLevel;

/**
 * A convenience base class for build database operations
 * @param <T>   the type for this operation
 *
 * @author Xavier Witdouck
 */
public abstract class DatabaseOperation<T> {

    /** The database this operation is associated with */
    @lombok.Getter(AccessLevel.PACKAGE)
    private Database db;
    /** The list of arguments for operation, if any */
    @lombok.Getter(AccessLevel.PACKAGE)
    private List<Object> args;
    /** The SQL expression for this operation */
    @lombok.Getter(AccessLevel.PACKAGE)
    private Option<String> sql;
    /** The SQL operation optional timeout */
    @lombok.Getter(AccessLevel.PACKAGE)
    private Option<Duration> timeout;
    /** The result set fetch size */
    @lombok.Getter(AccessLevel.PACKAGE)
    private Option<Integer> fetchSize;
    /** The max number of rows to include  */
    @lombok.Getter(AccessLevel.PACKAGE)
    private Option<Integer> limit;


    /**
     * Constructor
     * @param db    the database reference
     */
    DatabaseOperation(Database db) {
        this.db = db;
        this.sql = Option.empty();
        this.args = new ArrayList<>();
        this.timeout = Option.empty();
        this.limit = Option.empty();
        this.fetchSize = db.getConfig().getFetchSize();
    }


    /**
     * Binds a custom SQL statement to this operation
     * @param sql   the SQL statement or classpath resource to SQL statement
     * @return      this operation
     */
    @SuppressWarnings("unchecked")
    public T sql(String sql) {
        this.sql = Option.of(sql);
        return (T)this;
    }


    /**
     * Binds arguments to this operation
     * @param args  the arguments for SQL string
     * @return      this operation
     */
    @SuppressWarnings("unchecked")
    public T args(Object... args) {
        this.args = Arrays.asList(args);
        return (T)this;
    }


    /**
     * Sets the timeout for this operation
     * @param timeout   the timeout, can be null
     * @return          this operation
     */
    @SuppressWarnings("unchecked")
    public T timeout(Duration timeout) {
        this.timeout = Option.of(timeout);
        return (T)this;
    }


    /**
     * Sets the statement fetch size for this operation
     * @param fetchSize     the fetch size
     * @see java.sql.ResultSet#setFetchSize(int)
     * @return          this operation
     */
    @SuppressWarnings("unchecked")
    public T fetchSize(int fetchSize) {
        this.fetchSize = fetchSize > 0 ? Option.of(fetchSize) : Option.empty();
        return (T)this;
    }


    /**
     * Sets the max number of records to include in operation
     * @param limit    the max number of records, must be > 0
     * @return          this operation
     */
    @SuppressWarnings("unchecked")
    public T limit(int limit) {
        this.limit = limit > 0 ? Option.of(limit) : Option.empty();
        return (T)this;
    }


}
