/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import java.io.InputStream;
import java.io.Reader;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Currency;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicBoolean;

import com.d3x.core.util.Option;

/**
 * A component that defines how a data type is mapped to a database for select, insert, update and delete operations.
 *
 * @param <T>   the type for this mapping
 */
public interface DatabaseMapping<T> {

    /**
     * Returns the data type for this mapping
     * @return  the data type
     */
    Type type();

    /**
     * The mapper to map a row in a result set to some object
     * @return  the mapper to map a row in a result set to some object
     */
    default Mapper<T> select() {
        throw new UnsupportedOperationException("The select operation is not supported for type: " + type());
    }

    /**
     * The binder to bind arguments to a PreparedStatement for affecting inserts
     * @return binder to bind arguments to a PreparedStatement for affecting inserts
     */
    default Binder<T> insert()  {
        throw new UnsupportedOperationException("The insert operation is not supported for type: " + type());
    }

    /**
     * The binder to bind arguments to a PreparedStatement for affecting updates
     * @return binder to bind arguments to a PreparedStatement for affecting updates
     */
    default Binder<T> update() {
        throw new UnsupportedOperationException("The update operation is not supported for type: " + type());
    }

    /**
     * The binder to bind arguments to a PreparedStatement for affecting deletes
     * @return binder to bind arguments to a PreparedStatement for affecting deletes
     */
    default Binder<T> delete() {
        throw new UnsupportedOperationException("The delete operation is not supported for type: " + type());
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Boolean> readBoolean(ResultSet rs, int column) throws SQLException {
        var value = rs.getBoolean(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Boolean> readBoolean(ResultSet rs, String column) throws SQLException {
        var value = rs.getBoolean(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Short> readShort(ResultSet rs, int column) throws SQLException {
        var value = rs.getShort(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Short> readShort(ResultSet rs, String column) throws SQLException {
        var value = rs.getShort(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Integer> readInt(ResultSet rs, int column) throws SQLException {
        var value = rs.getInt(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Integer> readInt(ResultSet rs, String column) throws SQLException {
        var value = rs.getInt(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Long> readLong(ResultSet rs, int column) throws SQLException {
        var value = rs.getLong(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Long> readLong(ResultSet rs, String column) throws SQLException {
        var value = rs.getLong(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Float> readFloat(ResultSet rs, int column) throws SQLException {
        var value = rs.getFloat(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Float> readFloat(ResultSet rs, String column) throws SQLException {
        var value = rs.getFloat(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Double> readDouble(ResultSet rs, int column) throws SQLException {
        var value = rs.getDouble(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Double> readDouble(ResultSet rs, String column) throws SQLException {
        var value = rs.getDouble(column);
        return rs.wasNull() ? Option.empty() : Option.of(value);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<String> readString(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getString(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<String> readString(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getString(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Date> readDate(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getDate(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Date> readDate(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getDate(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize date
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Date> readDate(ResultSet rs, int column, Calendar calendar) throws SQLException {
        return Option.of(rs.getDate(column, calendar));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize date
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Date> readDate(ResultSet rs, String column, Calendar calendar) throws SQLException {
        return Option.of(rs.getDate(column, calendar));
    }


    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<LocalDate> readLocalDate(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getDate(column)).map(Date::toLocalDate);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<LocalDate> readLocalDate(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getDate(column)).map(Date::toLocalDate);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Time> readTime(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getTime(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Time> readTime(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getTime(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize time
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Time> readTime(ResultSet rs, int column, Calendar calendar) throws SQLException {
        return Option.of(rs.getTime(column, calendar));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize time
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Time> readTime(ResultSet rs, String column, Calendar calendar) throws SQLException {
        return Option.of(rs.getTime(column, calendar));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Timestamp> readTimestamp(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getTimestamp(column));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize timestamp
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Timestamp> readTimestamp(ResultSet rs, String column, Calendar calendar) throws SQLException {
        return Option.of(rs.getTimestamp(column, calendar));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @param calendar  the calendar to initialize timestamp
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Timestamp> readTimestamp(ResultSet rs, int column, Calendar calendar) throws SQLException {
        return Option.of(rs.getTimestamp(column, calendar));
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<Timestamp> readTimestamp(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getTimestamp(column));
    }


    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<LocalDateTime> readLocalDateTime(ResultSet rs, int column) throws SQLException {
        return Option.of(rs.getTimestamp(column)).map(Timestamp::toLocalDateTime);
    }

    /**
     * Returns an option on the value read from the ResultSet
     * @param rs        the result set to read from
     * @param column    the column to access
     * @return          the option on value
     * @throws SQLException if there is a SQL error
     */
    static Option<LocalDateTime> readLocalDateTime(ResultSet rs, String column) throws SQLException {
        return Option.of(rs.getTimestamp(column)).map(Timestamp::toLocalDateTime);
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyBoolean(PreparedStatement stmt, int paramIndex, Boolean value) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.INTEGER);
        } else {
            stmt.setBoolean(paramIndex, value);
        }
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyInt(PreparedStatement stmt, int paramIndex, Integer value) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.INTEGER);
        } else {
            stmt.setInt(paramIndex, value);
        }
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyLong(PreparedStatement stmt, int paramIndex, Long value) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.BIGINT);
        } else {
            stmt.setLong(paramIndex, value);
        }
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyDouble(PreparedStatement stmt, int paramIndex, Double value) throws SQLException {
        if (value == null || Double.isNaN(value)) {
            stmt.setNull(paramIndex, Types.DOUBLE);
        } else {
            stmt.setDouble(paramIndex, value);
        }
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyFloat(PreparedStatement stmt, int paramIndex, Float value) throws SQLException {
        if (value == null || Float.isNaN(value)) {
            stmt.setNull(paramIndex, Types.FLOAT);
        } else {
            stmt.setFloat(paramIndex, value);
        }
    }

    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyString(PreparedStatement stmt, int paramIndex, String value) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.VARCHAR);
        } else {
            stmt.setString(paramIndex, value);
        }
    }


    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @throws SQLException if there is a SQL error
     */
    static void applyDate(PreparedStatement stmt, int paramIndex, Date value) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.DATE);
        } else {
            stmt.setDate(paramIndex, value);
        }
    }


    /**
     * Applies the value to the statement, calling PreparedStatement.setNull() if value is null
     * @param stmt          the statement to apply value to
     * @param paramIndex    the parameter index
     * @param value         the value to apply, can be null
     * @param calendar      the calendar to control time zone
     * @throws SQLException if there is a SQL error
     */
    static void applyDate(PreparedStatement stmt, int paramIndex, Date value, Calendar calendar) throws SQLException {
        if (value == null) {
            stmt.setNull(paramIndex, Types.DATE);
        } else {
            stmt.setDate(paramIndex, value, calendar);
        }
    }


    /**
     * Extracts all data from result set into a list using this mapping
     * @param rs        the result set to process
     * @return          the list of results
     * @throws SQLException if there is a SQL exception
     */
    default Set<T> toSet(ResultSet rs) throws SQLException {
        var result = new HashSet<T>();
        var mapper = select();
        while (rs.next()) {
            var record = mapper.map(rs);
            result.add(record);
        }
        return result;
    }


    /**
     * Extracts all data from result set into a list using this mapping
     * @param rs        the result set to process
     * @return          the list of results
     * @throws SQLException if there is a SQL exception
     */
    default List<T> toList(ResultSet rs) throws SQLException {
        var result = new ArrayList<T>();
        var mapper = select();
        while (rs.next()) {
            var record = mapper.map(rs);
            result.add(record);
        }
        return result;
    }


    /**
     * Returns an iterator over the records generated by the ResultSet
     * @param rs    the SQL result set to extract records from
     * @return      the iterator with records
     */
    default Iterator<T> toIterator(ResultSet rs) {
        return DatabaseMapping.iterator(rs, select());
    }


    /**
     * Returns a command separated list of values for an in clause
     * @param values    the list of values, for example List.of("x", "y", "z")
     * @return          the in clause expression, for example ('x', 'y', 'z')
     */
    static String in(Collection<?> values) {
        return in(values, Option.empty());
    }


    /**
     * Returns a command separated list of values for an in clause
     * @param values    the list of values, for example List.of("x", "y", "z")
     * @param zoneId    the optional zone if for formatting LocalDateTime
     * @return          the in clause expression, for example ('x', 'y', 'z')
     */
    static String in(Collection<?> values, Option<ZoneId> zoneId) {
        if (values.isEmpty()) return "()";
        else {
            final StringBuilder in = new StringBuilder("(");
            for (Object value : values) {
                in.append(in.length() > 1 ? ", " : "");
                if (value instanceof String) {
                    in.append("'").append(value).append("'");
                } else if (value instanceof Number) {
                    in.append(value);
                } else if (value instanceof Boolean) {
                    final boolean entry = (Boolean)value;
                    in.append(entry ? "1" : "0");
                } else if (value instanceof Currency) {
                    final Currency entry = (Currency)value;
                    final String text = entry.getCurrencyCode();
                    in.append("'").append(text).append("'");
                } else if (value instanceof LocalDate) {
                    final LocalDate entry = (LocalDate)value;
                    final String text = DateTimeFormatter.ISO_LOCAL_DATE.format(entry);
                    in.append("'").append(text).append("'");
                } else if (value instanceof LocalTime) {
                    final LocalTime entry = (LocalTime)value;
                    final String text = DateTimeFormatter.ISO_LOCAL_TIME.format(entry);
                    in.append("'").append(text).append("'");
                } else if (value instanceof LocalDateTime) {
                    final LocalDateTime entry = (LocalDateTime)value;
                    final LocalDateTime dateTime = zoneId.map(entry::atZone).map(ZonedDateTime::toLocalDateTime).orElse(entry);
                    final String text = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(dateTime);
                    in.append("'").append(text).append("'");
                } else if (value instanceof ZonedDateTime) {
                    final ZonedDateTime entry = (ZonedDateTime)value;
                    final String text = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(entry);
                    in.append("'").append(text).append("'");
                }
            }
            return in.append(")").toString();
        }
    }


    /**
     * Binds an array of arguments to the prepared statement
     * @param stmt  the prepared statement object to bind args to
     * @param args  the array of args to bind to statement
     * @return      the statement
     */
    static PreparedStatement bindArgs(PreparedStatement stmt, List<Object> args) throws DatabaseException {
        return bindArgs(stmt, TimeZone.getDefault(), args);
    }

    /**
     * Binds an array of arguments to the prepared statement
     * @param stmt      the prepared statement object to bind args to
     * @param timeZone  the time zone for timestamp related fields
     * @param args      the array of args to bind to statement
     * @return          the statement
     */
    static PreparedStatement bindArgs(PreparedStatement stmt, TimeZone timeZone, List<Object> args) throws DatabaseException {
        try {
            var tz = timeZone != null ? timeZone : TimeZone.getDefault();
            for (int i=0; i<args.size(); ++i) {
                var paramIndex = i + 1;
                var arg = args.get(i);
                if (arg instanceof String) {
                    stmt.setString(paramIndex, (String) arg);
                } else if (arg instanceof Boolean) {
                    stmt.setBoolean(paramIndex, (Boolean)arg);
                } else if (arg instanceof Integer) {
                    stmt.setInt(paramIndex, (Integer)arg);
                } else if (arg instanceof Long) {
                    stmt.setLong(paramIndex, (Long)arg);
                } else if (arg instanceof Double) {
                    var doubleValue = (Double)arg;
                    if (Double.isNaN(doubleValue)) {
                        stmt.setNull(paramIndex, Types.DOUBLE);
                    } else {
                        stmt.setDouble(paramIndex, (Double) arg);
                    }
                } else if (arg instanceof Enum) {
                    stmt.setString(paramIndex, ((Enum)arg).name());
                } else if (arg instanceof Reader) {
                    stmt.setCharacterStream(paramIndex, (Reader) arg);
                } else if (arg instanceof InputStream) {
                    stmt.setBinaryStream(paramIndex, (InputStream)arg);
                } else if (arg instanceof LocalTime) {
                    stmt.setTime(paramIndex, Time.valueOf((LocalTime)arg));
                } else if (arg instanceof java.util.Date) {
                    var calendar = Calendar.getInstance(tz);
                    stmt.setDate(paramIndex, new java.sql.Date(((java.util.Date) arg).getTime()), calendar);
                } else if (arg instanceof LocalDate) {
                    var calendar = Calendar.getInstance(tz);
                    stmt.setDate(paramIndex, java.sql.Date.valueOf((LocalDate)arg), calendar);
                } else if (arg instanceof Instant) {
                    var calendar = Calendar.getInstance(tz);
                    stmt.setTimestamp(paramIndex, new Timestamp(((Instant)arg).toEpochMilli()), calendar);
                } else if (arg instanceof LocalDateTime) {
                    var zoneId = ZoneId.of(tz.getID());
                    var calendar = Calendar.getInstance(tz);
                    var localDateTime = (LocalDateTime)arg;
                    var zonedDateTime = localDateTime.atZone(zoneId);
                    stmt.setTimestamp(paramIndex, new Timestamp(zonedDateTime.toInstant().toEpochMilli()), calendar);
                } else if (arg instanceof ZonedDateTime) {
                    var dateTime = (ZonedDateTime)arg;
                    var zoneId = dateTime.getZone();
                    var calendar = Calendar.getInstance(TimeZone.getTimeZone(zoneId.getId()));
                    stmt.setTimestamp(paramIndex, new Timestamp(dateTime.toInstant().toEpochMilli()), calendar);
                } else {
                    throw new RuntimeException("Cannot bind arg to PreparedStatement, unsupported arg type:" + arg);
                }
            }
            return stmt;
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to bind SQL args to PreparedStatement", ex);
        }
    }


    /**
     * Returns the SQL resolved from the DatabaseSql annotation on the select method
     * @param mapping       the mapping instance
     * @return              the SQL resolved from method
     * @throws DatabaseException    if no annotation on method
     */
    static <T> String getSelectSql(DatabaseMapping<T> mapping) {
        return getSql(mapping, "select");
    }


    /**
     * Returns the SQL resolved from the DatabaseSql annotation on the insert method
     * @param mapping       the mapping instance
     * @return              the SQL resolved from method
     * @throws DatabaseException    if no annotation on method
     */
    static <T> String getInsertSql(DatabaseMapping<T> mapping) {
        return getSql(mapping, "insert");
    }


    /**
     * Returns the SQL resolved from the DatabaseSql annotation on the update method
     * @param mapping       the mapping instance
     * @return              the SQL resolved from method
     * @throws DatabaseException    if no annotation on method
     */
    static <T> String getUpdateSql(DatabaseMapping<T> mapping) {
        return getSql(mapping, "update");
    }


    /**
     * Returns the SQL resolved from the DatabaseSql annotation on the delete method
     * @param mapping       the mapping instance
     * @return              the SQL resolved from method
     * @throws DatabaseException    if no annotation on method
     */
    static <T> String getDeleteSql(DatabaseMapping<T> mapping) {
        return getSql(mapping, "delete");
    }


    /**
     * Extracts records from the result set using the mapper and returns them as a list
     * @param rs        the result set to extract data from
     * @param mapper    the mapper to generate records
     * @param <T>       the record type
     * @return          the list of records
     */
    static <T> List<T> list(ResultSet rs, Mapper<T> mapper) {
        try {
            var results = new ArrayList<T>();
            while (rs.next()) {
                var record = mapper.map(rs);
                results.add(record);
            }
            return results;
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to extract records from ResultSet", ex);
        }
    }


    /**
     * Extracts records from the result set using the mapper and returns them as a set
     * @param rs        the result set to extract data from
     * @param mapper    the mapper to generate records
     * @param <T>       the record type
     * @return          the set of records
     */
    static <T> Set<T> set(ResultSet rs, Mapper<T> mapper) {
        try {
            var results = new HashSet<T>();
            while (rs.next()) {
                var record = mapper.map(rs);
                results.add(record);
            }
            return results;
        } catch (SQLException ex) {
            throw new DatabaseException("Failed to extract records from ResultSet", ex);
        }
    }


    /**
     * Returns an iterator that extracts records from ResultSet using the mapper
     * @param rs        the result set
     * @param mapper    the record mapper
     * @param <T>       the record type
     * @return          the iterator
     */
    static <T> Iterator<T> iterator(ResultSet rs, Mapper<T> mapper) {
        return new Iterator<>() {
            private AtomicBoolean next = new AtomicBoolean(false);
            @Override
            public boolean hasNext() {
                try {
                    if (next.get()) {
                        return next.get();
                    } else {
                        this.next.set(rs.next());
                        return next.get();
                    }
                } catch (SQLException ex) {
                    throw new DatabaseException(ex.getMessage(), ex);
                }
            }
            @Override
            public T next() {
                try {
                    var result = mapper.map(rs);
                    this.next.set(false);
                    return result;
                } catch (SQLException ex) {
                    throw new DatabaseException("Failed to map record from SQL result set", ex);
                }
            }
        };
    }


    /**
     * Returns the SQL resolved from the DatabaseSql annotation on the named method
     * @param mapping       the mapping instance
     * @param methodName    the method name to inspect for annotation
     * @return              the SQL resolved from method
     * @throws DatabaseException    if no annotation on method
     */
    static String getSql(DatabaseMapping<?> mapping, String methodName) {
        try {
            final Class<?> clazz = mapping.getClass();
            final Method method = clazz.getDeclaredMethod(methodName);
            final DatabaseSql annotation = method.getAnnotation(DatabaseSql.class);
            if (annotation == null) {
                throw new DatabaseException("No 'DatabaseSql' annotation on insert method for: " + clazz);
            } else {
                final String value =  annotation.value();
                return value.startsWith("/") ? DatabaseUtils.loadSql(value) : value;
            }
        } catch (NoSuchMethodException ex) {
            throw new DatabaseException("Unable to resolve SQL from insert mapping", ex);
        }
    }



    /**
     * A Mapper that can create an Object from the current row in a ResultSet
     * @param <T>   the type produced by this Mapper
     */
    @FunctionalInterface()
    interface Mapper<T> {

        /**
         * Returns an record created from the current row in the ResultSet
         * @param rs        the SQL ResultSet reference
         * @return          the newly created record
         * @throws SQLException    if there is an error creating record
         */
        T map(ResultSet rs) throws SQLException;

    }


    /**
     * A Binder that can bind a record to a PreparedStatement object
     * @param <T>   the record type for this Binder
     */
    @FunctionalInterface()
    interface Binder<T> {

        /**
         * Binds the record provided to the PreparedStatement
         * @param record        the record to bind to statement
         * @param stmt          the statement to bind to
         * @throws SQLException     if fails to bind record
         */
        void bind(T record, PreparedStatement stmt) throws SQLException;

    }

}
