/*
 * Copyright (C) 2018-2019 D3X Systems - All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.d3x.core.db;

import javax.sql.DataSource;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Consumer;

import com.d3x.core.util.Consumers;
import com.d3x.core.util.IO;
import com.d3x.core.util.Lazy;
import com.d3x.core.util.LifeCycle;
import com.d3x.core.util.Option;

/**
 * A light weight SQL database abstraction layer using an Object / Functional approach to JDBC
 *
 * @author Xavier Witdouck
 */
@lombok.extern.slf4j.Slf4j
public class Database extends LifeCycle.Base {

    private static DataSourceAdapter dataSourceAdapter = new DataSourceAdapter.Apache();
    private static Lazy<ExecutorService> executor = Lazy.of(() -> Executors.newFixedThreadPool(10));

    private DatabaseConfig config;
    private DataSource dataSource;
    private Consumers<Database> onStart = new Consumers<>();
    private Map<Type,DatabaseMapping<?>> mappingsMap = new HashMap<>();
    private ThreadLocal<Connection> connectionThreadLocal = new ThreadLocal<>();


    /**
     * Constructor
     * @param config    the config supplier
     */
    public Database(DatabaseConfig config) {
        Objects.requireNonNull(config, "The config supplier cannot be null");
        this.config = config;
        this.register(new DatabaseRecord.Mapping());
        this.register(new BasicMapping<>(Types.BIT));
        this.register(new BasicMapping<>(Types.SMALLINT));
        this.register(new BasicMapping<>(Types.INTEGER));
        this.register(new BasicMapping<>(Types.BIGINT));
        this.register(new BasicMapping<>(Types.FLOAT));
        this.register(new BasicMapping<>(Types.DOUBLE));
        this.register(new BasicMapping<>(Types.VARCHAR));
        this.register(new BasicMapping<>(Types.DATE));
        this.register(new BasicMapping<>(Types.TIME));
        this.register(new BasicMapping<>(Types.TIMESTAMP));
    }

    /**
     * Returns a newly created Database instance based on args provided
     * @param config    the database config
     * @return          the newly created database
     */
    public static Database of(DatabaseConfig config) {
        return new Database(config);
    }

    /**
     * Returns a newly created Database instance based on args provided
     * @param config    the database config
     * @param setup     the consumer to perform additional setup
     * @return          the newly created database
     */
    public static Database of(DatabaseConfig config, Consumer<Database> setup) {
        var database = new Database(config);
        if (setup != null) {
            setup.accept(database);
        }
        return database;
    }


    /**
     * Returns the Executor thread pool for Database
     * @return      the Executor thread pool
     */
    static Executor getExecutor() {
        return executor.get();
    }


    /**
     * Sets the default executor for running async database operations
     * @param executor  the executor reference
     */
    public static void setExecutor(ExecutorService executor) {
        Database.executor = Lazy.of(() -> executor);
    }


    /**
     * Sets the data source adapter used to create DataSource objects
     * @param dataSourceAdapter    the data source adapter
     */
    public static void setDataSourceAdapter(DataSourceAdapter dataSourceAdapter) {
        Database.dataSourceAdapter = dataSourceAdapter;
    }


    /**
     * Returns the configuration for this database
     * @return  the database configuration
     */
    public DatabaseConfig getConfig() {
        return config;
    }


    /**
     * Returns a reference to the DataSource
     * @return  the DataSource reference
     */
    public DataSource getDataSource() {
        return Option.of(dataSource).orThrow("The database has not been started for: " + config.getUrl());
    }


    /**
     * Returns a reference to a database connection from the pool
     * @return      the database connection
     * @throws SQLException if fails to get a connection
     */
    Connection getConnection() throws SQLException {
        var conn = connectionThreadLocal.get();
        if (conn != null) {
            return conn;
        } else if (dataSource == null) {
            throw new DatabaseException("The database has not been started");
        } else {
            return dataSource.getConnection();
        }
    }


    /**
     * Adds a onStart consumer to be called when the Database starts up
     * @param onStart   the on-start consumer to call before startup
     * @return          this database
     */
    public Database onStart(Consumer<Database> onStart) {
        this.onStart.attach(onStart);
        return this;
    }


    @Override
    protected void doStart() throws RuntimeException {
        try {
            log.info("Starting database named " + config.getUrl());
            this.dataSource = dataSourceAdapter.create(config);
            this.onStart.accept(this);
        } catch (Exception ex) {
            throw new DatabaseException("Failed to start database component", ex);
        }
    }


    @Override
    protected void doStop() throws RuntimeException {
        try {
            log.info("Stopping database named " + config.getUrl());
            dataSourceAdapter.close(dataSource);
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
    }


    /**
     * Returns true if a mapping for the type exists
     * @param type  the data type
     * @return      true if a mapping exists
     */
    public boolean supports(Class<?> type) {
        return mappingsMap.containsKey(type);
    }


    /**
     * Registers a database mapping for a data type
     * @param mapping   the mapping reference
     * @param <T>       the type for mapping
     */
    public <T> void register(DatabaseMapping<T> mapping) {
        Objects.requireNonNull(mapping, "The mapper cannot be null");
        this.mappingsMap.put(mapping.type(), mapping);
    }


    /**
     * Returns true if the specified table exists
     * @param tableName the table name
     * @return          true if the table exists
     */
    public boolean tableExists(String tableName) {
        ResultSet rs = null;
        Connection conn = null;
        try {
            conn = getDataSource().getConnection();
            var metaData = conn.getMetaData();
            var nameSet = new HashSet<String>(Arrays.asList(tableName, tableName.toLowerCase(), tableName.toUpperCase()));
            for (String name : nameSet) {
                rs = metaData.getTables(null, null, name, null);
                if (rs.next()) {
                    return true;
                }
            }
            return false;
        } catch (SQLException ex) {
            throw new DatabaseException(ex.getMessage(), ex);
        } finally {
            IO.close(rs, conn);
        }
    }


    /**
     * Returns the record count from a select count(*) type query
     * @param sql       the SQL statement or path to classpath resource with SQL
     * @param args      the args if the SQL is a parameterized statement
     * @return          the record count
     */
    public int count(String sql, Object... args) {
        ResultSet rs = null;
        Connection conn = null;
        PreparedStatement stmt = null;
        var timeZone = TimeZone.getDefault();
        var sqlExpression = DatabaseUtils.loadSql(sql);
        try {
            conn = getDataSource().getConnection();
            stmt = conn.prepareStatement(sqlExpression);
            DatabaseMapping.bindArgs(stmt, timeZone, Arrays.asList(args));
            rs = stmt.executeQuery();
            return rs.next() ? rs.getInt(1) : 0;
        } catch (Exception ex) {
            throw new DatabaseException("Failed to execute sql: " + sqlExpression, ex);
        } finally {
            IO.close(rs, stmt, conn);
        }
    }


    public DatabaseExecute exec(String sql) {
        return new DatabaseExecute(this, sql);
    }


    /**
     * Executes the sql definition using java.sql.Statement.execute()
     * @param sql       the SQL statement or path to classpath resource with SQL
     * @return          the update count if applicable
     * @throws DatabaseException    if fails to execute sql
     */
    public Option<Integer> execute(String sql) throws DatabaseException {
        return execute(sql, Option.empty());
    }

    /**
     * Executes the sql definition using java.sql.Statement.execute()
     * @param sql       the SQL statement or path to classpath resource with SQL
     * @param handler   the optional handler for any ResultSets that may be produced
     * @return          the update count if applicable
     * @throws DatabaseException    if fails to execute sql
     */
    public Option<Integer> execute(String sql, Option<Consumer<ResultSet>> handler) throws DatabaseException {
        Connection conn = null;
        Statement stmt = null;
        var sqlExpression = DatabaseUtils.loadSql(sql);
        try {
            conn = getDataSource().getConnection();
            stmt = conn.createStatement();
            var results = stmt.execute(sqlExpression);
            if (results && handler.isPresent()) {
                handler.get().accept(stmt.getResultSet());
                while (stmt.getMoreResults()) {
                    handler.get().accept(stmt.getResultSet());
                }
            }
            var count = stmt.getUpdateCount();
            return count < 0 ? Option.empty() : Option.of(count);
        } catch (Exception ex) {
            throw new DatabaseException("Faile d to execute sql: " + sqlExpression, ex);
        } finally {
            IO.close(stmt, conn);
        }
    }


    /**
     * Performs a SQL executeUpdate() given the parameterized SQL and arguments
     * @param sql       the SQL statement or path to classpath resource with SQL
     * @param args      the arguments to apply to the expression
     * @return          the number of records affected
     */
    public int executeUpdate(String sql, Object... args) {
        return executeUpdate(sql, TimeZone.getDefault(), args);
    }


    /**
     * Performs a SQL executeUpdate() given the parameterized SQL and arguments
     * @param sql       the SQL statement or path to classpath resource with SQL
     * @param timeZone  the time zone to use to store timestamp related fields
     * @param args      the arguments to apply to the expression
     * @return          the number of records affected
     */
    public int executeUpdate(String sql, TimeZone timeZone, Object... args) {
        Connection conn = null;
        PreparedStatement stmt = null;
        var sqlExpression = DatabaseUtils.loadSql(sql);
        try {
            conn = dataSource.getConnection();
            stmt = conn.prepareStatement(sqlExpression);
            DatabaseMapping.bindArgs(stmt, timeZone, Arrays.asList(args));
            return stmt.executeUpdate();
        } catch (Exception ex) {
            throw new DatabaseException("Failed to execute sql: " + sqlExpression, ex);
        } finally {
            IO.close(stmt, conn);
        }
    }


    /**
     * Freestyle function to execute some logic against a database using the connection directly
     * @param handler   the function to take connection and do something
     * @param <R>       the result type for function
     * @return          the result of the function
     */
    public <R> R withConnection(ConnectionHandler<R> handler) throws DatabaseException {
        Connection conn = null;
        try {
            conn = getDataSource().getConnection();
            var connProxy = proxy(conn);
            this.connectionThreadLocal.set(connProxy);
            return handler.apply(connProxy);
        } catch (Exception ex) {
            throw new DatabaseException(ex.getMessage(), ex);
        } finally {
            this.connectionThreadLocal.set(null);
            IO.close(conn);
        }
    }


    /**
     * Freestyle function to execute some logic against a database within a transaction
     * @param handler   the function to take connection and do something
     * @param <R>       the result type for function
     * @return          the result of the function
     */
    public <R> R withTransaction(ConnectionHandler<R> handler) throws DatabaseException {
        try {
            Connection conn = null;
            try {
                conn = getDataSource().getConnection();
                conn.setAutoCommit(false);
                var connProxy = proxy(conn);
                this.connectionThreadLocal.set(connProxy);
                var result = handler.apply(connProxy);
                conn.commit();
                return result;
            } catch (Exception ex) {
                if (conn != null) conn.rollback();
                throw new DatabaseException(ex.getMessage(), ex);
            } finally {
                this.connectionThreadLocal.set(null);
                IO.close(conn);
            }
        } catch (SQLException ex) {
            throw new DatabaseException(ex.getMessage(), ex);
        }
    }


    /**
     * Returns a new select operation for the type specified
     * @param type  the data type for operation
     * @return      the database operation
     */
    public <T> DatabaseSelect<T> select(Class<T> type) {
        return new DatabaseSelect<>(this, type);
    }


    /**
     * Returns a new insert operation for the type specified
     * @param type  the data type for operation
     * @return      the database operation
     */
    public <T> DatabaseUpdate<T> insert(Class<T> type) {
        return new DatabaseUpdate<>(this, type, DatabaseUpdate.Type.INSERT);
    }


    /**
     * Returns a new update operation for the type specified
     * @param type  the data type for operation
     * @return      the database operation
     */
    public <T> DatabaseUpdate<T> update(Class<T> type) {
        return new DatabaseUpdate<>(this, type, DatabaseUpdate.Type.UPDATE);
    }


    /**
     * Returns a new delete operation for the type specified
     * @param type  the data type for operation
     * @return      the database operation
     */
    public <T> DatabaseUpdate<T> delete(Class<T> type) {
        return new DatabaseUpdate<>(this, type, DatabaseUpdate.Type.DELETE);
    }


    /**
     * Returns the mapping for the type specified
     * @param type  the data type for mapping
     * @return      the mapping
     * @throws DatabaseException    if no mapping for type
     */
    @SuppressWarnings("unchecked")
    <T> DatabaseMapping<T> mapping(Class<T> type) {
        var mapping = (DatabaseMapping<T>)mappingsMap.get(type);
        if (mapping == null) {
            throw new DatabaseException("No mapping registered for type: " + type);
        } else {
            return mapping;
        }
    }


    /**
     * Submits the Callable to the Database assigned ExecutorService
     * @param callable  the callable to execute
     * @param <R>       the type for callable
     * @return          the future returned by ExecutorService
     */
    <R> Future<R> submit(Callable<R> callable) {
        return executor.get().submit(callable);
    }


    /**
     * Returns a dynamic proxy over a connection to avoid closing to allow for multiple operations on same connection
     * @param connection    the connection to wrap with a dynamic proxy
     * @return              the dynamic proxy wrapper over the connection
     */
    private Connection proxy(Connection connection) {
        var classLoader = getClass().getClassLoader();
        var interfaces = new Class[] { Connection.class };
        return (Connection)Proxy.newProxyInstance(classLoader, interfaces, (proxy, method, args) -> {
            if (method.getName().equals("close")) {
                return null;
            } else {
                return method.invoke(connection, args);
            }
        });
    }


    /**
     * An interface to a component to implement bespoke functionality against a database
     * @param <R>   the return type for this function
     */
    public interface ConnectionHandler<R> {

        /**
         * Executes some database logic with the connection provided
         * @param connection    the database connection
         * @return              the result of this function
         * @throws SQLException if there is a database access error
         */
        R apply(Connection connection) throws SQLException;

    }


    /**
     * A generic mapping class for single type
     * @param <T>   the data type for mapping
     */
    @lombok.AllArgsConstructor()
    private static class BasicMapping<T> implements DatabaseMapping<T> {

        private int sqlType;

        @Override
        public Type type() {
            switch (sqlType) {
                case Types.BIT:         return Boolean.class;
                case Types.TINYINT:     return Short.class;
                case Types.SMALLINT:    return Short.class;
                case Types.INTEGER:     return Integer.class;
                case Types.BIGINT:      return Long.class;
                case Types.FLOAT:       return Float.class;
                case Types.DOUBLE:      return Double.class;
                case Types.DECIMAL:     return Double.class;
                case Types.VARCHAR:     return String.class;
                case Types.DATE:        return LocalDate.class;
                case Types.TIME:        return LocalTime.class;
                case Types.TIMESTAMP:   return LocalDateTime.class;
                default:    throw new IllegalStateException("Unsupported SQL Type: " + sqlType);
            }
        }

        @Override
        @SuppressWarnings("unchecked")
        public Mapper<T> select() {
            var cal = Calendar.getInstance(TimeZone.getDefault());
            switch (sqlType) {
                case Types.BIT:         return rs -> (T)DatabaseMapping.readBoolean(rs, 1).orNull();
                case Types.TINYINT:     return rs -> (T)DatabaseMapping.readShort(rs, 1).orNull();
                case Types.SMALLINT:    return rs -> (T)DatabaseMapping.readShort(rs, 1).orNull();
                case Types.INTEGER:     return rs -> (T)DatabaseMapping.readInt(rs, 1).orNull();
                case Types.BIGINT:      return rs -> (T)DatabaseMapping.readLong(rs, 1).orNull();
                case Types.FLOAT:       return rs -> (T)DatabaseMapping.readFloat(rs, 1).orNull();
                case Types.DOUBLE:      return rs -> (T)DatabaseMapping.readDouble(rs, 1).orNull();
                case Types.DECIMAL:     return rs -> (T)DatabaseMapping.readDouble(rs, 1).orNull();
                case Types.VARCHAR:     return rs -> (T)rs.getString(1);
                case Types.DATE:        return rs -> (T)DatabaseMapping.readDate(rs, 1, cal).map(Date::toLocalDate).orNull();
                case Types.TIME:        return rs -> (T)DatabaseMapping.readTime(rs, 1, cal).map(Time::toLocalTime).orNull();
                case Types.TIMESTAMP:   return rs -> (T)DatabaseMapping.readTimestamp(rs, 1, cal).map(Timestamp::toLocalDateTime).orNull();
                default:    throw new IllegalStateException("Unsupported SQL Type: " + sqlType);
            }
        }

        @Override
        public Binder<T> insert() {
            return binder();
        }

        @Override
        public Binder<T> delete() {
            return binder();
        }

        /**
         * Returns a newly created binder appropriate for type
         * @return      the newly created binder
         */
        private Binder<T> binder() {
            switch (sqlType) {
                case Types.BIT:         return (v, stmt) -> stmt.setBoolean(1, (Boolean)v);
                case Types.TINYINT:     return (v, stmt) -> stmt.setShort(1, ((Number)v).shortValue());
                case Types.SMALLINT:    return (v, stmt) -> stmt.setShort(1, ((Number)v).shortValue());
                case Types.INTEGER:     return (v, stmt) -> stmt.setInt(1, ((Number)v).intValue());
                case Types.BIGINT:      return (v, stmt) -> stmt.setLong(1, ((Number)v).longValue());
                case Types.FLOAT:       return (v, stmt) -> stmt.setFloat(1, ((Number)v).floatValue());
                case Types.DOUBLE:      return (v, stmt) -> stmt.setDouble(1, ((Number)v).doubleValue());
                case Types.DECIMAL:     return (v, stmt) -> stmt.setDouble(1, ((Number)v).doubleValue());
                case Types.VARCHAR:     return (v, stmt) -> stmt.setString(1, (String)v);
                case Types.DATE:        return (v, stmt) -> stmt.setDate(1, Date.valueOf((LocalDate)v));
                case Types.TIME:        return (v, stmt) -> stmt.setTime(1, Time.valueOf((LocalTime)v));
                case Types.TIMESTAMP:   return (v, stmt) -> stmt.setTimestamp(1, Timestamp.valueOf((LocalDateTime)v));
                default:    throw new IllegalStateException("Unsupported SQL Type: " + sqlType);
            }
        }
    }
}
