/*
 * Decompiled with CFR 0.152.
 */
package org.chromium.media;

import android.media.DeniedByServerException;
import android.media.MediaCrypto;
import android.media.MediaCryptoException;
import android.media.MediaDrm;
import android.media.MediaDrmException;
import android.media.NotProvisionedException;
import android.media.UnsupportedSchemeException;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Handler;
import android.util.Log;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.UUID;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.EntityUtils;
import org.chromium.base.CalledByNative;
import org.chromium.base.JNINamespace;

@JNINamespace(value="media")
public class MediaDrmBridge {
    private static final String TAG = "MediaDrmBridge";
    private static final String SECURITY_LEVEL = "securityLevel";
    private static final String PRIVACY_MODE = "privacyMode";
    private static final String SESSION_SHARING = "sessionSharing";
    private static final String ENABLE = "enable";
    private static final int INVALID_SESSION_ID = 0;
    private MediaDrm mMediaDrm;
    private long mNativeMediaDrmBridge;
    private UUID mSchemeUUID;
    private Handler mHandler;
    private boolean mSingleSessionMode;
    private ByteBuffer mMediaCryptoSession;
    private MediaCrypto mMediaCrypto;
    private HashMap<ByteBuffer, Integer> mSessionIds;
    private HashMap<ByteBuffer, String> mSessionMimeTypes;
    private ArrayDeque<PendingCreateSessionData> mPendingCreateSessionDataQueue;
    private boolean mResetDeviceCredentialsPending;
    private boolean mProvisioningPending;

    private static UUID getUUIDFromBytes(byte[] data) {
        int i;
        if (data.length != 16) {
            return null;
        }
        long mostSigBits = 0L;
        long leastSigBits = 0L;
        for (i = 0; i < 8; ++i) {
            mostSigBits = mostSigBits << 8 | (long)(data[i] & 0xFF);
        }
        for (i = 8; i < 16; ++i) {
            leastSigBits = leastSigBits << 8 | (long)(data[i] & 0xFF);
        }
        return new UUID(mostSigBits, leastSigBits);
    }

    private ByteBuffer getSession(int sessionId) {
        for (ByteBuffer session : this.mSessionIds.keySet()) {
            if (this.mSessionIds.get(session) != sessionId) continue;
            return session;
        }
        return null;
    }

    private MediaDrmBridge(UUID schemeUUID, long nativeMediaDrmBridge, boolean singleSessionMode) throws UnsupportedSchemeException {
        this.mSchemeUUID = schemeUUID;
        this.mMediaDrm = new MediaDrm(schemeUUID);
        this.mNativeMediaDrmBridge = nativeMediaDrmBridge;
        this.mHandler = new Handler();
        this.mSingleSessionMode = singleSessionMode;
        this.mSessionIds = new HashMap();
        this.mSessionMimeTypes = new HashMap();
        this.mPendingCreateSessionDataQueue = new ArrayDeque();
        this.mResetDeviceCredentialsPending = false;
        this.mProvisioningPending = false;
        this.mMediaDrm.setOnEventListener((MediaDrm.OnEventListener)new MediaDrmListener());
        this.mMediaDrm.setPropertyString(PRIVACY_MODE, ENABLE);
        if (!this.mSingleSessionMode) {
            this.mMediaDrm.setPropertyString(SESSION_SHARING, ENABLE);
        }
    }

    private boolean createMediaCrypto() throws NotProvisionedException {
        if (this.mMediaDrm == null) {
            return false;
        }
        assert (!this.mProvisioningPending);
        assert (this.mMediaCryptoSession == null);
        assert (this.mMediaCrypto == null);
        this.mMediaCryptoSession = this.openSession();
        if (this.mMediaCryptoSession == null) {
            Log.e((String)TAG, (String)"Cannot create MediaCrypto Session.");
            return false;
        }
        Log.d((String)TAG, (String)("MediaCrypto Session created: " + this.mMediaCryptoSession));
        try {
            if (MediaCrypto.isCryptoSchemeSupported((UUID)this.mSchemeUUID)) {
                byte[] mediaCryptoSession = this.mMediaCryptoSession.array();
                this.mMediaCrypto = new MediaCrypto(this.mSchemeUUID, mediaCryptoSession);
                assert (this.mMediaCrypto != null);
                Log.d((String)TAG, (String)"MediaCrypto successfully created!");
                this.mSessionIds.put(this.mMediaCryptoSession, 0);
                this.nativeOnMediaCryptoReady(this.mNativeMediaDrmBridge);
                return true;
            }
            Log.e((String)TAG, (String)"Cannot create MediaCrypto for unsupported scheme.");
        }
        catch (MediaCryptoException e) {
            Log.e((String)TAG, (String)"Cannot create MediaCrypto", (Throwable)e);
        }
        this.release();
        return false;
    }

    private ByteBuffer openSession() throws NotProvisionedException {
        assert (this.mMediaDrm != null);
        try {
            byte[] session = this.mMediaDrm.openSession();
            return ByteBuffer.wrap((byte[])session.clone());
        }
        catch (RuntimeException e) {
            Log.e((String)TAG, (String)"Cannot open a new session", (Throwable)e);
            this.release();
            return null;
        }
        catch (NotProvisionedException e) {
            throw e;
        }
        catch (MediaDrmException e) {
            Log.e((String)TAG, (String)"Cannot open a new session", (Throwable)e);
            this.release();
            return null;
        }
    }

    private void closeSession(ByteBuffer session) {
        assert (this.mMediaDrm != null);
        this.mMediaDrm.closeSession(session.array());
    }

    @CalledByNative
    private static boolean isCryptoSchemeSupported(byte[] schemeUUID, String containerMimeType) {
        UUID cryptoScheme = MediaDrmBridge.getUUIDFromBytes(schemeUUID);
        if (containerMimeType.isEmpty()) {
            return MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme);
        }
        return MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme, (String)containerMimeType);
    }

    @CalledByNative
    private static MediaDrmBridge create(byte[] schemeUUID, long nativeMediaDrmBridge) {
        UUID cryptoScheme = MediaDrmBridge.getUUIDFromBytes(schemeUUID);
        if (cryptoScheme == null || !MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme)) {
            return null;
        }
        boolean singleSessionMode = false;
        if (Build.VERSION.RELEASE.equals("4.4")) {
            singleSessionMode = true;
        }
        Log.d((String)TAG, (String)("MediaDrmBridge uses " + (singleSessionMode ? "single" : "multiple") + "-session mode."));
        MediaDrmBridge mediaDrmBridge = null;
        try {
            mediaDrmBridge = new MediaDrmBridge(cryptoScheme, nativeMediaDrmBridge, singleSessionMode);
            Log.d((String)TAG, (String)"MediaDrmBridge successfully created.");
        }
        catch (UnsupportedSchemeException e) {
            Log.e((String)TAG, (String)"Unsupported DRM scheme", (Throwable)e);
        }
        catch (IllegalArgumentException e) {
            Log.e((String)TAG, (String)"Failed to create MediaDrmBridge", (Throwable)e);
        }
        catch (IllegalStateException e) {
            Log.e((String)TAG, (String)"Failed to create MediaDrmBridge", (Throwable)e);
        }
        return mediaDrmBridge;
    }

    @CalledByNative
    private boolean setSecurityLevel(String securityLevel) {
        if (this.mMediaDrm == null || this.mMediaCrypto != null) {
            return false;
        }
        String currentSecurityLevel = this.mMediaDrm.getPropertyString(SECURITY_LEVEL);
        Log.e((String)TAG, (String)("Security level: current " + currentSecurityLevel + ", new " + securityLevel));
        if (securityLevel.equals(currentSecurityLevel)) {
            return true;
        }
        try {
            this.mMediaDrm.setPropertyString(SECURITY_LEVEL, securityLevel);
            return true;
        }
        catch (IllegalArgumentException e) {
            Log.e((String)TAG, (String)("Failed to set security level " + securityLevel), (Throwable)e);
        }
        catch (IllegalStateException e) {
            Log.e((String)TAG, (String)("Failed to set security level " + securityLevel), (Throwable)e);
        }
        Log.e((String)TAG, (String)("Security level " + securityLevel + " not supported!"));
        return false;
    }

    @CalledByNative
    private MediaCrypto getMediaCrypto() {
        return this.mMediaCrypto;
    }

    @CalledByNative
    private void resetDeviceCredentials() {
        this.mResetDeviceCredentialsPending = true;
        MediaDrm.ProvisionRequest request = this.mMediaDrm.getProvisionRequest();
        PostRequestTask postTask = new PostRequestTask(request.getData());
        postTask.execute(new String[]{request.getDefaultUrl()});
    }

    @CalledByNative
    private void release() {
        this.mPendingCreateSessionDataQueue.clear();
        this.mPendingCreateSessionDataQueue = null;
        for (ByteBuffer session : this.mSessionIds.keySet()) {
            this.closeSession(session);
        }
        this.mSessionIds.clear();
        this.mSessionIds = null;
        this.mSessionMimeTypes.clear();
        this.mSessionMimeTypes = null;
        this.mMediaCryptoSession = null;
        if (this.mMediaCrypto != null) {
            this.mMediaCrypto.release();
            this.mMediaCrypto = null;
        }
        if (this.mMediaDrm != null) {
            this.mMediaDrm.release();
            this.mMediaDrm = null;
        }
    }

    private MediaDrm.KeyRequest getKeyRequest(ByteBuffer session, byte[] data, String mime) throws NotProvisionedException {
        assert (this.mMediaDrm != null);
        assert (this.mMediaCrypto != null);
        assert (!this.mProvisioningPending);
        HashMap optionalParameters = new HashMap();
        MediaDrm.KeyRequest request = this.mMediaDrm.getKeyRequest(session.array(), data, mime, 1, optionalParameters);
        String result = request != null ? "successed" : "failed";
        Log.d((String)TAG, (String)("getKeyRequest " + result + "!"));
        return request;
    }

    private void savePendingCreateSessionData(int sessionId, byte[] initData, String mime) {
        Log.d((String)TAG, (String)"savePendingCreateSessionData()");
        this.mPendingCreateSessionDataQueue.offer(new PendingCreateSessionData(sessionId, initData, mime));
    }

    private void processPendingCreateSessionData() {
        Log.d((String)TAG, (String)"processPendingCreateSessionData()");
        assert (this.mMediaDrm != null);
        while (this.mMediaDrm != null && !this.mProvisioningPending && !this.mPendingCreateSessionDataQueue.isEmpty()) {
            PendingCreateSessionData pendingData = this.mPendingCreateSessionDataQueue.poll();
            int sessionId = pendingData.sessionId();
            byte[] initData = pendingData.initData();
            String mime = pendingData.mimeType();
            this.createSession(sessionId, initData, mime);
        }
    }

    private void resumePendingOperations() {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.processPendingCreateSessionData();
            }
        });
    }

    @CalledByNative
    private void createSession(int sessionId, byte[] initData, String mime) {
        Log.d((String)TAG, (String)"createSession()");
        if (this.mMediaDrm == null) {
            Log.e((String)TAG, (String)"createSession() called when MediaDrm is null.");
            return;
        }
        if (this.mProvisioningPending) {
            assert (this.mMediaCrypto == null);
            this.savePendingCreateSessionData(sessionId, initData, mime);
            return;
        }
        boolean newSessionOpened = false;
        ByteBuffer session = null;
        try {
            if (this.mMediaCrypto == null && !this.createMediaCrypto()) {
                this.onSessionError(sessionId);
                return;
            }
            assert (this.mMediaCrypto != null);
            assert (this.mSessionIds.containsKey(this.mMediaCryptoSession));
            if (this.mSingleSessionMode) {
                session = this.mMediaCryptoSession;
                if (this.mSessionMimeTypes.get(session) != null && !this.mSessionMimeTypes.get(session).equals(mime)) {
                    Log.e((String)TAG, (String)"Only one mime type is supported in single session mode.");
                    this.onSessionError(sessionId);
                    return;
                }
            } else {
                session = this.openSession();
                if (session == null) {
                    Log.e((String)TAG, (String)"Cannot open session in createSession().");
                    this.onSessionError(sessionId);
                    return;
                }
                newSessionOpened = true;
                assert (!this.mSessionIds.containsKey(session));
            }
            MediaDrm.KeyRequest request = null;
            request = this.getKeyRequest(session, initData, mime);
            if (request == null) {
                if (newSessionOpened) {
                    this.closeSession(session);
                }
                this.onSessionError(sessionId);
                return;
            }
            this.onSessionCreated(sessionId, this.getWebSessionId(session));
            this.onSessionMessage(sessionId, request);
            if (newSessionOpened) {
                Log.d((String)TAG, (String)("createSession(): Session " + this.getWebSessionId(session) + " (" + sessionId + ") created."));
            }
            this.mSessionIds.put(session, sessionId);
            this.mSessionMimeTypes.put(session, mime);
        }
        catch (NotProvisionedException e) {
            Log.e((String)TAG, (String)"Device not provisioned", (Throwable)e);
            if (newSessionOpened) {
                this.closeSession(session);
            }
            this.savePendingCreateSessionData(sessionId, initData, mime);
            this.startProvisioning();
        }
    }

    private boolean sessionExists(ByteBuffer session) {
        if (this.mMediaCryptoSession == null) {
            assert (this.mSessionIds.isEmpty());
            Log.e((String)TAG, (String)"Session doesn't exist because media crypto session is not created.");
            return false;
        }
        assert (this.mSessionIds.containsKey(this.mMediaCryptoSession));
        if (this.mSingleSessionMode) {
            return this.mMediaCryptoSession.equals(session);
        }
        return !session.equals(this.mMediaCryptoSession) && this.mSessionIds.containsKey(session);
    }

    @CalledByNative
    private void releaseSession(int sessionId) {
        Log.d((String)TAG, (String)("releaseSession(): " + sessionId));
        if (this.mMediaDrm == null) {
            Log.e((String)TAG, (String)"releaseSession() called when MediaDrm is null.");
            return;
        }
        ByteBuffer session = this.getSession(sessionId);
        if (session == null) {
            Log.e((String)TAG, (String)"Invalid sessionId in releaseSession.");
            this.onSessionError(sessionId);
            return;
        }
        this.mMediaDrm.removeKeys(session.array());
        if (!this.mSingleSessionMode) {
            Log.d((String)TAG, (String)("Session " + sessionId + "closed."));
            this.closeSession(session);
            this.mSessionIds.remove(session);
            this.onSessionClosed(sessionId);
        }
    }

    @CalledByNative
    private void updateSession(int sessionId, byte[] key) {
        Log.d((String)TAG, (String)("updateSession(): " + sessionId));
        if (this.mMediaDrm == null) {
            Log.e((String)TAG, (String)"updateSession() called when MediaDrm is null.");
            return;
        }
        ByteBuffer session = this.getSession(sessionId);
        if (!this.sessionExists(session)) {
            Log.e((String)TAG, (String)"Invalid session in updateSession.");
            this.onSessionError(sessionId);
            return;
        }
        try {
            try {
                this.mMediaDrm.provideKeyResponse(session.array(), key);
            }
            catch (IllegalStateException e) {
                Log.e((String)TAG, (String)"Exception intentionally caught when calling provideKeyResponse()", (Throwable)e);
            }
            this.onSessionReady(sessionId);
            Log.d((String)TAG, (String)("Key successfully added for session " + sessionId));
            return;
        }
        catch (NotProvisionedException e) {
            Log.e((String)TAG, (String)"failed to provide key response", (Throwable)e);
        }
        catch (DeniedByServerException e) {
            Log.e((String)TAG, (String)"failed to provide key response", (Throwable)e);
        }
        this.onSessionError(sessionId);
        this.release();
    }

    @CalledByNative
    private String getSecurityLevel() {
        if (this.mMediaDrm == null) {
            Log.e((String)TAG, (String)"getSecurityLevel() called when MediaDrm is null.");
            return null;
        }
        return this.mMediaDrm.getPropertyString(SECURITY_LEVEL);
    }

    private void startProvisioning() {
        Log.d((String)TAG, (String)"startProvisioning");
        assert (this.mMediaDrm != null);
        assert (!this.mProvisioningPending);
        this.mProvisioningPending = true;
        MediaDrm.ProvisionRequest request = this.mMediaDrm.getProvisionRequest();
        PostRequestTask postTask = new PostRequestTask(request.getData());
        postTask.execute(new String[]{request.getDefaultUrl()});
    }

    private void onProvisionResponse(byte[] response) {
        Log.d((String)TAG, (String)"onProvisionResponse()");
        assert (this.mProvisioningPending);
        this.mProvisioningPending = false;
        if (this.mMediaDrm == null) {
            return;
        }
        boolean success = this.provideProvisionResponse(response);
        if (this.mResetDeviceCredentialsPending) {
            this.nativeOnResetDeviceCredentialsCompleted(this.mNativeMediaDrmBridge, success);
            this.mResetDeviceCredentialsPending = false;
        }
        if (success) {
            this.resumePendingOperations();
        }
    }

    boolean provideProvisionResponse(byte[] response) {
        if (response == null || response.length == 0) {
            Log.e((String)TAG, (String)"Invalid provision response.");
            return false;
        }
        try {
            this.mMediaDrm.provideProvisionResponse(response);
            return true;
        }
        catch (DeniedByServerException e) {
            Log.e((String)TAG, (String)"failed to provide provision response", (Throwable)e);
        }
        catch (IllegalStateException e) {
            Log.e((String)TAG, (String)"failed to provide provision response", (Throwable)e);
        }
        return false;
    }

    private void onSessionCreated(final int sessionId, final String webSessionId) {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.nativeOnSessionCreated(MediaDrmBridge.this.mNativeMediaDrmBridge, sessionId, webSessionId);
            }
        });
    }

    private void onSessionMessage(final int sessionId, final MediaDrm.KeyRequest request) {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.nativeOnSessionMessage(MediaDrmBridge.this.mNativeMediaDrmBridge, sessionId, request.getData(), request.getDefaultUrl());
            }
        });
    }

    private void onSessionReady(final int sessionId) {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.nativeOnSessionReady(MediaDrmBridge.this.mNativeMediaDrmBridge, sessionId);
            }
        });
    }

    private void onSessionClosed(final int sessionId) {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.nativeOnSessionClosed(MediaDrmBridge.this.mNativeMediaDrmBridge, sessionId);
            }
        });
    }

    private void onSessionError(final int sessionId) {
        this.mHandler.post(new Runnable(){

            public void run() {
                MediaDrmBridge.this.nativeOnSessionError(MediaDrmBridge.this.mNativeMediaDrmBridge, sessionId);
            }
        });
    }

    private String getWebSessionId(ByteBuffer session) {
        String webSessionId = null;
        try {
            webSessionId = new String(session.array(), "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            Log.e((String)TAG, (String)"getWebSessionId failed", (Throwable)e);
        }
        catch (NullPointerException e) {
            Log.e((String)TAG, (String)"getWebSessionId failed", (Throwable)e);
        }
        return webSessionId;
    }

    public static void addKeySystemUuidMapping(String keySystem, UUID uuid) {
        ByteBuffer uuidBuffer = ByteBuffer.allocateDirect(16);
        uuidBuffer.order(ByteOrder.BIG_ENDIAN);
        uuidBuffer.putLong(uuid.getMostSignificantBits());
        uuidBuffer.putLong(uuid.getLeastSignificantBits());
        MediaDrmBridge.nativeAddKeySystemUuidMapping(keySystem, uuidBuffer);
    }

    private native void nativeOnMediaCryptoReady(long var1);

    private native void nativeOnSessionCreated(long var1, int var3, String var4);

    private native void nativeOnSessionMessage(long var1, int var3, byte[] var4, String var5);

    private native void nativeOnSessionReady(long var1, int var3);

    private native void nativeOnSessionClosed(long var1, int var3);

    private native void nativeOnSessionError(long var1, int var3);

    private native void nativeOnResetDeviceCredentialsCompleted(long var1, boolean var3);

    private static native void nativeAddKeySystemUuidMapping(String var0, ByteBuffer var1);

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private class PostRequestTask
    extends AsyncTask<String, Void, Void> {
        private static final String TAG = "PostRequestTask";
        private byte[] mDrmRequest;
        private byte[] mResponseBody;

        public PostRequestTask(byte[] drmRequest) {
            this.mDrmRequest = drmRequest;
        }

        protected Void doInBackground(String ... urls) {
            this.mResponseBody = this.postRequest(urls[0], this.mDrmRequest);
            if (this.mResponseBody != null) {
                Log.d((String)TAG, (String)("response length=" + this.mResponseBody.length));
            }
            return null;
        }

        private byte[] postRequest(String url, byte[] drmRequest) {
            DefaultHttpClient httpClient = new DefaultHttpClient();
            HttpPost httpPost = new HttpPost(url + "&signedRequest=" + new String(drmRequest));
            Log.d((String)TAG, (String)("PostRequest:" + httpPost.getRequestLine()));
            try {
                httpPost.setHeader("Accept", "*/*");
                httpPost.setHeader("User-Agent", "Widevine CDM v1.0");
                httpPost.setHeader("Content-Type", "application/json");
                HttpResponse response = httpClient.execute((HttpUriRequest)httpPost);
                int responseCode = response.getStatusLine().getStatusCode();
                if (responseCode != 200) {
                    Log.d((String)TAG, (String)("Server returned HTTP error code " + responseCode));
                    return null;
                }
                byte[] responseBody = EntityUtils.toByteArray((HttpEntity)response.getEntity());
                return responseBody;
            }
            catch (ClientProtocolException e) {
                e.printStackTrace();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            return null;
        }

        protected void onPostExecute(Void v) {
            MediaDrmBridge.this.onProvisionResponse(this.mResponseBody);
        }
    }

    private class MediaDrmListener
    implements MediaDrm.OnEventListener {
        private MediaDrmListener() {
        }

        public void onEvent(MediaDrm mediaDrm, byte[] session_array, int event, int extra, byte[] data) {
            if (session_array == null) {
                Log.e((String)MediaDrmBridge.TAG, (String)"MediaDrmListener: Null session.");
                return;
            }
            ByteBuffer session = ByteBuffer.wrap(session_array);
            if (!MediaDrmBridge.this.sessionExists(session)) {
                Log.e((String)MediaDrmBridge.TAG, (String)"MediaDrmListener: Invalid session.");
                return;
            }
            Integer sessionId = (Integer)MediaDrmBridge.this.mSessionIds.get(session);
            if (sessionId == null || sessionId == 0) {
                Log.e((String)MediaDrmBridge.TAG, (String)"MediaDrmListener: Invalid session ID.");
                return;
            }
            switch (event) {
                case 1: {
                    Log.d((String)MediaDrmBridge.TAG, (String)"MediaDrm.EVENT_PROVISION_REQUIRED");
                    break;
                }
                case 2: {
                    Log.d((String)MediaDrmBridge.TAG, (String)"MediaDrm.EVENT_KEY_REQUIRED");
                    if (MediaDrmBridge.this.mProvisioningPending) {
                        return;
                    }
                    String mime = (String)MediaDrmBridge.this.mSessionMimeTypes.get(session);
                    MediaDrm.KeyRequest request = null;
                    try {
                        request = MediaDrmBridge.this.getKeyRequest(session, data, mime);
                    }
                    catch (NotProvisionedException e) {
                        Log.e((String)MediaDrmBridge.TAG, (String)"Device not provisioned", (Throwable)e);
                        MediaDrmBridge.this.startProvisioning();
                        return;
                    }
                    if (request != null) {
                        MediaDrmBridge.this.onSessionMessage(sessionId, request);
                        break;
                    }
                    MediaDrmBridge.this.onSessionError(sessionId);
                    break;
                }
                case 3: {
                    Log.d((String)MediaDrmBridge.TAG, (String)"MediaDrm.EVENT_KEY_EXPIRED");
                    MediaDrmBridge.this.onSessionError(sessionId);
                    break;
                }
                case 4: {
                    Log.d((String)MediaDrmBridge.TAG, (String)"MediaDrm.EVENT_VENDOR_DEFINED");
                    assert (false);
                    break;
                }
                default: {
                    Log.e((String)MediaDrmBridge.TAG, (String)("Invalid DRM event " + event));
                    return;
                }
            }
        }
    }

    private static class PendingCreateSessionData {
        private final int mSessionId;
        private final byte[] mInitData;
        private final String mMimeType;

        private PendingCreateSessionData(int sessionId, byte[] initData, String mimeType) {
            this.mSessionId = sessionId;
            this.mInitData = initData;
            this.mMimeType = mimeType;
        }

        private int sessionId() {
            return this.mSessionId;
        }

        private byte[] initData() {
            return this.mInitData;
        }

        private String mimeType() {
            return this.mMimeType;
        }
    }
}

