package com.cz.library.widget;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.LayoutTransition;
import android.animation.ObjectAnimator;
import android.animation.PropertyValuesHolder;
import android.animation.ValueAnimator;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.support.annotation.DrawableRes;
import android.support.annotation.StringRes;
import android.util.AttributeSet;
import android.util.Log;
import android.util.SparseArray;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.cz.library.R;
import com.cz.library.util.Utils;


/**
 * Created by cz on 15/8/21.
 * 一个自定义LinearLayout,增加tab控制切换
 * tabWidth为0时,依靠verticalPadding设定大小,否则依靠tagWidth为主
 */
public class TabLinearLayout extends DivideLinearLayout {
    public static final int TEMPLATE_NONE=0x00;
    public static final int TEMPLATE_TEXT=0x01;
    public static final int TEMPLATE_IMAGE=0x02;

    public static final int STRIP = 0x00;
    public static final int CLIP = 0x01;
    public static final int FULL = 0x02;
    public static final String TAG = "TabLinearLayout";
    private Drawable tabDrawable;
    private int tabWidth;
    private int tabHeight;
    private int horizontalPadding;
    private int verticalPadding;
    private float fraction;//执行进度
    private int tabMode;
    private ValueAnimator valueAnimator;
    private OnSelectListener listener;
    private LayoutTransition layoutTransition;
    private int selectPosition;//选中位置
    private int lastPosition;//上一次选中位置
    private boolean tabSelectReverse;//是否允许反选
    private final SparseArray<View> templateItems;

    public TabLinearLayout(Context context) {
        this(context, null);
    }

    public TabLinearLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        initTransition();
        setWillNotDraw(false);
        setClipChildren(false);
        this.templateItems=new SparseArray<>();

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.TabLinearLayout);
        setTabImage(a.getDrawable(R.styleable.TabLinearLayout_tll_tabImage));
        setTabWidth((int) a.getDimension(R.styleable.TabLinearLayout_tll_tabWidth, 0));
        setTabHeight((int) a.getDimension(R.styleable.TabLinearLayout_tll_tabHeight, Utils.dip2px(2)));
        setTabMode(a.getInt(R.styleable.TabLinearLayout_tll_tabMode, STRIP));
        setHorizontalPadding((int) a.getDimension(R.styleable.TabLinearLayout_tll_horizontalPadding, 0));
        setVerticalPadding((int) a.getDimension(R.styleable.TabLinearLayout_tll_verticalPadding, 0));
        setTabSelectReverse(a.getBoolean(R.styleable.TabLinearLayout_tll_tabSelectReverse,false));
        a.recycle();
    }

    public void setHorizontalPadding(int padding) {
        this.horizontalPadding = padding;
        invalidate();
    }

    public void setVerticalPadding(int padding) {
        this.verticalPadding = padding;
        invalidate();
    }

    public void setTabWidth(int width) {
        this.tabWidth = width;
        invalidate();
    }

    public void setTabHeight(int height) {
        this.tabHeight = height;
        invalidate();
    }

    public void setTabImage(Drawable drawable) {
        tabDrawable = drawable;
        invalidate();
    }

    public void setTabMode(int mode) {
        this.tabMode = mode;
        ViewParent parent = getParent();
        if (parent instanceof ViewGroup) {
            ViewGroup viewParent = (ViewGroup) parent;
            viewParent.setClipChildren(CLIP != mode);
        }
        invalidate();
    }

    public void setTabSelectReverse(boolean reverse) {
        this.tabSelectReverse=reverse;
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        ViewParent parent = getParent();
        if (parent instanceof ViewGroup) {
            ViewGroup viewParent = (ViewGroup) parent;
            viewParent.setClipChildren(CLIP != tabMode);
        }
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        int childCount = getChildCount();
        for (int i = 0; i < childCount; i++) {
            View childView = getChildAt(i);
            childView.setSelected(i == selectPosition);
            LayoutParams layoutParams= (LayoutParams) childView.getLayoutParams();
            if(TEMPLATE_TEXT==layoutParams.mode&&childView instanceof TextView){
                templateItems.append(layoutParams.mode,childView);
            } else if(TEMPLATE_IMAGE==layoutParams.mode&&childView instanceof TextView) {
                templateItems.append(layoutParams.mode,childView);
            }
        }
    }

    private void initTransition() {
        layoutTransition = new LayoutTransition();
        //view出现时 view自身的动画效果
        ObjectAnimator animator1 = ObjectAnimator.ofFloat(null, "alpha", 0f, 1F).setDuration(layoutTransition.getDuration(LayoutTransition.APPEARING));
        layoutTransition.setAnimator(LayoutTransition.APPEARING, animator1);

        ObjectAnimator animator2 = ObjectAnimator.ofFloat(null, "alpha", 1F, 0f).setDuration(layoutTransition.getDuration(LayoutTransition.DISAPPEARING));
        layoutTransition.setAnimator(LayoutTransition.DISAPPEARING, animator2);
        //view 动画改变时，布局中的每个子view动画的时间间隔
        layoutTransition.setStagger(LayoutTransition.CHANGE_APPEARING, 30);
        layoutTransition.setStagger(LayoutTransition.CHANGE_DISAPPEARING, 30);

        PropertyValuesHolder pvhLeft = PropertyValuesHolder.ofInt("left", 0, 1);
        PropertyValuesHolder pvhTop = PropertyValuesHolder.ofInt("top", 0, 1);
        PropertyValuesHolder pvhRight = PropertyValuesHolder.ofInt("right", 0, 1);
        PropertyValuesHolder pvhBottom = PropertyValuesHolder.ofInt("bottom", 0, 1);
        final ObjectAnimator changeIn = ObjectAnimator.ofPropertyValuesHolder(this, pvhLeft, pvhTop, pvhRight, pvhBottom).
                setDuration(layoutTransition.getDuration(LayoutTransition.CHANGE_APPEARING));
        layoutTransition.setAnimator(LayoutTransition.CHANGE_APPEARING, changeIn);

        final ObjectAnimator changeOut = ObjectAnimator.ofPropertyValuesHolder(this, pvhLeft, pvhTop, pvhRight, pvhBottom).
                setDuration(layoutTransition.getDuration(LayoutTransition.CHANGE_DISAPPEARING));
        layoutTransition.setAnimator(LayoutTransition.CHANGE_DISAPPEARING, changeOut);

        setLayoutTransition(layoutTransition);
    }

    @Override
    public void addView(final View child, int index, ViewGroup.LayoutParams params) {
        super.addView(child, index, params);
        child.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                setSelectPosition(indexOfChild(v));
            }
        });
    }

    @Override
    public void removeView(View view) {
        int index = indexOfChild(view);
        if(selectPosition==index&&0<getChildCount()){
            //当前移除对象,正是选中对象.
            setCurrentPosition(0);
        }
        super.removeView(view);
    }

    @Override
    public void removeViewAt(int index) {
        if(selectPosition==index&&0<getChildCount()){
            //当前移除对象,正是选中对象.
            setCurrentPosition(0);
        }
        super.removeViewAt(index);
    }

    /**
     * 选中其中一个条目
     *
     * @param index
     */
    public void setSelectPosition(int index) {
        View selectView=getChildAt(index);
        if (index != selectPosition) {
            if (null != valueAnimator) {
                valueAnimator.cancel();
            }
            selectPosition = index;
            Log.e("valueAnimator","selectPosition:"+selectPosition);
            valueAnimator =ValueAnimator.ofFloat(1.0f);
            valueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator valueAnimator) {
                    TabLinearLayout.this.fraction = valueAnimator.getAnimatedFraction();
                    Log.e("valueAnimator_update","lastPosition:"+lastPosition+" selectPosition:"+selectPosition);
                    invalidate();
                }
            });
            valueAnimator.addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    super.onAnimationEnd(animation);
                    Log.e("valueAnimator_end","lastPosition:"+lastPosition+" selectPosition:"+selectPosition);
                    lastPosition = selectPosition;
                }
            });
            valueAnimator.start();
            View lastView = getChildAt(lastPosition);
            if (null != selectView) {
                selectView.setSelected(true);
            }
            if (null != lastView) {
                lastView.setSelected(false);
            }
        } else if(tabSelectReverse){
            lastPosition=selectPosition;
            selectView.setSelected(!selectView.isSelected());
        }
        //回调监听
        if (null != listener) {
            listener.onSelectTab(selectView, selectPosition, lastPosition);
        }
    }

    /**
     * 直接设定选中位置
     * @param index
     */
    public void setCurrentPosition(int index){
        lastPosition = selectPosition=index;
        //回调监听
        if (null != listener) {
            listener.onSelectTab(getChildAt(index), selectPosition, lastPosition);
        }
        invalidate();
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        int childCount = getChildCount();
        if (0 >= childCount || null == tabDrawable) return;
        //在控件之上
        View lastView = getChildAt(lastPosition);
        View selectView = getChildAt(selectPosition);
        Rect outRect1 = new Rect();
        Rect outRect2 = new Rect();
        lastView.getHitRect(outRect1);
        selectView.getHitRect(outRect2);
        tabDrawable.setBounds(getBoundLeft(outRect1, outRect2),
                getBoundTop(outRect1, outRect2),
                getBoundRight(outRect1, outRect2),
                getBoundBottom(outRect1, outRect2));
        tabDrawable.draw(canvas);
    }


    private int getBoundLeft(Rect rect1, Rect rect2) {
        int orientation = getOrientation();
        int start = 0, end = 0;
        if (HORIZONTAL == orientation) {
            if (0 < tabWidth) {
                //width决定
                start = rect1.centerX() - tabWidth / 2;
                end = rect2.centerX() - tabWidth / 2;
            } else {
                //内padding决定
                start = rect1.left + horizontalPadding;
                end = rect2.left + horizontalPadding;
            }
        } else if (VERTICAL == orientation) {
            //纵向左边距
            switch (tabMode) {
                case STRIP:
                    //width决定,靠底
                    start = rect1.right - verticalPadding - tabHeight;
                    end = rect2.right - verticalPadding - tabHeight;
                    break;
                case CLIP:
                    start = rect1.right + verticalPadding;
                    end = rect2.right + verticalPadding;
                    break;
                case FULL:
                    start = rect1.left + verticalPadding;
                    end = rect2.left + verticalPadding;
                    break;
            }
        }
        return start + (int) ((end - start) * fraction);
    }

    /**
     * 获得绘制顶点
     * 获得绘制顶点
     *
     * @param rect1
     * @param rect2
     * @return
     */
    private int getBoundTop(Rect rect1, Rect rect2) {
        int start = 0, end = 0;
        int orientation = getOrientation();
        if (HORIZONTAL == orientation) {
            switch (tabMode) {
                case STRIP:
                    //width决定,靠底
                    start = rect1.bottom - verticalPadding - tabHeight;
                    end = rect2.bottom - verticalPadding - tabHeight;
                    break;
                case CLIP:
                    start = rect1.bottom + verticalPadding;
                    end = rect2.bottom + verticalPadding;
                    break;
                case FULL:
                    start = rect1.top + verticalPadding;
                    end = rect2.top + verticalPadding;
                    break;
            }
        } else if (VERTICAL == orientation) {
            //width决定,靠底
            if (0 < tabWidth) {
                start = rect1.centerY() - tabWidth / 2;
                end = rect2.centerY() - tabWidth / 2;
            } else {
                start = rect1.top + horizontalPadding;
                end = rect2.top + horizontalPadding;
            }
        }
        return start + (int) ((end - start) * fraction);
    }

    private int getBoundRight(Rect rect1, Rect rect2) {
        int orientation = getOrientation();
        int start = 0, end = 0;
        if (HORIZONTAL == orientation) {
            if (0 < tabWidth) {
                //width决定
                start = rect1.centerX() + tabWidth / 2;
                end = rect2.centerX() + tabWidth / 2;
            } else {
                //内padding决定
                start = rect1.right - horizontalPadding;
                end = rect2.right - horizontalPadding;
            }
        } else if (VERTICAL == orientation) {
            //纵向左边距
            switch (tabMode) {
                case STRIP:
                    //width决定,靠底
                    start = rect1.right - verticalPadding;
                    end = rect2.right - verticalPadding;
                    break;
                case CLIP:
                    start = rect1.right + verticalPadding + tabHeight;
                    end = rect2.right + verticalPadding + tabHeight;
                    break;
                case FULL:
                    start = rect1.right - verticalPadding;
                    end = rect2.right - verticalPadding;
                    break;
            }
        }
        return start + (int) ((end - start) * fraction);
    }

    private int getBoundBottom(Rect rect1, Rect rect2) {
        int start = 0, end = 0;
        int orientation = getOrientation();
        if (HORIZONTAL == orientation) {
            switch (tabMode) {
                case STRIP:
                    //width决定,靠底
                    start = rect1.bottom - verticalPadding;
                    end = rect2.bottom - verticalPadding;
                    break;
                case CLIP:
                    start = rect1.bottom + verticalPadding + tabHeight;
                    end = rect2.bottom + verticalPadding + tabHeight;
                    break;
                case FULL:
                    start = rect1.bottom - verticalPadding;
                    end = rect2.bottom - verticalPadding;
                    break;
            }
        } else if (VERTICAL == orientation) {
            //width决定,靠底
            if (0 < tabWidth) {
                start = rect1.centerY() + tabWidth / 2;
                end = rect2.centerY() + tabWidth / 2;
            } else {
                start = rect1.bottom - horizontalPadding;
                end = rect2.bottom - horizontalPadding;
            }
        }
        return start + (int) ((end - start) * fraction);
    }


    public void scrollToPosition(int position,int fraction){
        this.lastPosition=selectPosition;
        this.selectPosition=position;
        this.fraction=fraction;
        invalidate();
    }

    public int getSelectPosition(){
        return selectPosition;
    }

    public void addTemplateTextView(String text){
        TextView textView = getTemplateTextView();
        if(null!=textView){
            textView.setText(text);
            addView(textView);
        }
    }

    public void addTemplateTextView(@StringRes int res){
        TextView textView = getTemplateTextView();
        if(null!=textView){
            textView.setText(res);
            addView(textView);
        }
    }

    private TextView getTemplateTextView() {
        TextView newTextView=null;
        View childView = templateItems.get(TEMPLATE_TEXT);
        if(null!=childView){
            TextView textView= (TextView) childView;
            newTextView=new TextView(getContext());
            newTextView.setTextColor(textView.getTextColors());
            newTextView.setGravity(textView.getGravity());
            newTextView.setTextSize(TypedValue.COMPLEX_UNIT_PX,textView.getTextSize());
            Drawable newDrawable = textView.getBackground().getConstantState().newDrawable();
            if(null!=newDrawable){
                if(Build.VERSION.SDK_INT<Build.VERSION_CODES.JELLY_BEAN){
                    newTextView.setBackgroundDrawable(newDrawable);
                } else {
                    newTextView.setBackground(newDrawable);
                }
            }
            newTextView.setPadding(textView.getPaddingLeft(),
                    textView.getPaddingTop(),
                    textView.getPaddingRight(),
                    textView.getPaddingBottom());
        }
        return newTextView;
    }

    public void addTemplateImageView(@DrawableRes int res){
        ImageView templateImageView = getTemplateImageView();
        if(null!=templateImageView){
            templateImageView.setImageResource(res);
        }
    }

    public void addTemplateImageView(Drawable drawable){
        ImageView templateImageView = getTemplateImageView();
        if(null!=templateImageView){
            templateImageView.setImageDrawable(drawable);
        }
    }

    public void setTabText(int index,String text){
        if(0<=index&&index<getChildCount()){
            View childView=getChildAt(index);
            if(null!=childView&&childView instanceof TextView){
                TextView textView= (TextView) childView;
                textView.setText(text);
            }
        }
    }

    public void setTabText(int index,@StringRes int res){
        if(0<=index&&index<getChildCount()){
            View childView=getChildAt(index);
            if(null!=childView&&childView instanceof TextView){
                TextView textView= (TextView) childView;
                textView.setText(res);
            }
        }
    }

    private ImageView getTemplateImageView() {
        ImageView newImageView= null;
        View childView = templateItems.get(TEMPLATE_IMAGE);
        if(null!=childView){
            ImageView imageView= (ImageView) childView;
            newImageView=new ImageView(getContext());
            newImageView.setScaleType(imageView.getScaleType());
            Drawable newDrawable = imageView.getBackground().getConstantState().newDrawable();
            if(null!=newDrawable){
                if(Build.VERSION.SDK_INT<Build.VERSION_CODES.JELLY_BEAN){
                    newImageView.setBackgroundDrawable(newDrawable);
                } else {
                    newImageView.setBackground(newDrawable);
                }
            }
            newImageView.setPadding(imageView.getPaddingLeft(),
                    imageView.getPaddingTop(),
                    imageView.getPaddingRight(),
                    imageView.getPaddingBottom());
        }
        return newImageView;
    }

    public void setOnSelectListener(OnSelectListener listener) {
        this.listener = listener;
    }

    public interface OnSelectListener {
        void onSelectTab(View v, int position, int lastPosition);
    }

    @Override
    protected LinearLayout.LayoutParams generateLayoutParams(ViewGroup.LayoutParams p) {
        return new TabLinearLayout.LayoutParams(p);
    }

    @Override
    public LinearLayout.LayoutParams generateLayoutParams(AttributeSet attrs) {
        return new TabLinearLayout.LayoutParams(getContext(),attrs);
    }

    public static class LayoutParams extends LinearLayout.LayoutParams{
        public int mode;
        public LayoutParams(Context c, AttributeSet attrs) {
            super(c, attrs);
            TypedArray a = c.obtainStyledAttributes(attrs, R.styleable.TabLinearLayout);
            mode = a.getInt(R.styleable.TabLinearLayout_layout_templateView, TEMPLATE_NONE);
            a.recycle();
        }

        public LayoutParams(ViewGroup.LayoutParams p) {
            super(p);
            this.mode = TEMPLATE_NONE;
        }
    }


}
