package com.cz.library.widget;

import android.animation.ValueAnimator;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.SweepGradient;
import android.os.Build;
import android.os.Bundle;
import android.os.Parcelable;
import android.support.annotation.ArrayRes;
import android.text.TextPaint;
import android.util.AttributeSet;
import android.view.View;

import com.cz.library.R;


/***
 * 自定义圆弧进度条
 * 
 * @author cz
 */
public class ProgressView extends View {
    public Paint ringPaint;
    public Paint progressPaint;
    public Paint backgroundPaint;
    public TextPaint textPaint;
    private float ringPadding;
    private int[] ringColorArray;
    private int[] levelArray;
    private int currentValue;
    private float currentDegrees;
    private int animDuration;
    private int progressStartAngle;

    public ProgressView(Context context) {
        this(context, null,R.attr.progressView);
    }


    public ProgressView(Context context, AttributeSet attrs) {
        this(context, attrs, R.attr.progressView);
    }

    public ProgressView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        progressPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        progressPaint.setStyle(Paint.Style.STROKE);
        ringPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        ringPaint.setStyle(Paint.Style.STROKE);
        backgroundPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        textPaint = new TextPaint(Paint.ANTI_ALIAS_FLAG);
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.ProgressView,defStyleAttr,R.style.ProgressView);
        setTextSize(a.getDimensionPixelOffset(R.styleable.ProgressView_pv_textSize,0));
        setTextColor(a.getColor(R.styleable.ProgressView_pv_textColor,Color.WHITE));
        setRingColor(a.getColor(R.styleable.ProgressView_pv_ringColor,Color.DKGRAY));
        setRingColorArray(a.getResourceId(R.styleable.ProgressView_pv_ringColorRes,NO_ID));
        setRingSize(a.getDimension(R.styleable.ProgressView_pv_ringSize,0));
        setRingPadding(a.getDimension(R.styleable.ProgressView_pv_ringPadding,0));
        setLevelArray(a.getResourceId(R.styleable.ProgressView_pv_levelArray,NO_ID));
        setCircleBackgroundColor(a.getColor(R.styleable.ProgressView_pv_circleBackgroundColor,Color.WHITE));
        setAnimDuration(a.getInteger(R.styleable.ProgressView_pv_animDuration,1000));
        setProgressPaintRound(a.getBoolean(R.styleable.ProgressView_pv_progressPaintRound,false));
        setProgressStartAngle(a.getInteger(R.styleable.ProgressView_pv_progressStartAngle,0));
        a.recycle();
    }

    public void setTextSize(int textSize) {
        textPaint.setTextSize(textSize);
        invalidate();
    }

    public void setTextColor(int color) {
        textPaint.setColor(color);
        invalidate();
    }

    public void setRingColor(int color) {
        ringPaint.setColor(color);
        invalidate();
    }

    public void setRingColorArray(@ArrayRes int res) {
        setRingColorArray(getResources().getIntArray(res));
    }

    public void setRingColorArray(int[] colorArray){
        this.ringColorArray=colorArray;
        invalidate();
    }

    public void setLevelArray(@ArrayRes int res){
        if(NO_ID!=res){
            setLevelArray(getResources().getIntArray(res));
        }
    }

    public void setLevelArray(int[] levelArray){
        this.levelArray=levelArray;
        this.currentValue=levelArray[0];
        invalidate();
    }

    public void setCurrentValue(int value){
        int itemPosition = getItemPosition(value)-1;
        int itemDegrees = 360 / (levelArray.length - 1);
        int startValue = levelArray[itemPosition];
        int itemValue=itemPosition<levelArray.length-1?levelArray[itemPosition+1]-startValue:startValue;
        float fraction=(value-startValue)*1.0f/itemValue;
        currentDegrees=itemPosition*itemDegrees+fraction*itemDegrees;
        currentValue=value;
        invalidate();
    }

    public void setProgressPaintRound(boolean round) {
        progressPaint.setStrokeCap(round?Paint.Cap.ROUND:Paint.Cap.BUTT);
    }

    public void setProgressStartAngle(int angle) {
        progressStartAngle=angle;
    }

    public void setRingSize(float size) {
        ringPaint.setStrokeWidth(size);
        progressPaint.setStrokeWidth(size);
        invalidate();
    }

    public void setRingPadding(float padding) {
        this.ringPadding=padding;
        requestLayout();
    }

    public void setCircleBackgroundColor(int color) {
        this.backgroundPaint.setColor(color);
        invalidate();
    }

    public void setAnimDuration(int duration) {
        this.animDuration=duration;
    }

    public int getMaxValue(){
        int value=0;
        if(null!=levelArray&&0<levelArray.length){
            value=levelArray[levelArray.length-1];
        }
        return value;
    }

    @Override
    public void invalidate() {
        if(hasWindowFocus()) super.invalidate();
    }

    @Override
    public void postInvalidate(int left, int top, int right, int bottom) {
        if(hasWindowFocus()) super.postInvalidate(left, top, right, bottom);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        progressPaint.setShader(new SweepGradient(w/2 , h/2, ringColorArray, null));
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int widthSpecMode = MeasureSpec.getMode(widthMeasureSpec);
        int heightSpecMode = MeasureSpec.getMode(heightMeasureSpec);
        float width=MeasureSpec.getSize(widthMeasureSpec);
        float height=MeasureSpec.getSize(heightMeasureSpec);
        String text=String.valueOf(currentValue);
        float textWidth = textPaint.measureText(text);
        Rect bounds=new Rect();
        textPaint.getTextBounds(text,0,text.length(),bounds);
        float size=Math.max(textWidth,bounds.height());
        if (MeasureSpec.AT_MOST==widthSpecMode) {
            //动态计算
            width=size+ringPadding+ progressPaint.getStrokeWidth();
        }
        if(MeasureSpec.AT_MOST==heightSpecMode){
            height=size+ringPadding+ progressPaint.getStrokeWidth();
        }
        setMeasuredDimension(resolveSize((int) width,widthMeasureSpec), resolveSize((int) height,heightMeasureSpec));
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        int width = getWidth();
        int height = getHeight();
        //draw background color
        canvas.drawCircle(width/2,height/2,Math.min(width,height)/2,backgroundPaint);
        //draw text
        String text = String.valueOf(currentValue);
        canvas.drawText(text,(width-textPaint.measureText(text,0,text.length()))/2, (height-(textPaint.descent() + textPaint.ascent()))/2, textPaint);
        //draw ring
        float strokeWidth = progressPaint.getStrokeWidth()/2;
        canvas.drawCircle(width/2,height/2,Math.min(width,height)/2-strokeWidth,ringPaint);
        //draw progress ring
        canvas.save();
        canvas.rotate(270+progressStartAngle,width/2,height/2);
        if(Build.VERSION.SDK_INT<Build.VERSION_CODES.LOLLIPOP){
            canvas.drawArc(new RectF(strokeWidth,strokeWidth,width-strokeWidth,height-strokeWidth), 0, currentDegrees, false, progressPaint);
        } else {
            canvas.drawArc(strokeWidth,strokeWidth,width-strokeWidth,height-strokeWidth, 0, currentDegrees, false, progressPaint);
        }
        canvas.restore();
    }

    public void setLevelValueTo(final int endValue){
        //current value->value
        final int startValue=currentValue;
        ValueAnimator valueAnimator = ValueAnimator.ofInt(startValue, endValue);
        valueAnimator.setDuration(animDuration);
        valueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                setCurrentValue((int) valueAnimator.getAnimatedValue());
            }
        });
        valueAnimator.start();
    }

    /**
     * 使用二分查找法,根据firstVisiblePosition找到SelectPositions中的位置
     *
     * @return
     */
    private int getItemPosition(int currentValue){
        int[] positions = levelArray;
        int start = 0, end = positions.length - 1, result = -1;
        while (start <= end) {
            int middle = (start + end) / 2;
            if (currentValue == positions[middle]) {
                result = middle + 1;
                break;
            } else if (currentValue < positions[middle]) {
                end = middle - 1;
            } else {
                start = middle + 1;
            }
        }
        return -1==result?start:result;
    }

    @Override
    protected Parcelable onSaveInstanceState() {
        Bundle bundle=new Bundle();
        bundle.putParcelable("super",super.onSaveInstanceState());
        bundle.putInt("value",currentValue);
        bundle.putFloat("degrees",currentDegrees);
        bundle.putFloat("ringPadding",ringPadding);
        bundle.putIntArray("levelArray",levelArray);
        bundle.putIntArray("colorArray",ringColorArray);
        bundle.putFloat("textSize",textPaint.getTextSize());
        bundle.putInt("textColor",textPaint.getColor());
        return bundle;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        Bundle bundle= (Bundle) state;
        Parcelable superState = bundle.getParcelable("super");
        super.onRestoreInstanceState(superState);
        currentValue=bundle.getInt("value");
        currentDegrees=bundle.getFloat("degrees");
        ringPadding=bundle.getFloat("ringPadding");
        levelArray=bundle.getIntArray("levelArray");
        ringColorArray=bundle.getIntArray("colorArray");
        textPaint.setTextSize(bundle.getFloat("textSize"));
        textPaint.setColor(bundle.getInt("textColor"));
    }
}