package com.cz.library.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.text.InputType;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.BaseInputConnection;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputConnection;
import android.view.inputmethod.InputMethodManager;

import com.cz.library.R;

import java.util.ArrayList;

/**
 * Created by cz on 1/12/17.
 * 一个隐私加密锁
 */
public class PrivacyLockView extends View {
    public static final int AUTO=0;
    public static final int ITEM_PADDING=1;

    private final StringBuilder editable;
    private final ArrayList<OnTextChangedListener> listeners;
    private OnTextSubmitListener submitListener;
    private Drawable itemDrawable;
    private int itemPadding;
    private int itemCount;
    private Drawable privacyDrawable;
    private int privacyDrawableWidth;
    private int privacyDrawableHeight;
    private int itemDrawableSize;
    private int intervalMode;

    public PrivacyLockView(Context context) {
        this(context,null);
    }

    public PrivacyLockView(Context context, AttributeSet attrs) {
        this(context, attrs,0);
    }

    public PrivacyLockView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        listeners=new ArrayList<>();
        editable=new StringBuilder();
        setFocusableInTouchMode(true);
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.PrivacyLockView);
        setItemDrawable(a.getDrawable(R.styleable.PrivacyLockView_pv_itemDrawable));
        setItemDrawableSize((int) a.getDimension(R.styleable.PrivacyLockView_pv_itemDrawableSize,0));
        setItemPadding((int) a.getDimension(R.styleable.PrivacyLockView_pv_itemPadding,0));
        setItemCount(a.getInteger(R.styleable.PrivacyLockView_pv_itemCount,4));
        setPrivacyDrawable(a.getDrawable(R.styleable.PrivacyLockView_pv_privacyDrawable));
        setPrivacyDrawableWidth((int) a.getDimension(R.styleable.PrivacyLockView_pv_privacyDrawableWidth,0));
        setPrivacyDrawableHeight((int) a.getDimension(R.styleable.PrivacyLockView_pv_privacyDrawableHeight,0));
        setIntervalMode(a.getInt(R.styleable.PrivacyLockView_pv_intervalMode,AUTO));
        a.recycle();
    }

    public void setItemDrawable(Drawable drawable) {
        this.itemDrawable=drawable;
        invalidate();
    }

    public void setItemPadding(int itemPadding) {
        this.itemPadding=itemPadding;
        invalidate();
    }

    public void setItemCount(int count) {
        this.itemCount=count;
        requestLayout();
    }

    public void setPrivacyDrawable(Drawable drawable) {
        this.privacyDrawable=drawable;
        invalidate();
    }

    public void setPrivacyDrawableWidth(int drawableWidth) {
        this.privacyDrawableWidth=drawableWidth;
        invalidate();
    }

    public void setItemDrawableSize(int size) {
        this.itemDrawableSize=size;
        invalidate();
    }

    public void setPrivacyDrawableHeight(int drawableHeight) {
        this.privacyDrawableHeight=drawableHeight;
        invalidate();
    }

    public void setIntervalMode(int mode) {
        this.intervalMode=mode;
        invalidate();
    }

    private void appendEditText(CharSequence text) {
        if(editable.length()<itemCount){
            editable.append(text);
            for(OnTextChangedListener listener:listeners){
                listener.onTextChanged(editable,editable.length()-1,editable.length());
            }
            if(editable.length()==itemCount){
                hideSortInput();
                if(null!=submitListener){
                    submitListener.onSubmit(editable);
                }
            }
            invalidate();
        }
    }

    private void deleteLastEditText() {
        if(!TextUtils.isEmpty(editable)){
            editable.deleteCharAt(editable.length()-1);
            for(OnTextChangedListener listener:listeners){
                listener.onTextChanged(editable,editable.length()+1,editable.length());
            }
            invalidate();
        }
    }

    public void clearEditText(){
        editable.delete(0,editable.length());
        invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        int measureWidth = MeasureSpec.getSize(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int measureHeight=getPaddingTop()+getPaddingBottom();
        if(MeasureSpec.EXACTLY==heightMode){
            measureHeight=MeasureSpec.getSize(heightMeasureSpec);
        } else {
            if(null!=itemDrawable){
                if(0==itemDrawableSize||AUTO==intervalMode){
                    measureHeight+= (measureWidth - getPaddingLeft() - getPaddingRight()-(itemCount-1)*itemPadding) / itemCount;
                } else if(ITEM_PADDING==intervalMode){
                    measureHeight+=itemDrawableSize;
                }
            }
        }
        setMeasuredDimension(measureWidth,measureHeight);
    }

    @Override
    protected void onFocusChanged(boolean gainFocus, int direction, Rect previouslyFocusedRect) {
        super.onFocusChanged(gainFocus, direction, previouslyFocusedRect);
        InputMethodManager imm = (InputMethodManager) getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
        if (gainFocus&&isFocusableInTouchMode()) {
            imm.showSoftInput(this, InputMethodManager.SHOW_FORCED);
        } else {
            imm.hideSoftInputFromWindow(getWindowToken(), 0);
        }
    }

    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        super.onWindowFocusChanged(hasWindowFocus);
        InputMethodManager imm = (InputMethodManager) getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
        if (hasWindowFocus&&isFocusableInTouchMode()) {
            imm.showSoftInput(this, InputMethodManager.SHOW_FORCED);
        } else {
            imm.hideSoftInputFromWindow(getWindowToken(), 0);
        }
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        super.onTouchEvent(event);
        if (event.getAction() == MotionEvent.ACTION_UP&&hasFocus()) {
            setFocusableInTouchMode(true);
            InputMethodManager input = (InputMethodManager) getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
            input.showSoftInput(PrivacyLockView.this,InputMethodManager.SHOW_FORCED);
        }
        return true;
    }

    @Override
    public InputConnection onCreateInputConnection(EditorInfo outAttrs) {
        InputConnection inputConnection=super.onCreateInputConnection(outAttrs);
        if (onCheckIsTextEditor() && isEnabled()) {
            outAttrs.inputType = InputType.TYPE_CLASS_NUMBER|InputType.TYPE_NUMBER_FLAG_SIGNED;
            outAttrs.imeOptions = EditorInfo.IME_ACTION_DONE;
            inputConnection=new MyBaseInputConnection(this,true);
        }
        return inputConnection;
    }

    @Override
    public boolean onKeyUp(int keyCode, KeyEvent event) {
        if(KeyEvent.KEYCODE_DEL==keyCode){
            //delete
            deleteLastEditText();
            return true;
        } else if(KeyEvent.KEYCODE_ENTER==keyCode){
            //submit
            hideSortInput();
            return true;
        } else {
            return super.onKeyUp(keyCode, event);
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        drawItemDrawable(canvas);
        drawPrivacyDrawable(canvas);
    }

    private void drawItemDrawable(Canvas canvas) {
        if(null!=itemDrawable){
            int width = getWidth();
            int paddingTop = getPaddingTop();
            int itemSize=0;
            if(0==itemDrawableSize||AUTO==intervalMode){
                //取动态计算
                itemSize = (width - getPaddingLeft() - getPaddingRight()-(itemCount-1)*itemPadding) / itemCount;
            } else if(ITEM_PADDING==intervalMode){
                //取设定值
                itemSize =itemDrawableSize;
            }
            int offsetLeft=getPaddingLeft();
            for(int i=0;i<itemCount;i++){
                itemDrawable.setBounds(offsetLeft,paddingTop,offsetLeft+itemSize,paddingTop+itemSize);
                offsetLeft+=(itemSize+itemPadding);
                itemDrawable.draw(canvas);
            }
        }
    }

    private void drawPrivacyDrawable(Canvas canvas) {
        if(null!=privacyDrawable&&0!=privacyDrawableWidth&&0!=privacyDrawableHeight){
            int width = getWidth();
            int paddingTop = getPaddingTop();
            int itemSize=0;
            if(0==itemDrawableSize||AUTO==intervalMode){
                //取动态计算
                itemSize = (width - getPaddingLeft() - getPaddingRight()-(itemCount-1)*itemPadding) / itemCount;
            } else if(ITEM_PADDING==intervalMode){
                //取设定值
                itemSize =itemDrawableSize;
            }
            int offsetLeft=getPaddingLeft();
            for(int i=0;i<editable.length();i++){
                privacyDrawable.setBounds(offsetLeft+itemSize/2-privacyDrawableWidth/2,
                        paddingTop+itemSize/2-privacyDrawableHeight/2,
                        offsetLeft+itemSize/2+privacyDrawableWidth/2,
                        paddingTop+itemSize/2+privacyDrawableHeight/2);
                offsetLeft+=(itemSize+itemPadding);
                privacyDrawable.draw(canvas);
            }
        }
    }

    @Override
    public boolean onCheckIsTextEditor() {
        return true;
    }

    @Override
    protected void onDetachedFromWindow() {
        hideSortInput();
        super.onDetachedFromWindow();
    }

    private void hideSortInput() {
        InputMethodManager imm = (InputMethodManager) getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(getWindowToken(), 0);
    }

    public class MyBaseInputConnection extends BaseInputConnection {


        public MyBaseInputConnection(View targetView, boolean fullEditor) {
            super(targetView, fullEditor);
        }

        @Override
        public boolean commitText(CharSequence text, int newCursorPosition) {
            appendEditText(text);
            return true;
        }

        @Override
        public boolean deleteSurroundingText(int beforeLength, int afterLength) {
            boolean result;
            if((Build.VERSION.SDK_INT >= 14) && (beforeLength == 1 && afterLength == 0)) {
                result= super.sendKeyEvent(new KeyEvent(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_DEL))
                        && super.sendKeyEvent(new KeyEvent(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_DEL));
            } else {
                result= super.deleteSurroundingText(beforeLength, afterLength);
            }
            return result;
        }
    }

    public void addOnTextChangedListener(OnTextChangedListener listener){
        if(null!=listener&&!this.listeners.contains(listener)){
            this.listeners.add(listener);
        }
    }

    public void removeOnTextChangedListener(OnTextChangedListener listener){
        if(null!=listener){
            this.listeners.remove(listener);
        }
    }
    public void setOnTextSubmitListener(OnTextSubmitListener listener){
        this.submitListener=listener;
    }

    public interface OnTextChangedListener{
        void onTextChanged(CharSequence editable,int lastLength,int length);
    }


    public interface OnTextSubmitListener{
        void onSubmit(CharSequence editable);
    }
}
