package com.cz.library.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.support.annotation.IntDef;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;

import com.cz.library.R;
import com.cz.library.callback.OnItemClickListener;
import com.cz.library.util.ViewCompat;

/**
 * Created by cz on 15/6/17.
 * 带指示器的线性布局
 * like
 * HORIZONTAL
 * ---------------------------
 * |  1  2  3  4  5  6  7  8  |
 * |  V1 V2 V3 V4 V5 V6 V7 V8 |
 * ---------------------------
 */
public class IndicateLayout<T> extends DivideLinearLayout {
    private static final String TAG = "IndicateView";
    public static final int START = 0;
    public static final int CENTER = 1;
    public static final int END = 2;

    public static final int FROM_START=0;
    public static final int FROM_END=1;

    private final LayoutInflater layoutInflater;
    private final TextPaint textPaint;
    private final Paint dividePaint;
    private final Paint selectDividePaint;
    private final Paint backgroundPaint;
    private float indicatePadding;//内边距
    private float indicateSize;//指示器大小
    private Drawable indicateDrawable;//指示器drawable
    private Drawable selectIndicateDrawable;//指示器drawable
    private OnTemplateAddedListener listener;
    private OnItemClickListener itemClickListener;
    private OnDrawIndicatorTextCallback drawIndicatorTextCallback;
    private float selectIndicatePadding;
    private int templateLayout;//模板控件的资源id
    private int selectStartIndex;//选定的起始位置
    private boolean showEdgeDevice;//显示边缘分隔线
    private float indicatorStartOffset;//指示器偏移
    private int gravity;
    private int from;

    @IntDef(value = {START, CENTER, END})
    private @interface IndicateGravity {
    }

    @IntDef(value = {FROM_START, FROM_END})
    private @interface IndicateFrom {
    }

    public IndicateLayout(Context context) {
        this(context, null);
    }

    public IndicateLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        setWillNotDraw(false);
        ViewCompat.setDefaultViewTranslation(this);
        layoutInflater=LayoutInflater.from(context);
        textPaint = new TextPaint(Paint.ANTI_ALIAS_FLAG);
        dividePaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        selectDividePaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        backgroundPaint=new Paint(Paint.ANTI_ALIAS_FLAG);

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.IndicateLayout,R.attr.indicatorView,R.style.IndicatorView);

        setShowEdgeDivide(a.getBoolean(R.styleable.IndicateLayout_iv_showEdgeDivide,false));
        setTemplateLayout(a.getResourceId(R.styleable.IndicateLayout_iv_templateLayout,NO_ID));
        setTextColor(a.getColor(R.styleable.IndicateLayout_iv_textColor, Color.WHITE));
        setTextSize(a.getDimensionPixelSize(R.styleable.IndicateLayout_iv_textSize, 0));
        setIndicatePadding(a.getDimension(R.styleable.IndicateLayout_iv_indicatePadding, 0));
        setIndicateDivideColor(a.getColor(R.styleable.IndicateLayout_iv_divideColor, Color.WHITE));
        setIndicateDivideSize(a.getDimension(R.styleable.IndicateLayout_iv_divideSize, 0));
        setIndicateBackground(a.getColor(R.styleable.IndicateLayout_iv_indicateBackground,Color.WHITE));
        setIndicateDrawable(a.getDrawable(R.styleable.IndicateLayout_iv_indicateDrawable));
        setIndicateSize(a.getDimension(R.styleable.IndicateLayout_iv_indicateSize, 0));
        setSelectIndicatePadding(a.getDimension(R.styleable.IndicateLayout_iv_selectIndicatePadding,0));
        setSelectIndicateDrawable(a.getDrawable(R.styleable.IndicateLayout_iv_selectIndicateDrawable));
        setSelectDivideColor(a.getColor(R.styleable.IndicateLayout_iv_selectDivideColor,Color.WHITE));
        setSelectDivideSize(a.getDimension(R.styleable.IndicateLayout_iv_selectDivideSize,0));
        setFromInner(a.getInt(R.styleable.IndicateLayout_iv_from,FROM_START));
        setIndicatorStartOffset(a.getDimension(R.styleable.IndicateLayout_iv_indicatorStartOffset,0));


        this.gravity = a.getInt(R.styleable.IndicateLayout_iv_indicateGravity, START);
        a.recycle();
    }

    public void setIndicatorStartOffset(float offset) {
        this.indicatorStartOffset=offset;
        invalidate();
    }

    public void setShowEdgeDivide(boolean show) {
        this.showEdgeDevice=show;
        invalidate();
    }

    public void setTemplateLayout(int layoutId) {
        this.templateLayout=layoutId;
    }

    public void setTextSize(float textSize) {
        this.textPaint.setTextSize(textSize);
        invalidate();
    }

    public void setTextColor(int color) {
        this.textPaint.setColor(color);
        invalidate();
    }

    public void setIndicatePadding(float padding) {
        this.indicatePadding = padding;
        invalidate();
    }

    public void setIndicateDivideSize(float size) {
        this.dividePaint.setStrokeWidth(size);
        invalidate();
    }

    public void setIndicateDivideColor(int color) {
        this.dividePaint.setColor(color);
        invalidate();
    }

    public void setIndicateBackground(int color) {
        backgroundPaint.setColor(color);
    }

    /**
     * 设置指示器颜色
     *
     * @param color
     */
    public void setIndicateDrawable(int color) {
        setIndicateDrawable(new ColorDrawable(color));
    }

    public void setIndicateDrawable(Drawable drawable) {
        if (null !=drawable){
            this.indicateDrawable = drawable;
            invalidate();
        }
    }

    public void setSelectIndicatePadding(float padding) {
        this.selectIndicatePadding=padding;
        invalidate();
    }

    /**
     * 设置指示器颜色
     *
     * @param color
     */
    public void setSelectIndicateDrawable(int color) {
        setSelectIndicateDrawable(new ColorDrawable(color));
    }

    public void setSelectIndicateDrawable(Drawable drawable) {
        if (null !=drawable){
            this.selectIndicateDrawable = drawable;
            invalidate();
        }
    }

    public void setIndicateSize(float size) {
        this.indicateSize = size;
        invalidate();
    }

    public void setSelectDivideColor(int color) {
        selectDividePaint.setColor(color);
        invalidate();
    }

    public void setSelectDivideSize(float divideSize) {
        selectDividePaint.setStrokeWidth(divideSize);
        invalidate();
    }

    public void setSelectStartIndex(int index){
        if(0<=index&&index<getChildCount()){
            this.selectStartIndex=index;
        }
    }

    public void addTemplateLayout(T item){
        if(NO_ID!=templateLayout){
            View newLayout = layoutInflater.inflate(templateLayout, this, false);
            if(null!=listener){
                listener.onTemplateAdded(newLayout,item);
            }
            addView(newLayout, ViewGroup.LayoutParams.MATCH_PARENT,ViewGroup.LayoutParams.WRAP_CONTENT);
        }
    }

    @Override
    public void addView(View child, int index, ViewGroup.LayoutParams params) {
        super.addView(child, index, params);
        if(null!=itemClickListener){
            child.setOnClickListener(new OnClickListener() {
                @Override
                public void onClick(View v) {
                    itemClickListener.onItemClick(v,indexOfChild(v));
                }
            });
        }
    }

    private void setFromInner(int from) {
        setFrom(from);
    }

    public void setFrom(@IndicateFrom int from) {
        this.from=from;
        invalidate();
    }

    /**
     * 设置指示器方向
     *
     * @param gravity
     */
    public void setIndicateGravity(@IndicateGravity int gravity) {
        this.gravity = gravity;
        invalidate();
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        //根据方向移动整体控件位置
        int childCount = getChildCount();
        int orientation = getOrientation();
        for (int i = 0; i < childCount; i++) {
            View childView = getChildAt(i);
            LinearLayout.LayoutParams layoutParams = (LayoutParams) childView.getLayoutParams();
            if (VERTICAL == orientation) {
                layoutParams.leftMargin = (int) indicateSize;
            } else if (HORIZONTAL == orientation) {
                layoutParams.topMargin = (int) indicateSize;
            }
        }
        requestLayout();
    }

    private float getStartX() {
        float startX = getViewFromStart();
        if (VERTICAL == getOrientation()) {
            startX = indicateSize / 2;
        }
        return startX;
    }

    private float getStartY() {
        float startY = getViewFromStart();
        if (HORIZONTAL == getOrientation()) {
            startY = indicateSize / 2;
        }
        return startY;
    }

    private float getViewFromStart(){
        float from=0;
        int childCount = getChildCount();
        if(0<childCount&&!showEdgeDevice){
            View childView = getChildAt(0);
            if(VERTICAL==getOrientation()){
                if(CENTER==gravity){
                    from=(childView.getBottom()+childView.getTop())/2;
                } else if(END==gravity){
                    from=childView.getBottom();
                }
            } else if(HORIZONTAL==getOrientation()){
                if(CENTER==gravity){
                    from=(childView.getRight()+childView.getLeft())/2;
                } else if(END==gravity){
                    from=childView.getRight();
                }
            }
        }
        return from;
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        //绘分隔线
        if(0<getChildCount()){
            float startX = getStartX();
            float startY = getStartY();
            drawDivide(canvas,startX,startY);
            drawIndicatorItem(canvas,startX,startY);
        }
    }

    /**
     * 绘制指示器
     * @param canvas
     */
    private void drawIndicatorItem(Canvas canvas,float startX,float startY) {

        for (int i = 0; i < getChildCount(); i++) {
            String text=null;
            if(null!=drawIndicatorTextCallback){
                text=drawIndicatorTextCallback.callText(i);
            }
            if(!TextUtils.isEmpty(text)){
                drawText(canvas,text,startX,startY,i);
            } else {
                drawIndicatorDrawable(canvas,startX,startY,i);
            }
        }
    }

    /**
     * 绘分隔线,背景
     *
     * @param canvas
     */
    private void drawDivide(Canvas canvas,float startX,float startY) {
        int width = getWidth();
        int height = getHeight();
        int orientation = getOrientation();
        float radius = (indicateSize - indicatePadding ) / 2;
        if (VERTICAL == orientation) {
            canvas.drawRect(0, 0, indicateSize, height, backgroundPaint);
            PointF startPoint = getCenterPoint(0,startX,startY,radius);
            if(0<selectStartIndex&&selectStartIndex<getChildCount()){
                PointF selectPoint = getCenterPoint(selectStartIndex,startX,startY,radius);
                canvas.drawLine(indicateSize/2, startPoint.y, indicateSize/2, selectPoint.y, selectDividePaint);
                startPoint=selectPoint;
            }
            PointF endPoint = getCenterPoint(getChildCount()-1,startX,startY,radius);
            canvas.drawLine(indicateSize/2, startPoint.y, indicateSize/2, endPoint.y, dividePaint);
        } else if (HORIZONTAL == orientation) {
            canvas.drawRect(0, 0, width, indicateSize, backgroundPaint);
            PointF startPoint = getCenterPoint(0,startX,startY,radius);
            if(0<selectStartIndex&&selectStartIndex<getChildCount()){
                PointF selectPoint = getCenterPoint(selectStartIndex,startX,startY,radius);
                canvas.drawLine(startPoint.x, indicateSize/2, selectPoint.x, indicateSize/2, selectDividePaint);
                startPoint=selectPoint;
            }
            PointF endPoint = getCenterPoint(getChildCount()-1,startX,startY,radius);
            canvas.drawLine(startPoint.x, indicateSize/2, endPoint.x, indicateSize/2, selectDividePaint);
        }
    }

    /**
     * 绘制文字
     *
     * @param canvas
     * @param index
     */
    private void drawText(Canvas canvas,String text,float startX,float startY,  int index) {
        //绘背景
        Rect textRect = new Rect();//文字绘制矩阵
        Rect outRect = new Rect();
        View childView = getChildAt(index);
        childView.getHitRect(outRect);
        int orientation = getOrientation();
        textPaint.getTextBounds(text, 0, text.length(), textRect);
        float textWidth = textPaint.measureText(text);
        Paint.FontMetrics fm = textPaint.getFontMetrics();
        float centerBaselineY = textRect.height() / 2 - fm.descent + (fm.descent - fm.ascent) / 2;
        float radius = (indicateSize  - textPaint.getStrokeWidth() / 2) / 2;
        float textLeft = 0, textTop = 0, circleX = 0, circleY = 0;
        if (VERTICAL == orientation) {
            switch (gravity) {
                case START:
                    textTop = outRect.top + radius + centerBaselineY / 2;
                    circleY = outRect.top + radius;
                    break;
                case CENTER:
                    textTop = outRect.centerY() + centerBaselineY / 2;
                    circleY = outRect.centerY();
                    break;
                case END:
                    textTop = outRect.bottom - radius + centerBaselineY / 2;
                    circleY = outRect.bottom - radius;
                    break;
            }
            textLeft = startX - textWidth / 2;
            textTop+=indicatorStartOffset;
            circleX = startX;
            circleY+=indicatorStartOffset;
        } else if (HORIZONTAL == orientation) {
            switch (gravity) {
                case START:
                    textLeft = outRect.left + radius - textWidth / 2;
                    circleX = outRect.left + radius;
                    break;
                case CENTER:
                    textLeft = outRect.centerX() - textWidth / 2;
                    circleX = outRect.centerX();
                    break;
                case END:
                    textLeft = outRect.right - radius - textWidth / 2;
                    circleX = outRect.right - radius;
                    break;
            }
            textLeft+=indicatorStartOffset;
            textTop = startY + textRect.height() / 2;
            circleX+=indicatorStartOffset;
            circleY = startY;
        }
        if(index<=selectStartIndex){
            drawDrawable(selectIndicateDrawable,canvas,circleX,circleY,radius-selectIndicatePadding);
        } else {
            drawDrawable(indicateDrawable,canvas,circleX,circleY,radius- indicatePadding);
        }
        canvas.drawText(text, textLeft, textTop, textPaint);
    }

    private void drawIndicatorDrawable(Canvas canvas,float startX,float startY, int index) {
        float radius = (indicateSize - indicatePadding ) / 2;
        PointF pointF=getCenterPoint(index,startX,startY,radius);
        if(index<=selectStartIndex){
            drawDrawable(selectIndicateDrawable,canvas,pointF.x,pointF.y,radius-selectIndicatePadding);
        } else {
            drawDrawable(indicateDrawable,canvas,pointF.x,pointF.y,radius- indicatePadding);
        }
    }

    private PointF getCenterPoint(int index,float startX,float startY,float radius){
        Rect outRect = new Rect();
        View childView = getChildAt(index);
        childView.getHitRect(outRect);
        float circleX = 0, circleY = 0;
        if (VERTICAL == getOrientation()) {
            switch (gravity) {
                case START:
                    circleY = outRect.top + radius;
                    break;
                case CENTER:
                    circleY = outRect.centerY();
                    break;
                case END:
                    circleY = outRect.bottom - radius;
                    break;
            }
            circleX = startX;
            circleY+=indicatorStartOffset;
        } else if (HORIZONTAL == getOrientation()) {
            switch (gravity) {
                case START:
                    circleX = outRect.left + radius;
                    break;
                case CENTER:
                    circleX = outRect.centerX();
                    break;
                case END:
                    circleX = outRect.right - radius;
                    break;
            }
            circleX+=indicatorStartOffset;
            circleY = startY;
        }
        return new PointF(circleX,circleY);
    }

    /**
     * 绘制指定drawable对象
     * @param drawable
     * @param canvas
     * @param circleX
     * @param circleY
     * @param radius
     */
    private void drawDrawable(Drawable drawable,Canvas canvas,float circleX,float circleY,float radius){
        if(null!=drawable){
            drawable.setBounds((int) (circleX - radius),
                    (int) (circleY - radius),
                    (int) (circleX + radius),
                    (int) (circleY + radius));
            drawable.draw(canvas);
        }
    }


    public void setOnItemClickListener(OnItemClickListener listener){
        this.itemClickListener=listener;
    }

    public void setOnTemplateAddedListener(OnTemplateAddedListener listener){
        this.listener=listener;
    }

    public void setOnDrawIndicatorTextCallback(OnDrawIndicatorTextCallback callback){
        this.drawIndicatorTextCallback=callback;
    }

    public interface OnDrawIndicatorTextCallback{
        String callText(int index);
    }

    public interface OnTemplateAddedListener<T>{
        void onTemplateAdded(View view,T item);
    }

}
