package com.cz.library.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.StringRes;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.util.SparseArray;
import android.widget.EditText;

import com.cz.library.R;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by cz on 1/3/17.
 * 可预定义格式化的 editText
 * 1:设定类似正则的通用化格式:138----1381 3813 11位:3-{4}4 4
 */
public class FormatEditText extends EditText {
    private CharSequence originalText;
    private final FormatTextWatcher textWatcher;
    private final List<FormatItem> formatItems;
    private final SparseArray<FormatItem> afterPositionItems;
    private final SparseArray<FormatItem> beforePositionItems;

    public FormatEditText(Context context) {
        this(context,null,0);
    }

    public FormatEditText(Context context, AttributeSet attrs) {
        super(context, attrs);
        formatItems=new ArrayList<>();
        textWatcher=new FormatTextWatcher();
        afterPositionItems =new SparseArray<>();
        beforePositionItems=new SparseArray<>();
        addTextChangedListener(new FormatTextWatcher());
        initAttr(context, attrs);
    }

    public FormatEditText(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        formatItems=new ArrayList<>();
        textWatcher=new FormatTextWatcher();
        afterPositionItems =new SparseArray<>();
        beforePositionItems=new SparseArray<>();
        addTextChangedListener(new FormatTextWatcher());
        initAttr(context, attrs);
    }

    private void initAttr(Context context, AttributeSet attrs) {
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.FormatEditText);
        setFormat(a.getString(R.styleable.FormatEditText_ft_format));
        a.recycle();
    }

    public void setFormat(@StringRes int res){
        setFormat(getResources().getString(res));
    }

    public void setFormat(String value) {
        if(!TextUtils.isEmpty(value)){
            resetFormatItems(value);
            resetText(formatItems);
        }
    }

    /**
     * 格式化预设格式
     * @param value
     */
    private void resetFormatItems(String value) {
        formatItems.clear();
        afterPositionItems.clear();
        int index=0;
        int interval=0;
        int lastCharCount=0;
        Matcher matcher = Pattern.compile("(\\d+)((.)(\\{(\\d+)\\})?)?").matcher(value);
        while(matcher.find()){
            FormatItem formatItem =null;
            try{
                String charValue = matcher.group(3);
                if(!TextUtils.isEmpty(charValue)){
                    String charCountValue = matcher.group(5);
                    int charCount=TextUtils.isEmpty(charCountValue)?1:Integer.valueOf(charCountValue);
                    formatItem = new FormatItem(index+=Integer.valueOf(matcher.group(1)),index+interval,matcher.group(3),charCount);
                    interval+=charCount;
                }
            } catch (Exception e){
                e.printStackTrace();
            }
            if(null!=formatItem){
                formatItems.add(formatItem);
                afterPositionItems.put(index+interval,formatItem);
                beforePositionItems.put(index+lastCharCount,formatItem);
                lastCharCount=formatItem.charCount;
            }
        }
    }

    /**
     * 重设 text
     */
    private void resetText(List<FormatItem> formatItems) {
        if(null!=formatItems&&!formatItems.isEmpty()){
            int lastIndex=0;
            int length = originalText.length();
            StringBuilder out=new StringBuilder();
            for(FormatItem item:formatItems){
                if(item.index<length){
                    CharSequence text = originalText.subSequence(lastIndex, item.index);
                    if(TextUtils.isEmpty(item.charItem)){
                        out.append(text);
                    } else {
                        out.append(text+getFormatChar(item.charItem,item.charCount));
                    }
                    lastIndex=item.index;
                }
            }
            if(lastIndex<length){
                out.append(originalText.subSequence(lastIndex,length));
            }
            if(0<out.length()){
                super.setText(out.toString(),BufferType.EDITABLE);
            }
        }
    }

    private String getFormatChar(String charItem,int count){
        StringBuilder out=new StringBuilder();
        for(int i=0;i<count;out.append(charItem),i++);
        return out.toString();
    }

    public CharSequence getOriginalText(){
        return originalText;
    }
    @Override
    public void setText(CharSequence text, BufferType type) {
        this.originalText=text;
        super.setText(text, type);
        resetText(formatItems);
    }


    public class FormatItem{
        public final int index;
        public final int realIndex;
        public final String charItem;
        public final int charCount;

        public FormatItem(int index, int realIndex, String charItem, int charCount) {
            this.index = index;
            this.realIndex = realIndex;
            this.charItem = charItem;
            this.charCount = charCount;
        }
    }


    public class FormatTextWatcher implements TextWatcher{
        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {

        }

        @Override
        public void onTextChanged(CharSequence text, int start, int before, int count) {
            if (0!=start&&start < text.length()) {
                int findPosition = getFormatPosition(beforePositionItems,start);
                //新增
                if(0<=findPosition) {
                    FormatItem formatItem = beforePositionItems.valueAt(findPosition);
                    removeTextChangedListener(textWatcher);
                    StringBuilder out=new StringBuilder(text.subSequence(0, formatItem.realIndex));
                    out.append(getFormatChar(formatItem.charItem,formatItem.charCount));
                    out.append(text.subSequence(formatItem.realIndex,text.length()));
                    FormatEditText.super.setText(out.toString(), BufferType.EDITABLE);
                    FormatEditText.super.setSelection(out.length());
                    addTextChangedListener(textWatcher);
                    text=out.toString();
                }
            } else {
                int findPosition = getFormatPosition(afterPositionItems,start);
                if(0<=findPosition) {
                    FormatItem formatItem = afterPositionItems.valueAt(findPosition);
                    //减少
                    removeTextChangedListener(textWatcher);
                    text = text.subSequence(0, start - formatItem.charCount);
                    FormatEditText.super.setText(text, BufferType.EDITABLE);
                    FormatEditText.super.setSelection(text.length());
                    addTextChangedListener(textWatcher);
                }
            }
            //更新原始 text
            originalText=getOriginalText(text);
        }

        @Override
        public void afterTextChanged(Editable s) {
        }

        /**
         * 获取格式化的对象位置
         * @return
         */
        private int getFormatPosition(SparseArray<FormatItem> formatItems,int position) {
            int findPosition=-1;
            int size = formatItems.size();
            for(int i=0;i<size;i++){
                if(position== formatItems.keyAt(i)){
                    findPosition=i;
                    break;
                }
            }
            return findPosition;
        }

        /**
         * 获取格式化的对象位置
         * @return
         */
        private CharSequence getOriginalText(CharSequence text) {
            StringBuilder out=new StringBuilder(text);
            int size = formatItems.size();
            for(int i=0;i<size;i++){
                FormatItem formatItem = formatItems.get(i);
                if(formatItem.index<out.length()&&formatItem.index+formatItem.charCount<out.length()){
                    String formatValue = out.substring(formatItem.index, formatItem.index+formatItem.charCount);
                    if(isMatcher(formatValue,formatItem.charItem)){
                        out.delete(formatItem.index,formatItem.index+formatItem.charCount);
                    }
                }
            }
            return out;
        }

        private boolean isMatcher(CharSequence value,String charItem){
            boolean result=false;
            if(!TextUtils.isEmpty(value)){
                for(int i=0;i<value.length();i++){
                    if(result|new Character(value.charAt(i)).toString().equals(charItem)){
                        result=true;
                        break;
                    }
                }
            }
            return result;
        }
    }
}