package com.cz.library.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Parcelable;
import android.support.annotation.IntDef;
import android.support.annotation.StringRes;
import android.text.BoringLayout;
import android.text.Layout;
import android.text.StaticLayout;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.MotionEvent;

import com.cz.library.R;


/**
 * Created by czz on 2016/12/20.
 * 此控件实现功能,为TextView 附加另一边Text
 * @attr ref R.styleable#SuperTextView_st_extTextWidth      固定文字宽
 * @attr ref R.styleable#SuperTextView_st_extTextHeight     固定文字高,但未实现
 * @attr ref R.styleable#SuperTextView_st_extTextWeight     match状态下,ext的占比,isHorizontal:则为横向占比,isVertical时为纵向占比
 * @attr ref R.styleable#SuperTextView_st_extTextWeightSum  weightSum权重总数,默认为100
 * @attr ref R.styleable#SuperTextView_st_extText           附加文字
 * @attr ref R.styleable#SuperTextView_st_extTextColor      附加文字颜色
 * @attr ref R.styleable#SuperTextView_st_extTextSize       附加文字大小
 * @attr ref R.styleable#SuperTextView_st_extLines          附加文字限定行数,此属性暂未实现,因为Layout排版对象并未暴露,且暂时没太多时间
 * @attr ref R.styleable#SuperTextView_st_padding           附加文字内边距
 * @attr ref R.styleable#SuperTextView_st_paddingLeft       左边距
 * @attr ref R.styleable#SuperTextView_st_paddingTop        上边距
 * @attr ref R.styleable#SuperTextView_st_paddingRight      右边距
 * @attr ref R.styleable#SuperTextView_st_paddingBottom     下边距
 * @attr ref R.styleable#SuperTextView_st_extBackground     附加文字背景,会与ext_padding结合
 * @attr ref R.styleable#SuperTextView_st_extConstrains     附加文字约束:主要为上下左右,四种相对关系
 *      @menu align_left/align_top/align_right/align_bottom
 *
 * @attr ref R.styleable#SuperTextView_st_extTextStyle      文字样式
 *      @menu normal/bold/italic/bold_italic
 *
 * @attr ref R.styleable#SuperTextView_st_extGravity        文字在可活动空间方向,与TextView#gravity一致
 *      @flag left/top/right/bottom/center_horizontal/center_vertical/center
 *
 * 此控件用于做暂为复杂的TextView的文字附加排版功能,如  $24元,元字大小不一致  如线性布局,左与右分别放一段文字,以及上或者下.等等让人感觉用一个TextView或者嵌套实现起来非常别扭实现的解决方案,
 *
 * TODO 1:独立的背景2:独立的点击,3:背景的两种模式:wrap/match
 */
public class SuperTextView extends DivideTextView {
    private static final String TAG="SuperTextView";
    private static final boolean debug=false;
    public static final int ALIGN_LEFT=0x01;
    public static final int ALIGN_TOP=0x02;
    public static final int ALIGN_RIGHT=0x03;
    public static final int ALIGN_BOTTOM=0x04;

    public static final int LEFT=0x01;
    public static final int TOP=0x02;
    public static final int RIGHT=0x04;
    public static final int BOTTOM=0x08;
    public static final int CENTER_HORIZONTAL=0x10;
    public static final int CENTER_VERTICAL=0x20;
    public static final int CENTER=0x30;

    private final TextPaint textPaint;
    private final int[] originPadding;
    private final int[] extPadding;
    private int extGravity;
    private Layout extLayout;
    private int extLines;
    private int constrains;
    private String extText;
    private float extTextWidth;
    private float extTextHeight;
    private float fixedTextWidth;
    private float fixedTextHeight;
    private int extTextWeight;
    private int extTextWeightSum;
    private Drawable extBackgroundDrawable;
    private Drawable backgroundDrawable;

    @IntDef({ALIGN_TOP,ALIGN_RIGHT,ALIGN_BOTTOM})
    public @interface Constrains{
    }

    @IntDef({LEFT,TOP,RIGHT,BOTTOM,CENTER_HORIZONTAL,CENTER_VERTICAL,CENTER})
    public @interface Gravity{
    }


    public SuperTextView(Context context) {
        this(context,null,0);
    }

    public SuperTextView(Context context, AttributeSet attrs) {
        this(context, attrs,R.attr.superTextView);
    }

    public SuperTextView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        originPadding =new int[4];
        extPadding=new int[4];
        textPaint=new TextPaint(Paint.ANTI_ALIAS_FLAG);
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.SuperTextView, defStyleAttr, R.style.SuperTextView);
        setExtText(a.getString(R.styleable.SuperTextView_st_extText));
        setExtTextColor(a.getColor(R.styleable.SuperTextView_st_extTextColor,0));
        setExtTextSize(a.getDimensionPixelSize(R.styleable.SuperTextView_st_extTextSize,0));
        setExtLines(a.getInteger(R.styleable.SuperTextView_st_extLines,0));
        setExtConstrainsInner(a.getInt(R.styleable.SuperTextView_st_extConstrains,ALIGN_LEFT));
        setExtTextWidth(a.getDimension(R.styleable.SuperTextView_st_extTextWidth,0));
        setExtTextHeight(a.getDimension(R.styleable.SuperTextView_st_extTextHeight,0));
        setExtTextWeight(a.getInteger(R.styleable.SuperTextView_st_extTextWeight,0));
        setExtTextWeightSum(a.getInteger(R.styleable.SuperTextView_st_extTextWeightSum,0));
        setExtGravityInner(a.getInt(R.styleable.SuperTextView_st_extGravity,LEFT));
        setExtBackground(a.getDrawable(R.styleable.SuperTextView_st_extBackground));
        setExtTextStyle(a.getInt(R.styleable.SuperTextView_st_extTextStyle, Typeface.NORMAL));
        int extPadding= (int) a.getDimension(R.styleable.SuperTextView_st_padding,0);
        int extPaddingLeft= (int) a.getDimension(R.styleable.SuperTextView_st_paddingLeft,0);
        int extPaddingTop= (int) a.getDimension(R.styleable.SuperTextView_st_paddingTop,0);
        int extPaddingRight= (int) a.getDimension(R.styleable.SuperTextView_st_paddingRight,0);
        int extPaddingBottom= (int) a.getDimension(R.styleable.SuperTextView_st_paddingBottom,0);
        if(0!=extPadding){
            //取 padding
            setExtPadding(extPadding,extPadding,extPadding,extPadding);
        } else {
            //取各处设置
            setExtPadding(extPaddingLeft,extPaddingTop,extPaddingRight,extPaddingBottom);
        }
        a.recycle();

        a=context.obtainStyledAttributes(attrs,new int[]{android.R.attr.padding, android.R.attr.paddingLeft, android.R.attr.paddingTop, android.R.attr.paddingRight, android.R.attr.paddingBottom});
        int padding= (int) a.getDimension(0,0);
        int paddingLeft=(int)a.getDimension(1,0);
        int paddingTop=(int)a.getDimension(2,0);
        int paddingRight=(int)a.getDimension(3,0);
        int paddingBottom=(int)a.getDimension(4,0);
        if(0!=padding){
            //取 padding
            setPaddingInner(padding,padding,padding,padding);
        } else {
            //取各处设置
            setPaddingInner(paddingLeft,paddingTop,paddingRight,paddingBottom);
        }
        a.recycle();

    }

    /**
     * 设置文字样式
     * @param style
     */
    public void setExtTextStyle(int style) {
        this.textPaint.setTypeface(Typeface.defaultFromStyle(style));
        invalidate();
    }

    /**
     * 设置附加的文字活动块背景
     * @param drawable
     */
    private void setExtBackground(Drawable drawable) {
        this.extBackgroundDrawable=drawable;
        invalidate();
    }

    /**
     * 设置附加文字排版方向
     * @param gravity
     */
    public void setExtGravity(@Gravity int gravity) {
        setExtGravityInner(gravity);
    }

    /**
     * 非约束设置文字方向.
     * @param gravity
     */
    private void setExtGravityInner(int gravity) {
        this.extGravity=gravity;
        if(isShown()) requestLayout();
    }

    /**
     * 设置附加文字
     * @param text
     */
    public void setExtText(String text) {
        this.extText=text;
        if(isShown()) requestLayout();
    }

    /**
     * 设置附加文字
     * @param res
     */
    public void setExtText(@StringRes int res) {
        this.extText=getResources().getString(res);
        if(isShown()) requestLayout();
    }

    /**
     * 设置附加文字颜色
     * @param color
     */
    public void setExtTextColor(int color) {
        textPaint.setColor(color);
        invalidate();
    }

    /**
     * 设置附加文字大小
     * @param textSize
     */
    public void setExtTextSize(int textSize) {
        textPaint.setTextSize(textSize);
        if(isShown()) requestLayout();
    }

    /**
     * 设置附加文字排版行数,暂未实现
     * @param lines
     */
    public void setExtLines(int lines) {
        this.extLines=lines;
        if(isShown())  requestLayout();
    }

    /**
     * 设置文字排版约束
     * @param constrains
     */
    public void setExtConstrains(@Constrains int constrains) {
        setExtConstrainsInner(constrains);
    }

    public void setExtConstrainsInner(int constrains) {
        extTextWidth=extTextHeight=0;
        this.constrains=constrains;
        if(isShown()) requestLayout();
    }

    /**
     * 设置文字固定宽
     * @param textWidth
     */
    public void setExtTextWidth(float textWidth) {
        this.fixedTextWidth =textWidth;
        if(isShown()) requestLayout();
    }

    /**
     * 设置文字固定高,暂未实现
     * @param textHeight
     */
    public void setExtTextHeight(float textHeight) {
        this.fixedTextHeight =textHeight;
        if(isShown()) requestLayout();
    }

    /**
     * 当match状态下的空间占比权重
     * @param weight
     */
    public void setExtTextWeight(int weight) {
        this.extTextWeight=weight;
        if(isShown()) requestLayout();
    }

    /**
     * 设置文字权重和
     * @param sum
     */
    public void setExtTextWeightSum(int sum) {
        this.extTextWeightSum=sum;
        if(isShown()) requestLayout();
    }

    /**
     * 设置系统padding
     * @param left
     * @param top
     * @param right
     * @param bottom
     * @deprecated use {@link #setRealPadding} instead
     */
    @Override
    public final void setPadding(int left, int top, int right, int bottom) {
        super.setPadding(left, top, right, bottom);
    }

    /**
     * 设置系统 padding
     * @param left
     * @param top
     * @param right
     * @param bottom
     */
    public void setRealPadding(int left, int top, int right, int bottom){
        super.setPadding(left, top, right, bottom);
        setPaddingInner(left,top,right,bottom);
    }

    /**
     * 设置附加文字内padding
     * @param left
     * @param top
     * @param right
     * @param bottom
     */
    public void setExtPadding(int left,int top,int right,int bottom){
        extPadding[0]=left;
        extPadding[1]=top;
        extPadding[2]=right;
        extPadding[3]=bottom;
        if(isShown()) requestLayout();
    }

    private void setPaddingInner(int left, int top, int right, int bottom) {
        originPadding[0]=left;
        originPadding[1]=top;
        originPadding[2]=right;
        originPadding[3]=bottom;
    }

//    @Override
//    public void setBackgroundDrawable(Drawable background) {
//        this.backgroundDrawable=background;
//        invalidate();
//    }

    public String getExtText() {
        return extText;
    }

    public float getExtTextWidth() {
        return extTextWidth;
    }

    public float getExtTextHeight() {
        return extTextHeight;
    }

    public int getExtGravity() {
        return extGravity;
    }

    public int getExtTextWeight() {
        return extTextWeight;
    }

    public int getExtTextWeightSum() {
        return extTextWeightSum;
    }

    public int[] getExtPadding() {
        return extPadding;
    }

    public Layout getExtLayout() {
        return extLayout;
    }

    public Drawable getExtBackgroundDrawable() {
        return extBackgroundDrawable;
    }

    public boolean isTextEmpty(){
        return TextUtils.isEmpty(extText);
    }

    /**
     * 测量计算控件,主要流程:横向/纵向约束->MeasureSpecMode->Weight/FixedSize->获得文字宽
     * 1:横向纵向约束:决定控件是否如何排版,控制控件排版主要通过自维所的padding+textWidth+extPadding实现
     * 2:MeasureSpecMode:主要分为控件march_parent/wrap_content/in scrollView/ListView... 其中有一处兼容问题为:魅族手机处于UNSPECIFIED状态时,getSize为0,而小米手机则为控件高
     * 3:weight/FixedSize:当控件为wrap_content时,weight不生效,因为没有活动空间,fixedSize:width会生效.
     *
     * @param widthMeasureSpec
     * @param heightMeasureSpec
     */
    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int measureWidth = MeasureSpec.getSize(widthMeasureSpec);
        int measureHeight = MeasureSpec.getSize(heightMeasureSpec);
        int paddingLeft = originPadding[0];
        int paddingTop = originPadding[1];
        int paddingRight = originPadding[2];
        int paddingBottom = originPadding[3];
        //计算除文字外,宽高.的尺寸
        int wrapWidth= paddingLeft + paddingRight + getDrawableSize(0) + getDrawableSize(2);
        int wrapHeight= paddingTop + paddingBottom + getDrawableSize(1) + getDrawableSize(3);
        float textWidth=0,textHeight=0;
        //如果能占满全屏,就按权重算,如果不能,就按wrap_content 算
        float measureTextWidth =0;
        if(!TextUtils.isEmpty(extText)){
            measureTextWidth = textPaint.measureText(extText, 0, extText.length());
        }
        //1:动态根据extTextWidth/extTextWeight 设定附加文字与TextView文字本身文字占比,这里需要注意高度的设定问题
        if(isHorizontalConstrains()){
            //当控件处于match时,且当weight不为0时,按比较计算
            if((MeasureSpec.EXACTLY==widthMode||measureWidth<=measureTextWidth)&&0!=extTextWeight){
                textWidth=extTextWeight*1.0f/extTextWeightSum*measureWidth;
            } else if(0>= fixedTextWidth){
                //设定文字宽,此时一般为wrap_content,所以取最小
                textWidth=Math.min(measureWidth-wrapWidth,measureTextWidth);
            } else {
                //取固定宽
                textWidth= fixedTextWidth;
            }
        } else if(isVerticalConstrains()){
            if(MeasureSpec.EXACTLY==heightMode&&0!=extTextWeight){
                //高度按权重划分
                textHeight=extTextWeight*1.0f/extTextWeightSum*measureHeight;
            } else if(0< fixedTextHeight){
                textHeight= fixedTextHeight;
            }
            //当控件为纵向排版时,仍需要再次计算文字宽,
            if(MeasureSpec.AT_MOST==widthMode){
                //当wrap_content时,取最小
                textWidth=Math.min(measureWidth-wrapWidth,measureTextWidth);
            } else {
                //matcher时,取边距外最大.这里不与系统设定的padding关联,extPadding是自有ext空间内边距,系统padding仅为原生TextView的padding约束,两边无关联
                textWidth=measureWidth-extPadding[0]-extPadding[2];
            }
        }
        //计算文字宽/高
        Layout layout = getLayout();
        if(!TextUtils.isEmpty(extText)){
            //根据文字宽与设定文字计算排版
            int layoutTextWidth = Math.round(textWidth);
            if(1==extLines){
                extLayout=new BoringLayout(extText,textPaint,0<layoutTextWidth?layoutTextWidth:0, Layout.Alignment.ALIGN_NORMAL,1.0f,0,new BoringLayout.Metrics(),true);
            } else {
                extLayout=new StaticLayout(extText,textPaint,0<layoutTextWidth?layoutTextWidth:0, Layout.Alignment.ALIGN_NORMAL,1.0f,0,false);
            }
            this.extTextWidth=layoutTextWidth;
            this.extTextHeight=0!=textHeight?textHeight:getExtLayoutHeight();
        } else if(null!=extLayout){
            //当文字被加载过,又被置空,还原信息
            extLayout=null;
            super.setPadding(originPadding[0],originPadding[1],originPadding[2],originPadding[3]);
        }
        //以上,计算出文字排版,以及文字所占宽高,下面开始计算实际控件所占宽高
        if(MeasureSpec.AT_MOST==widthMode){
            //当wrap_content时,所占宽高
            if(null!=layout){
                wrapWidth +=layout.getWidth();
            }
             if(isHorizontalConstrains()){
                 //横向时,所占宽为系统文字,drawable,padding 以及extPadding,extText,所占宽总和,少于则取计算结果,超过,则放过取值,直接取measureWidth
                wrapWidth+=(getExtLayoutWidth()+extPadding[0]+extPadding[2]);
                if(wrapWidth<measureWidth){
                    measureWidth=wrapWidth;
                }
            } else if(isVerticalConstrains()&&wrapWidth<measureWidth&&(getExtLayoutWidth()+extPadding[0]+extPadding[2])<measureWidth){
                 //纵向排版时,则取宽度大的一方.但小于measureWidth
                measureWidth=Math.max(wrapWidth,getExtLayoutWidth()+extPadding[0]+extPadding[2]);
            }
        } else if(MeasureSpec.UNSPECIFIED==widthMode){
            //处于ListView/ScrollView时,取最大值
            if(isHorizontalConstrains()){
                measureWidth+=(layout.getWidth()+getExtLayoutWidth()+wrapWidth+extPadding[0]+extPadding[2]);
            } else {
                measureWidth=Math.max(layout.getWidth()+wrapWidth,getExtLayoutWidth()+extPadding[0]+extPadding[2]);
            }
        }
        if(MeasureSpec.EXACTLY==heightMode){
            //matcher时,取最大.
            measureHeight=MeasureSpec.getSize(heightMeasureSpec);
        } else if(0>= fixedTextHeight &&MeasureSpec.AT_MOST==heightMode){
            //1:首先获取原生 textView 计算高度
            if(null!=layout){
                wrapHeight +=layout.getHeight();
            }
            //2:判断如果为纵向排版,则加上绘制文字高,否则则判断绘制文与原排版高度,取最大
            if(isVerticalConstrains()){
                wrapHeight+=(getExtLayoutHeight()+extPadding[1]+extPadding[3]);
            } else if(wrapHeight<Math.min(measureHeight,getExtLayoutHeight()+extPadding[1]+extPadding[3])){
                wrapHeight=Math.min(measureHeight,getExtLayoutHeight()+extPadding[1]+extPadding[3]);
            }
            measureHeight=Math.min(wrapHeight,measureHeight);
        } else if(MeasureSpec.UNSPECIFIED==heightMode){
            //处于ListView/ScrollView时,取最大值
            if(isVerticalConstrains()){
                measureHeight=(wrapHeight+layout.getHeight()+getExtLayoutHeight()+extPadding[1]+extPadding[3]);
            } else {
                measureHeight=Math.max(layout.getHeight()+wrapHeight,getExtLayoutHeight()+extPadding[1]+extPadding[3]);
            }
        } else if(0< fixedTextHeight){
            measureHeight= (int) fixedTextHeight;
        }
        //重置系统padding,以系统padding约束原生text的空间,并给予空间绘制extPadding,此处会导致多次measure,不过可以接受
        if(null!=extLayout){
            if(isHorizontalConstrains()){
                resetLayoutPadding((int) textWidth+extPadding[0]+extPadding[2]);
            } else if(isVerticalConstrains()){
                if(MeasureSpec.AT_MOST==heightMeasureSpec){
                    resetLayoutPadding((int) Math.min(extTextHeight+extPadding[1]+extPadding[3],measureHeight));
                }else if(0!=textHeight){
                    resetLayoutPadding((int) (textHeight+extPadding[1]+extPadding[3]));
                } else {
                    resetLayoutPadding((int) (extTextHeight+extPadding[1]+extPadding[3]));
                }
            }
        }
        setMeasuredDimension(measureWidth,measureHeight);
    }

    public int getExtLayoutWidth(){
        return null==extLayout?0:extLayout.getWidth();
    }

    public int getExtLayoutHeight(){
        return null==extLayout?0:extLayout.getHeight();
    }

    private int getDrawableSize(int index){
        int drawableSize=0;
        int compoundDrawablePadding = getCompoundDrawablePadding();
        Drawable[] compoundDrawables = getCompoundDrawables();
        if(null!=compoundDrawables[index]){
            int intrinsicSize=0;
            if(0==index||2==index){
                //横向
                intrinsicSize=compoundDrawables[index].getIntrinsicWidth();
            } else if(1==index||3==index){
                intrinsicSize=compoundDrawables[index].getIntrinsicHeight();
            }
            drawableSize+=(intrinsicSize+compoundDrawablePadding);
        }
        return drawableSize;
    }

    /**
     * 重置系统padding
     * @param extTextSize
     */
    private void resetLayoutPadding(int extTextSize) {
        switch(constrains){
            case ALIGN_LEFT:
                if(originPadding[0]+extTextSize!=getPaddingLeft()){
                    super.setPadding(originPadding[0]+extTextSize, originPadding[1], originPadding[2], originPadding[3]);
                }
                break;
            case ALIGN_TOP:
                if(originPadding[1]+extTextSize!=getPaddingTop()){
                    super.setPadding(originPadding[0], originPadding[1]+extTextSize, originPadding[2], originPadding[3]);
                }
                break;
            case ALIGN_RIGHT:
                if(originPadding[2]+extTextSize!=getPaddingRight()){
                    super.setPadding(originPadding[0], originPadding[1], originPadding[2]+extTextSize, originPadding[3]);
                }
                break;
            case ALIGN_BOTTOM:
                if(originPadding[3]+extTextSize!=getPaddingBottom()){
                    super.setPadding(originPadding[0], originPadding[1], originPadding[2],originPadding[3]+extTextSize);
                }
                break;
        }
    }


    public boolean isHorizontalConstrains(){
        return constrains==ALIGN_LEFT||constrains==ALIGN_RIGHT;
    }

    public boolean isVerticalConstrains(){
        return constrains==ALIGN_TOP|| constrains==ALIGN_BOTTOM;
    }

    /**
     * 根据extGravity获取绘制文字的平移x,y坐标,因extGravity支持位运算,6种组合,10多种不同排版
     * @param x
     * @param y
     * @return
     */
    private PointF getTranslatePoint(float x, float y) {
        int width = getWidth();
        int height = getHeight();
        float lineWidth = null==extLayout?0:extLayout.getLineRight(0);
        int layoutHeight = getExtLayoutHeight();
        if(0!=(extGravity&RIGHT)){
            if(lineWidth<extTextWidth){
                x+=(extTextWidth-lineWidth);
            } else if(extTextWidth<width){
                x+=(width-extTextWidth);
            }
        } else if(0!=(extGravity&CENTER_HORIZONTAL)){
            if(lineWidth<extTextWidth){
                x+=(extTextWidth-lineWidth)/2;
            } else if(!isHorizontalConstrains()&&lineWidth<width){
                x+=(width-lineWidth)/2;
            }
        }
        if(0!=(extGravity&BOTTOM)){
            if(layoutHeight<extTextHeight){
                y+=(extTextHeight-layoutHeight);
            } else if(!isVerticalConstrains()&&extTextHeight<height){
                y+=(height-extTextHeight);
            }
        } else if(0!=(extGravity&BOTTOM)&&layoutHeight>extTextHeight){
            y+=(extTextHeight-layoutHeight);
        } else if(0!=(extGravity&CENTER_VERTICAL)){
            if(layoutHeight<extTextHeight){
                y+=(extTextHeight-layoutHeight)/2;
            } else if(!isVerticalConstrains()&&layoutHeight<height){
                y=(height-layoutHeight)/2;
            }
        }

        //当系统文字方向为bottom 时,且文字长度大过当前控件高时,动态平移
        //代码为：TextView onDraw 5691..end:canvas.translate(compoundPaddingLeft, extendedPaddingTop + voffsetText);
        //此处不做操作的结果是,所有绘制都会随着系统的文字宽往上平移.导致按照常规x:0 y:0坐标绘制的东西被控件宽,顶到-height以上,
        // 如控件因文字只占1个字,而高度变为10000时,而Gravity为bottom靠齐时.则此时的面板绘制为x:0的位置为-10000+height,很难解释清...
        Layout layout = getLayout();
        if(null!=layout){
            int textHeight = layout.getHeight();
            int fillTextHeight = getHeight() - getPaddingTop() - getPaddingBottom() - getDrawableSize(1) - getDrawableSize(3);
            int gravity = getGravity();
            if(0!=(gravity&android.view.Gravity.BOTTOM)&&textHeight>fillTextHeight){
                y+=(textHeight-fillTextHeight);
            }
        }
        return new PointF(x,y);
    }

    /**
     * 获取各种相对位置的平移点
     * @return
     */
    public PointF getAlignTranslatePointF(){
        PointF point=null;
        int width = getWidth();
        int height = getHeight();
        switch (constrains){
            case ALIGN_LEFT:
                point=getTranslatePoint(extPadding[0], extPadding[1]);
                break;
            case ALIGN_TOP:
                point=getTranslatePoint(extPadding[0], extPadding[1]);
                break;
            case ALIGN_RIGHT:
                point=getTranslatePoint(width -extTextWidth- extPadding[2], extPadding[1]);
                break;
            case ALIGN_BOTTOM:
                point=getTranslatePoint(extPadding[0], height -extTextHeight- extPadding[3]);
                break;
        }
        return point;
    }
    @Override
    public Drawable getBackground() {
        return backgroundDrawable;
    }


    @Override
    protected void drawableStateChanged() {
        super.drawableStateChanged();
        if (null!=backgroundDrawable) {
            backgroundDrawable.setState(getDrawableState());
        }
        if (null!=extBackgroundDrawable) {
            extBackgroundDrawable.setState(getDrawableState());
        }
    }


    @Override
    public void jumpDrawablesToCurrentState() {
        super.jumpDrawablesToCurrentState();
        if (null!=backgroundDrawable) {
            backgroundDrawable.jumpToCurrentState();
        }
        if (null!=extBackgroundDrawable) {
            extBackgroundDrawable.jumpToCurrentState();
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        PointF point = getAlignTranslatePointF();
        drawBackground(canvas);
        drawExtBackground(canvas,point);
        drawExtText(canvas,point);
        if(debug){
            int width = getWidth();
            int height = getHeight();

            Paint debugPaint=new Paint();
            debugPaint.setStrokeWidth(2);
            debugPaint.setColor(Color.RED);
            canvas.drawLine(0, height /2, width, height /2,debugPaint);
            canvas.drawLine(width /2,0, width /2, height,debugPaint);
        }
    }

    /**
     * TODO 等待实现
     * @param canvas
     */
    private void drawBackground(Canvas canvas) {
        if(null!=backgroundDrawable){
            int width = getWidth();
            int height = getHeight();
            int left=0,top=0,right=0,bottom=0;
            int textWidth=0,textHeight=0;
            Layout layout = getLayout();
            if(null!=layout){
                textWidth=layout.getWidth();
                textHeight=layout.getHeight();
            }
            switch (constrains){
                case ALIGN_LEFT:
                    right=width-getPaddingRight();
                    bottom=Math.min(height,extLayout.getHeight()+extPadding[1]+extPadding[3]);
                    break;
                case ALIGN_TOP:
                    right=Math.min(width,extLayout.getWidth()+extPadding[0]+extPadding[2]);
                    bottom=height-getPaddingBottom();
                    break;
                case ALIGN_RIGHT:
                    right=width-getPaddingLeft();
                    bottom=Math.min(height,extLayout.getHeight()+extPadding[1]+extPadding[3]);
                    break;
                case ALIGN_BOTTOM:
                    top=0;
                    right=Math.min(width,extLayout.getWidth()+extPadding[0]+extPadding[2]);
                    bottom=height;
                    break;
            }
            backgroundDrawable.setBounds(0,top,right,bottom);
            backgroundDrawable.draw(canvas);
        }
    }

    /**
     * 绘制附加文字背景
     * @param canvas
     * @param point 绘制文字平移位置
     */
    private void drawExtBackground(Canvas canvas,PointF point) {
        if(null!=extBackgroundDrawable){
            int width = getWidth();
            int height = getHeight();
            int left=0,top=0,right=0,bottom=0;
            switch (constrains){
                case ALIGN_LEFT:
                    if(null!=point) top= (int) point.y-extPadding[1];
                    right=getPaddingLeft()-originPadding[0];
                    bottom=Math.min(height,top+extLayout.getHeight()+extPadding[1]+extPadding[3]);
                    break;
                case ALIGN_TOP:
                    //此方式为背景wrap时使用
//                    if(null!=point) left= (int) point.x-extPadding[0];
//                    right=Math.min(width,left+extLayout.getWidth()+extPadding[0]+extPadding[2]);
                    right=Math.min(width,extLayout.getWidth()+extPadding[0]+extPadding[2]);
                    bottom=getPaddingTop()-originPadding[1];
                    break;
                case ALIGN_RIGHT:
                    if(null!=point) top= (int) point.y-extPadding[1];
                    left=width-getPaddingRight()+originPadding[2];
                    right=width;
                    bottom=Math.min(height,top+extLayout.getHeight()+extPadding[1]+extPadding[3]);
                    break;
                case ALIGN_BOTTOM:
                    top=height-getPaddingBottom()+originPadding[3];
                    right=Math.min(width,extLayout.getWidth()+extPadding[0]+extPadding[2]);
                    bottom=height;
                    break;
            }
            extBackgroundDrawable.setBounds(left,top,right,bottom);
            extBackgroundDrawable.draw(canvas);
        }
    }

    /**
     * 绘制附加文字
     * @param canvas
     * @param point 平移 x,y 坐标
     */
    private void drawExtText(Canvas canvas,PointF point) {
        if(null!=extLayout){
            canvas.save();
            if(null!=point){
                canvas.translate(point.x,point.y);
            }
            extLayout.draw(canvas);
            canvas.restore();
        }

    }


    @Override
    public boolean onTouchEvent(MotionEvent event) {
        return super.onTouchEvent(event);
    }

    @Override
    public Parcelable onSaveInstanceState() {
        Bundle bundle=new Bundle();
        Parcelable parcelable = super.onSaveInstanceState();
        bundle.putParcelable("super",parcelable);
        bundle.putString("text",extText);
        bundle.putFloat("textSize",textPaint.getTextSize());
        bundle.putInt("textColor",textPaint.getColor());
        bundle.putInt("paddingLeft",extPadding[0]);
        bundle.putInt("paddingTop",extPadding[1]);
        bundle.putInt("paddingRight",extPadding[2]);
        bundle.putInt("paddingBottom",extPadding[3]);
        bundle.putFloat("textWidth",extTextWidth);
        bundle.putFloat("textHeight",extTextHeight);
        bundle.putInt("textWeight",extTextWeight);
        bundle.putInt("weightSum",extTextWeightSum);
        bundle.putInt("gravity",extGravity);
        bundle.putInt("constrains",constrains);
        return bundle;
    }

    @Override
    public void onRestoreInstanceState(Parcelable state) {
        Bundle bundle= (Bundle) state;
        Parcelable parcelable = bundle.getParcelable("super");
        super.onRestoreInstanceState(parcelable);
        this.extText=bundle.getString("text");
        this.textPaint.setColor(bundle.getInt("textColor"));
        this.textPaint.setTextSize(bundle.getFloat("textSize"));
        this.extPadding[0]=bundle.getInt("paddingLeft");
        this.extPadding[1]=bundle.getInt("paddingTop");
        this.extPadding[2]=bundle.getInt("paddingRight");
        this.extPadding[3]=bundle.getInt("paddingBottom");
        this.extTextWidth=bundle.getFloat("textWidth");
        this.extTextHeight=bundle.getFloat("textHeight");
        this.extTextWeight=bundle.getInt("textWeight");
        this.extTextWeightSum=bundle.getInt("weightSum");
        this.extGravity=bundle.getInt("gravity");
        this.constrains=bundle.getInt("constrains");
    }
}
