package com.cz.library.widget;

import android.animation.ValueAnimator;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PathMeasure;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.SweepGradient;
import android.os.Build;
import android.support.annotation.ArrayRes;
import android.support.v4.util.Pair;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;

import com.cz.library.R;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by czz on 2016/9/26.
 */
public class OvalDialView extends View{
    private static final String TAG = "OvalDialView";
    private final boolean DEBUG=false;
    private final Paint ringPaint;
    private final Paint fillPaint;
    private final Paint levelPaint;
    private final TextPaint levelTextPaint;
    private final TextPaint levelValuePaint;
    private final TextPaint infoPaint;

    private final RectF ovalRect;
    private final Rect textRect;
    private final List<Pair<Path,Float>> pathItems;
    private CharSequence[] levelTextItems;
    private int[] ringColorArray;
    private int[] levelValueArray;
    private int levelMinValue;
    private int levelMaxValue;
    private int currentLevelValue;
    private float currentLevelDegrees;
    private int animDuration;
    private int itemCount;
    private int dialPadding;
    private float textPadding;
    private OnLevelValueChangeListener listener;
    private float levelStrokePadding;
    private float levelDivideSize;


    public OvalDialView(Context context) {
        this(context,null,0);
    }

    public OvalDialView(Context context, AttributeSet attrs) {
        this(context, attrs,0);
    }

    public OvalDialView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        ringPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        fillPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        ringPaint.setStyle(Paint.Style.STROKE);
        fillPaint.setStyle(Paint.Style.STROKE);

        pathItems=new ArrayList<>();
        levelPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        levelTextPaint=new TextPaint(Paint.ANTI_ALIAS_FLAG);
        levelValuePaint=new TextPaint(Paint.ANTI_ALIAS_FLAG);
        infoPaint =new TextPaint(Paint.ANTI_ALIAS_FLAG);
        textRect=new Rect();
        ovalRect=new RectF();



        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.OvalDialView,R.attr.ovalDialView,R.style.OvalDialView);
        setPaintColor(a.getColor(R.styleable.OvalDialView_ov_paintColor, Color.TRANSPARENT));
        setStrokeWidth(a.getDimension(R.styleable.OvalDialView_ov_strokeWidth, 0));

        setFillPaintColorArray(a.getResourceId(R.styleable.OvalDialView_ov_fillPaintColorArray,NO_ID));
        setFillStrokeWidth(a.getDimension(R.styleable.OvalDialView_ov_fillStrokeWidth, 0));

        setTextPadding(a.getDimension(R.styleable.OvalDialView_ov_textPadding,0));
        setCornerRound(a.getBoolean(R.styleable.OvalDialView_ov_cornerRound,true));
        setDialItemCount(a.getInteger(R.styleable.OvalDialView_ov_dialItemCount, 0));
        setLevelTextArray(a.getTextArray(R.styleable.OvalDialView_ov_dialLevelTextArray));

        setLevelDivideSize(a.getDimension(R.styleable.OvalDialView_ov_levelDivideSize,0));
        setLevelStrokePadding(a.getDimension(R.styleable.OvalDialView_ov_levelStrokePadding,0));
        int resId = a.getResourceId(R.styleable.OvalDialView_ov_levelValue, NO_ID);
        if(NO_ID!=resId){
            setLevelValue(getResources().getIntArray(resId));
        }

        setLevelStrokeWidth(a.getDimension(R.styleable.OvalDialView_ov_levelStrokeWidth,0));
        setLevelDivideColor(a.getColor(R.styleable.OvalDialView_ov_levelDivideColor,Color.TRANSPARENT));
        setDialPadding((int) a.getDimension(R.styleable.OvalDialView_ov_dialPadding, 0));
        setLevelTextSize(a.getDimensionPixelSize(R.styleable.OvalDialView_ov_levelTextSize,0));
        setLevelTextColor(a.getColor(R.styleable.OvalDialView_ov_levelTextColor,Color.TRANSPARENT));
        setLevelInfoTextSize(a.getDimensionPixelSize(R.styleable.OvalDialView_ov_levelInfoTextSize, 0));
        setLevelInfoTextColor(a.getColor(R.styleable.OvalDialView_ov_levelInfoTextColor,Color.TRANSPARENT));
        setLevelValueTextSize(a.getDimensionPixelSize(R.styleable.OvalDialView_ov_levelValueTextSize, 0));
        setLevelValueTextColor(a.getColor(R.styleable.OvalDialView_ov_levelValueTextColor,Color.TRANSPARENT));
        setLevelAnimDuration(a.getInteger(R.styleable.OvalDialView_ov_levelAnimDuration,300));
        a.recycle();
    }


    private void setTextPadding(float textPadding) {
        this.textPadding=textPadding;
        invalidate();
    }

    public void setStrokeWidth(float width) {
        this.ringPaint.setStrokeWidth(width);
        invalidate();
    }

    public void setPaintColor(int color) {
        this.ringPaint.setColor(color);
        invalidate();
    }


    public void setFillPaintColorArray(@ArrayRes int res) {
        if(NO_ID!=res){
            setFillPaintColorArray(getResources().getIntArray(res));
        }
    }


    public void setFillPaintColorArray(int[] colorArray){
        this.ringColorArray=colorArray;
        invalidate();
    }
    public void setFillStrokeWidth(float strokeWidth) {
        this.fillPaint.setStrokeWidth(strokeWidth);
        invalidate();
    }

    public void setDialItemCount(int count) {
        this.itemCount=count;
        invalidate();
    }

    public void setCornerRound(boolean round) {
        this.ringPaint.setStrokeCap(round?Paint.Cap.ROUND:Paint.Cap.BUTT);
        this.fillPaint.setStrokeCap(round?Paint.Cap.ROUND:Paint.Cap.BUTT);
    }

    public void setLevelDivideSize(float size) {
        this.levelDivideSize =size;
        invalidate();
    }

    public void setLevelStrokePadding(float strokePadding) {
        this.levelStrokePadding=strokePadding;
        invalidate();
    }


    public void setLevelTextSize(int textSize) {
        this.levelTextPaint.setTextSize(textSize);
        invalidate();
    }
    public void setLevelInfoTextSize(int textSize) {
        this.infoPaint.setTextSize(textSize);
        invalidate();
    }
    public void setLevelValueTextSize(int textSize) {
        this.levelValuePaint.setTextSize(textSize);
        invalidate();
}

    public void setLevelValueTextColor(int color) {
        this.levelValuePaint.setColor(color);
        invalidate();
    }

    public void setLevelTextColor(int color) {
        this.levelTextPaint.setColor(color);
        invalidate();
    }

    private void setLevelInfoTextColor(int color) {
        this.infoPaint.setColor(color);
        invalidate();
    }

    public void setDialPadding(int padding) {
        this.dialPadding=padding;
        invalidate();
    }


    public void setLevelTextArray(CharSequence[] items) {
        if(null==items||items.length<itemCount){
            throw new IllegalArgumentException("items's height must be the same as item count!");
        } else {
            this.levelTextItems=items;
            invalidate();
        }
    }

    public void setLevelAnimDuration(int duration) {
        this.animDuration=duration;
    }

    public void setLevelDivideColor(int color) {
        this.levelPaint.setColor(color);
        invalidate();
    }

    public void setLevelStrokeWidth(float strokeWidth) {
        this.levelPaint.setStrokeWidth(strokeWidth);
        invalidate();
    }


    public void setLevelValue(int[] array) {
        if(null==array||array.length<itemCount){
            throw new IllegalArgumentException("items's height must be the same as item count!");
        } else {
            this.levelValueArray=array;
            this.currentLevelValue=array[0];
            this.levelMinValue=array[0];
            this.levelMaxValue=array[array.length-1];
            invalidate();
        }
    }


    public int getMinLevelValue(){
        return levelMinValue;
    }

    public int getMaxLevelValue(){
        return levelMaxValue;
    }

    public int getCurrentLevelValue(){
        return this.currentLevelValue;
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        int width = getWidth();
        int height = getHeight();
        int centerX=width/2;
        int centerY=height/2;
        float radius=Math.min(width,height)/2-ringPaint.getStrokeWidth()-levelStrokePadding*2- levelDivideSize;
        ovalRect.set(centerX-radius,centerY-radius,centerX+radius,centerY+radius);

        //重置path
        pathItems.clear();
        float itemDegrees=360f/itemCount;
        for(int i=0;i<itemCount;i++){
            Path path = new Path();
            path.addArc(ovalRect, i * itemDegrees, itemDegrees);
            PathMeasure pathMeasure=new PathMeasure(path,false);
            pathItems.add(new Pair<>(path,pathMeasure.getLength()));
        }
        if(null!=ringColorArray){
            fillPaint.setShader(new SweepGradient(w/2 , h/2, ringColorArray, null));
        }
    }

    public void setLevelValue(int value){
        int itemPosition = getItemPosition(value)-1;
        if(itemPosition<levelValueArray.length-1){
            int itemDegrees = 360 / itemCount;
            int startValue = levelValueArray[itemPosition];
            int itemValue=levelValueArray[itemPosition+1]-startValue;
            float fraction=(value-startValue)*1.0f/itemValue;
            currentLevelDegrees=itemPosition*itemDegrees+fraction*itemDegrees;
            currentLevelValue=value;
            invalidate();
        }
    }

    public void setLevelValueTo(final int endValue){
        //current value->value
        if(levelMinValue<=endValue&&levelMaxValue>=endValue){
            final int startValue=currentLevelValue;
            ValueAnimator valueAnimator = ValueAnimator.ofInt(startValue, endValue);
            valueAnimator.setDuration(animDuration);
            valueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator valueAnimator) {
                    Integer value = Integer.valueOf(valueAnimator.getAnimatedValue().toString());
                    int itemPosition = getItemPosition(value)-1;
                    if(itemPosition<levelValueArray.length-1){
                        int itemDegrees = 360 / itemCount;
                        int startValue = levelValueArray[itemPosition];
                        int itemValue=levelValueArray[itemPosition+1]-startValue;
                        float fraction=(value-startValue)*1.0f/itemValue;
                        currentLevelDegrees=itemPosition*itemDegrees+fraction*itemDegrees;
                        currentLevelValue=value;
                        if(null!=listener){
                            listener.onLevelValueChanged(startValue, value, endValue,itemPosition,fraction);
                        }
                        invalidate();
                    }
                }
            });
            valueAnimator.start();
        }
    }



    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        long st = System.currentTimeMillis();
        //第一步,绘外层环
        int width = getWidth();
        int height = getHeight();
        int centerX=width/2;
        int centerY=height/2;
        float itemDegrees=360f/itemCount;
        float radius=Math.min(width,height)/2-dialPadding-ringPaint.getStrokeWidth()/2;


        drawRing(canvas, centerX, centerY, itemDegrees, radius);

        drawRingItems(canvas,itemDegrees,centerX,centerY);

        //draw level;
        drawLevelItems(canvas,centerX,centerY);
        //draw value
        drawLevelTextValue(canvas, width, height);
//
        if(DEBUG){
            canvas.drawLine(0,height/2,width,height/2,fillPaint);
            canvas.drawLine(width/2,0,width/2,height,fillPaint);
            Log.e(TAG,"time:"+(System.currentTimeMillis()-st));
        }

    }

    /**
     * 绘外环,及进度环
     * @param canvas
     * @param centerX
     * @param centerY
     * @param itemDegrees
     * @param radius
     */
    private void drawRing(Canvas canvas, int centerX, int centerY, float itemDegrees, float radius) {
        canvas.save();
        canvas.rotate((180-itemDegrees)/2,centerX,centerY);
        if(Build.VERSION.SDK_INT>=Build.VERSION_CODES.LOLLIPOP){
            canvas.drawArc(centerX-radius,centerY-radius,centerX+radius,centerY+radius,itemDegrees,360-itemDegrees,false,ringPaint);
        } else {
            canvas.drawArc(new RectF(centerX-radius,centerY-radius,centerX+radius,centerY+radius),itemDegrees,360-itemDegrees,false,ringPaint);
        }
        //draw fill ring
        int sweepAngle= (int) (360*(currentLevelDegrees*1.0f/360));
        if(Build.VERSION.SDK_INT>=Build.VERSION_CODES.LOLLIPOP){
            canvas.drawArc(centerX-radius,centerY-radius,centerX+radius,centerY+radius,itemDegrees,0==sweepAngle?1:sweepAngle,false,fillPaint);
        } else {
            canvas.drawArc(new RectF(centerX-radius,centerY-radius,centerX+radius,centerY+radius),itemDegrees,0==sweepAngle?1:sweepAngle,false,fillPaint);
        }

        canvas.restore();
    }

    private void drawLevelItems(Canvas canvas,int centerX,int centerY) {
        if(levelMinValue!=levelMaxValue&&levelMaxValue-levelMinValue>itemCount) {
            for(int i=0;i<itemCount;i++){
                String text = String.valueOf(levelValueArray[i]);
                float textWidth = levelValuePaint.measureText(text, 0, text.length());
                levelValuePaint.getTextBounds(text, 0, text.length(), textRect);
                Pair<Path, Float> item = pathItems.get(i);
                canvas.save();
                canvas.rotate(90,centerX,centerY);
                canvas.drawTextOnPath(text,item.first,(item.second-textWidth)/2,textRect.height(),levelValuePaint);
                canvas.restore();
            }
        }
    }


    private void drawRingItems(Canvas canvas,float itemDegrees,int centerX,int centerY) {
        float outPadding=dialPadding+ringPaint.getStrokeWidth()+levelStrokePadding;
        for(int i=0;i<itemCount;i++){
            canvas.save();
            canvas.rotate(itemDegrees*i,centerX,centerY);
            canvas.drawLine(centerX, outPadding, centerX, outPadding+ levelDivideSize, levelPaint);
            canvas.restore();
        }
    }

    private void drawLevelTextValue(Canvas canvas, int width, int height) {
        String value=String.valueOf(currentLevelValue);
        float textWidth = levelTextPaint.measureText(value, 0, value.length());
        float levelValueTextTop=-(levelTextPaint.descent() + levelTextPaint.ascent());
        float levelInfoTextTop=-(infoPaint.descent()+ infoPaint.ascent())+levelValueTextTop;

        canvas.save();
        canvas.translate(0,(height-levelInfoTextTop)/2);
        //先绘等级值
        canvas.drawText(value,(width-textWidth)/2, levelValueTextTop, levelTextPaint);
        if(null!=levelTextItems&&levelTextItems.length==itemCount){
            int itemPosition = getItemPosition(currentLevelValue)-1;
            CharSequence text = levelTextItems[itemPosition];
            if(!TextUtils.isEmpty(text)){
                textWidth = infoPaint.measureText(text, 0, text.length());
                //再绘信用级提示
                canvas.drawText(text.toString(),(width-textWidth)/2,-(infoPaint.descent()+ infoPaint.ascent())+levelValueTextTop+textPadding, infoPaint);
            }
        }
        canvas.restore();
    }



    /**
     * 使用二分查找法,根据firstVisiblePosition找到SelectPositions中的位置
     *
     * @return
     */
    private int getItemPosition(int currentValue){
        int[] positions = levelValueArray;
        int start = 0, end = positions.length - 1, result = -1;
        while (start <= end) {
            int middle = (start + end) / 2;
            if (currentValue == positions[middle]) {
                result = middle + 1;
                break;
            } else if (currentValue < positions[middle]) {
                end = middle - 1;
            } else {
                start = middle + 1;
            }
        }
        return -1==result?start:result;
    }

    public void setOnLevelValueChangeListener(OnLevelValueChangeListener listener){
        this.listener=listener;
    }
    public interface OnLevelValueChangeListener{
        void onLevelValueChanged(int startValue, int currentValue, int endValue, int level, float fraction);
    }


}
