package com.cz.library.widget;

import android.animation.LayoutTransition;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Rect;
import android.support.annotation.IntDef;
import android.util.AttributeSet;
import android.util.Log;
import android.util.SparseArray;
import android.view.View;
import android.view.ViewGroup;

import com.cz.library.R;
import com.cz.library.callback.OnItemClickListener;
import com.cz.library.util.ViewCompat;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by cz on 1/6/17.
 * 流水式布局,类似于swing FlowLayout,作用为为某些搜索历史/商品分类作自动的流水式排序
 * 主要功能为:
 * 1:增加智能的MinItemPadding与HorizontalPadding之间的动态padding处理
 *  如最小约束为8px,而设定horizontalPadding为20,当条目添加时检测到小于horizontal却大于minPadding时,自动适应.此模式更符合流水式布局的特性,避免因空余40,却因条目为41时的舍弃动作
 * 2:增加LayoutGravity:分别为:left/center/right排版模式,适应一些其他需求
 * 3:增加LayoutTransition,处理计算时条目Padding的多次计算更改时的平滑性.
 *
 * @see {@link com.cz.library.widget.FlowLayout#debug} 调试模式
 */
public class FlowLayout extends ViewGroup {
    private static final String TAG="FlowLayout";
    public static final int LEFT=0;
    public static final int CENTER=1;
    public static final int RIGHT=2;

    private int horizontalPadding;
    private int minHorizontalPadding;
    private int verticalPadding;
    private int layoutGravity;
    private boolean debug;

    private final SparseArray<FlowInfomation> flowInformations;
    private OnItemClickListener listener;

    @IntDef({LEFT,CENTER})
    public @interface LayoutGravity{
    }

    public FlowLayout(Context context) {
        this(context,null,0);
    }

    public FlowLayout(Context context, AttributeSet attrs) {
        this(context, attrs,0);
    }

    public FlowLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        if(debug) setWillNotDraw(false);
        ViewCompat.setDefaultViewTranslation(this);
        flowInformations=new SparseArray<>();
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.FlowLayout);
        setHorizontalPadding((int) a.getDimension(R.styleable.FlowLayout_fl_horizontalPadding,0));
        setMinHorizontalPadding((int) a.getDimension(R.styleable.FlowLayout_fl_minHorizontalPadding,0));
        setVerticalPadding((int) a.getDimension(R.styleable.FlowLayout_fl_verticalPadding,0));
        setLayoutGravityInner(a.getInteger(R.styleable.FlowLayout_fl_layoutGravity,LEFT));
        a.recycle();
    }


    /**
     * 设置横向边距
     * @param padding
     */
    public void setHorizontalPadding(int padding) {
        this.horizontalPadding=padding;
        requestLayout();
    }

    public void setMinHorizontalPadding(int padding) {
        this.minHorizontalPadding=padding;
        requestLayout();
    }

    /**
     * 设置纵向边距
     * @param padding
     */
    public void setVerticalPadding(int padding) {
        this.verticalPadding=padding;
        requestLayout();
    }

    public void setLayoutGravity(@LayoutGravity int gravity) {
        setLayoutGravityInner(gravity);
    }

    /**
     * 设置布局排版模式
     * @param gravity
     */
    public void setLayoutGravityInner(int gravity) {
        this.layoutGravity=gravity;
        requestLayout();
    }

    public void setDebug(boolean debug){
        this.debug=debug;
        invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        measureChildren(widthMeasureSpec,heightMeasureSpec);
        int measureWidth = MeasureSpec.getSize(widthMeasureSpec);
        int measureHeight = MeasureSpec.getSize(heightMeasureSpec);
        int heightMode=MeasureSpec.getMode(heightMeasureSpec);
        int paddingLeft = getPaddingLeft();
        int paddingRight = getPaddingRight();
        int childCount = getChildCount();

        int rowCount=0;
        int columnCount=0;
        int offsetLeft=0;
        int wrapHeight=getPaddingTop();
        flowInformations.clear();
        List<FlowInfomation> currentRowItems=new ArrayList<>();
        for(int i=0;i<childCount;i++){
            FlowInfomation flowInfomation =new FlowInfomation();
            View childView = getChildAt(i);
            int childWidth = childView.getMeasuredWidth();
            int childHeight = childView.getMeasuredHeight();

            //探测最大/最小边距下,是否可以放下控件
            if((paddingLeft+offsetLeft+childWidth+paddingRight+horizontalPadding*(columnCount+1))<=measureWidth||
                    (paddingLeft+offsetLeft+childWidth+paddingRight+minHorizontalPadding*(columnCount+1))<=measureWidth){
                flowInfomation.column=columnCount++;
                offsetLeft+=childWidth;
                currentRowItems.add(flowInfomation);
                //检测是否超出,如果超出,则动态更新当前列,每个条目padding
                if(paddingLeft+offsetLeft+paddingRight+horizontalPadding*columnCount>measureWidth){
                    flowInfomation.padding=Math.min((measureWidth-offsetLeft-paddingLeft-paddingRight)/columnCount,horizontalPadding);
                    //超出动态控件边距
                    for(FlowInfomation item:currentRowItems){
                        item.padding=flowInfomation.padding;
                    }
                }
            } else {
                //越界,分行处理,最后一行,不加 VerticalPadding
                rowCount++;
                //取自动测试的边距,与设定的横向边距两值中最小值
                flowInfomation.padding=Math.min((measureWidth-offsetLeft-paddingLeft-paddingRight)/columnCount,horizontalPadding);
                for(FlowInfomation item:currentRowItems){
                    item.offset=offsetLeft;
                    item.columnCount=columnCount;
                    item.padding=flowInfomation.padding;
                }
                currentRowItems.clear();
                currentRowItems.add(flowInfomation);
                wrapHeight+=(childHeight+verticalPadding);
                offsetLeft=childWidth;
                columnCount=1;
            }
            flowInfomation.rect.left=paddingLeft+offsetLeft-childWidth;
            flowInfomation.rect.top=wrapHeight;
            flowInfomation.rect.right=paddingLeft+offsetLeft;
            flowInfomation.rect.bottom=wrapHeight+childHeight;

            flowInfomation.row=rowCount;
            flowInformations.put(i, flowInfomation);

            //最后一列
            if(offsetLeft<=measureWidth&&i==childCount-1){
                wrapHeight+=childHeight;
                //记录未满一行/最后行所有条目使用总长度
                for(FlowInfomation item:currentRowItems){
                    item.columnCount=columnCount;
                    item.offset=offsetLeft;
                }
            }
        }
        //debug log
        for(int i=0;debug&&i<flowInformations.size();i++){
            Log.e(TAG,"index:"+i+" "+flowInformations.valueAt(i).offset+" "+flowInformations.valueAt(i).columnCount);
        }
        if(MeasureSpec.EXACTLY==heightMode){
            setMeasuredDimension(measureWidth,measureHeight);
        } else {
            setMeasuredDimension(measureWidth,wrapHeight+getPaddingBottom());
        }
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        int childCount = getChildCount();
        int columnWidth = getWidth()-getPaddingLeft()-getPaddingRight();
        for(int i=0;i<childCount;i++){
            View childView = getChildAt(i);
            FlowInfomation item = flowInformations.get(i);
            int itemPadding;
            if(LEFT==layoutGravity){
                //添加但未赋item padding属性的默认以horizontalPadding排版
                itemPadding=-1==item.padding?horizontalPadding:item.padding;
                int itemLeft=item.rect.left+(itemPadding*item.column);
                int itemRight=item.rect.right+(itemPadding*item.column);
                childView.layout(itemLeft,item.rect.top,itemRight,item.rect.bottom);
            } else if(CENTER==layoutGravity){
                itemPadding=(columnWidth-item.offset)/item.columnCount;
                int itemLeft=item.rect.left+(itemPadding*item.column)+itemPadding/2;
                int itemRight=item.rect.right+(itemPadding*item.column)+itemPadding/2;
                childView.layout(itemLeft,item.rect.top,itemRight,item.rect.bottom);
            } else if(RIGHT==layoutGravity){
                //ltr mode
//                itemPadding=(columnWidth-item.offset)/item.columnCount;
//                int itemLeft=(r-item.rect.right)-(itemPadding*item.column);
//                int itemRight=(r-item.rect.left)-(itemPadding*item.column);
//                childView.layout(itemLeft,item.rect.top,itemRight,item.rect.bottom);
                itemPadding=-1==item.padding?horizontalPadding:item.padding;
                //此处排序为向最右靠齐,offsetLeft为靠左边距,所有最后一个无边距,从0开始,每个为 item.columnCount-item.column-1 层边距
                int offsetLeft=(item.column==item.columnCount-1?0:item.columnCount-item.column-1)*itemPadding;
                int itemLeft=(columnWidth-item.offset)+item.rect.left-offsetLeft;
                int itemRight=(columnWidth-item.offset)+item.rect.right-offsetLeft;
                childView.layout(itemLeft,item.rect.top,itemRight,item.rect.bottom);
            }
        }
    }

    @Override
    public void addView(View child, int index, LayoutParams params) {
        super.addView(child, index, params);
        child.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (null != listener) {
                    listener.onItemClick(v, indexOfChild(v));
                }
            }
        });
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        if(debug){
            int width = getWidth();
            int height = getHeight();
            Paint paint=new Paint();
            paint.setColor(Color.RED);
            paint.setStyle(Paint.Style.STROKE);
            paint.setStrokeWidth(4);
            canvas.drawLine(0,height/2,width,height/2,paint);
            canvas.drawLine(width/2,0,width/2,height,paint);

            canvas.drawRect(getPaddingLeft(),getPaddingTop(),width-getPaddingRight(),height-getPaddingBottom(),paint);
        }
    }

    class FlowInfomation {
        public int row;
        public int column;
        public int padding;
        public int columnCount;
        public int offset;
        public final Rect rect;

        public FlowInfomation() {
            padding=-1;
            rect=new Rect();
        }
    }

    public void setOnItemClickListener(OnItemClickListener listener){
        this.listener=listener;
    }
}
