package com.cz.library.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Region;
import android.os.Build;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;

import com.cz.library.R;

/**
 * Created by Administrator on 2017/3/20.
 */

public class CutterLayout extends FrameLayout {
    public static final String TAG="CutterLayout";
    private final int MAX_POINT=2;
    private final Paint paint;
    private final Paint outPaint;
    private float cutSize;
    private float dragEdgeSize;
    private boolean isBeginDragging;
    private final RectF clipRect;

    public CutterLayout(Context context) {
        this(context,null,0 );
    }

    public CutterLayout(Context context, AttributeSet attrs) {
        this(context, attrs,0);
    }

    public CutterLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        setWillNotDraw(false);
        this.clipRect=new RectF();
        this.paint=new Paint(Paint.ANTI_ALIAS_FLAG);
        this.paint.setStyle(Paint.Style.STROKE);
        this.outPaint=new Paint(Paint.ANTI_ALIAS_FLAG);
        //4.1开启硬件加速,解决clipRect失效问题
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.JELLY_BEAN) {
            setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        }
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.CutterLayout);
        setCutSize(a.getDimension(R.styleable.CutterLayout_cv_cutSize,0));
        setDragEdgeSize(a.getDimension(R.styleable.CutterLayout_cv_dragEdgeSize,0));
        setBorderWidth(a.getDimension(R.styleable.CutterLayout_cv_borderWidth,0));
        setBorderColor(a.getColor(R.styleable.CutterLayout_cv_borderColor, Color.WHITE));
        setOutRectColor(a.getColor(R.styleable.CutterLayout_cv_outRectColor, Color.WHITE));
        a.recycle();
    }

    public void setCutSize(float size) {
        this.cutSize=size;
        invalidate();
    }

    public void setDragEdgeSize(float edgeSize) {
        this.dragEdgeSize=edgeSize;
        invalidate();
    }


    public void setBorderWidth(float borderWidth) {
        paint.setStrokeWidth(borderWidth);
        invalidate();
    }

    public void setBorderColor(int color) {
        paint.setColor(color);
        invalidate();
    }

    private void setOutRectColor(int color) {
        outPaint.setColor(color);
        invalidate();
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        float left=(w-cutSize)/2;
        float paintHalfStrokeWidth = paint.getStrokeWidth()/2;
        clipRect.set(left,paintHalfStrokeWidth,left+cutSize,h-paintHalfStrokeWidth);
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        int width = getWidth();
        int height = getHeight();
        //截取此区域,实现中间区域不绘制
        //特别注意,此方法受硬件加速影响.需要测试4.1手机硬件加速支持情况
        canvas.clipRect(clipRect, Region.Op.DIFFERENCE);

        //绘制蒙层
        float halfStrokeWidth = outPaint.getStrokeWidth();
        canvas.drawRect(halfStrokeWidth,halfStrokeWidth,width-halfStrokeWidth,height-halfStrokeWidth,outPaint);

        //绘制显示层
        canvas.drawRect(clipRect,paint);
    }

    @Override
    public boolean onInterceptTouchEvent(MotionEvent ev) {
        float x = ev.getX();
        float y = ev.getY();
        int actionMasked = ev.getActionMasked();
        switch (actionMasked){
            case MotionEvent.ACTION_DOWN:
                //检测按下区域是否在可拖动区域
                RectF leftRect = new RectF(clipRect.left - dragEdgeSize, clipRect.top, clipRect.left + dragEdgeSize, clipRect.bottom);
                RectF rightRect = new RectF(clipRect.right - dragEdgeSize, clipRect.top, clipRect.right + dragEdgeSize, clipRect.bottom);
                isBeginDragging=leftRect.contains(x,y)||rightRect.contains(x,y);
                break;
            case MotionEvent.ACTION_CANCEL:
            case MotionEvent.ACTION_UP:
                isBeginDragging=false;
                break;
        }
        return isBeginDragging;
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        //此方法仅为isBeginDragging=true 时才会调用,否则会绘发事件至子控件
        int actionMasked = event.getActionMasked();
        if(MotionEvent.ACTION_MOVE==actionMasked&&isBeginDragging) {
            //多点处理,最大允许2点
            int pointerCount = Math.min(event.getPointerCount(),MAX_POINT);
            for (int i = 0; i < pointerCount; i++) {
                int pointerId = event.getPointerId(i);
                int pointerIndex = event.findPointerIndex(pointerId);
                float x = event.getX(pointerIndex);
                //靠左
                float width = clipRect.width();
                if (x < clipRect.centerX() && (width >= cutSize)) {
                    if(clipRect.right-x<cutSize){
                        //越界
                        clipRect.left=clipRect.right-cutSize;
                    } else {
                        clipRect.left = x;
                    }
                }
                //靠右
                if (x > clipRect.centerX() && (width >= cutSize)) {
                    if(x-clipRect.left<cutSize){
                        //越界
                        clipRect.right = clipRect.left+cutSize;
                    } else {
                        clipRect.right = x;
                    }
                }
                invalidate();
            }
        }
        return true;
    }



}
