package com.cz.dblibrary;

import android.app.Application;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Pair;

import com.cz.dblibrary.annotation.FieldFilter;
import com.cz.dblibrary.annotation.Table;
import com.cz.dblibrary.provider.DbProvider;
import com.cz.dblibrary.annotation.TableField;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by cz on 16/3/15.
 */
public class DbHelper {
    private static String packageName;
    private static final DbHelper helper;
    private final static Context appContext;
    private OnDbUpgradeListener upgradeListener;

    static {
        appContext = reflexContext();
        packageName = getPackageName();
        helper=new DbHelper();
    }

    private static Context reflexContext(){
        Context context=null;
        try{
            final Class<?> activityThreadClass = DbHelper.class.getClassLoader().loadClass("android.app.ActivityThread");
            Application application= (Application) activityThreadClass.getMethod("currentApplication").invoke(null);
            context=application.getApplicationContext();
        } catch (Exception e){
            e.printStackTrace();
        }
        return context;
    }
    public static String getPackageName() {
        if (packageName == null) {
            try {
                final Class<?> activityThreadClass = DbHelper.class.getClassLoader().loadClass("android.app.ActivityThread");
                final Method currentPackageName = activityThreadClass.getDeclaredMethod("currentPackageName");
                packageName = (String) currentPackageName.invoke(null);
            } catch (final Exception e) {
                e.printStackTrace();
            }
        }
        return packageName;
    }

    public static Context getContext(){
        return appContext;
    }

    public static String getDefaultDatabaseName(){
        return getPackageName()+"_db";
    }

    public static DbHelper get(){
        return helper;
    }

    private DbHelper(){
    }

    private boolean tableExist(String tableName){
        boolean result = false;
        if(!TextUtils.isEmpty(tableName)) {
            Cursor cursor=null;
            try {
                String sql = "select * from sqlite_master where name="+"'"+tableName+"'";
                cursor = rawQuery(sql, null);
                if (cursor.moveToNext()) {
                    result = cursor.getCount()!=0;
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(null!=cursor){
                    cursor.close();
                }
            }
        }
        return result;
    }

    private void ensureTable(Class<?> clazz){
        if(!tableExist(DbTable.getTable(clazz))){
            createTable(clazz);
        } else {
            DbProvider.ensureTable(clazz);
        }
    }

    /**
     * 创建表
     * @param clazz
     */
    private void createTable(Class<?> clazz) {
        Field[] fields = clazz.getDeclaredFields();
        List<Pair<String,Boolean>> primaryKeys=new ArrayList<>();
        HashMap<String,String> fieldItems=new HashMap<>();
        for (int i = 0; i < fields.length; i++) {
            Class<?> type = fields[i].getType();
            String fieldType;
            if (int.class == type || short.class == type || Integer.class == type || Short.class == type) {
                fieldType = " INTEGER";
            } else if (float.class == type || double.class == type || Float.class == type || Double.class == type) {
                fieldType = " FLOAT";
            } else if (boolean.class == type || Boolean.class == type) {
                fieldType = " BOOLEAN";
            } else if (long.class == type || Long.class == type) {
                fieldType = " LONG";
            } else {
                fieldType = " TEXT";
            }
            //过滤字段
            FieldFilter fieldFilter = fields[i].getAnnotation(FieldFilter.class);
            if(null==fieldFilter||!fieldFilter.value()){
                String fieldName;
                TableField tableField = fields[i].getAnnotation(TableField.class);
                if (null != tableField && !TextUtils.isEmpty(tableField.value())) {
                    fieldName = tableField.value();
                    if (tableField.primaryKey()) {
                        //主键
                        primaryKeys.add(new Pair<>(fieldName,tableField.autoIncrement()));
                    }
                } else {
                    fieldName = fields[i].getName();
                }
                fieldItems.put(fieldName,fieldType);
            }
        }
        String tableName = DbTable.getTable(clazz);
        String sql = "CREATE TABLE " +tableName + "(";
        if(primaryKeys.isEmpty()){
            //当一个字段主键都未设置时,检测Table注释中是否设置默认主键
            Table table = clazz.getAnnotation(Table.class);
            if(null!=table&&!TextUtils.isEmpty(table.primaryKey())){
                sql += (table.primaryKey()+" INTEGER PRIMARY KEY "+(table.autoIncrement()?"AUTOINCREMENT":"")+",");
            }
        }
        //一个主键时,设置单个主键
        int size = primaryKeys.size();
        if(1==size){
            Pair<String, Boolean> primaryPair = primaryKeys.get(0);
            sql += (primaryPair.first+" "+fieldItems.get(primaryPair.first)+" PRIMARY KEY "+(primaryPair.second?"AUTOINCREMENT":"")+",");
        }
        int index=0;
        for(Map.Entry<String,String> entry:fieldItems.entrySet()){
            sql+=(entry.getKey()+" "+entry.getValue()+" "+(index++!=fieldItems.size()-1?",":" "));
        }
        //多个主键时,设置联合主键
        if(1<size){
            sql += (", PRIMARY KEY(");
            for(int i=0;i<size;i++){
                Pair<String, Boolean> pair = primaryKeys.get(i);
                sql+=(pair.first+(i!=size-1?",":"))"));
            }
        } else {
           sql+=")";
        }
        DbProvider.execSQL(sql);
        DbProvider.addTable(clazz);
    }

    public void execSQL(String sql,String[] bindArgs){
        DbProvider.execSQL(sql,bindArgs);
    }

    public Cursor rawQuery(String sql,String[] selectionArgs){
        return DbProvider.rawQuery(sql,selectionArgs);
    }


    /**
     * 插入数据
     * @param item
     */
    public final Uri insertItem(Object item){
        Uri rtnUri=null;
        if(null!=item){
            ensureTable(item.getClass());
            Uri uri = DbTable.getUri(item.getClass());
            Context context = getContext();
            if (null != context && null != uri) {
                ContentResolver resolver = context.getContentResolver();
                rtnUri=resolver.insert(uri, DbTable.getContentValue(item));
            }
        }
        return rtnUri;
    }

    public final int bindInsert(final ArrayList items) {
        int code=-1;
        if (null != items &&!items.isEmpty()){
            Object obj = items.get(0);
            ensureTable(obj.getClass());
            Uri uri = DbTable.getUri(obj.getClass());
            ContentValues[] values = new ContentValues[items.size()];
            for (int i = 0; i < items.size(); i++) {
                values[i] = DbTable.getContentValue(items.get(i));
            }
            Context context = getContext();
            if (null != context && null != uri) {
                ContentResolver resolver = context.getContentResolver();
                code=resolver.bulkInsert(uri,values);
            }
        }
        return code;
    }

    public final int updateItem(Object item,String where,String... whereArgs){
        int code=-1;
        if(null!=item){
            ensureTable(item.getClass());
            Uri uri = DbTable.getUri(item.getClass());
            Context context = getContext();
            if (null != context && null != uri) {
                ContentResolver resolver = context.getContentResolver();
                code = resolver.update(uri, DbTable.getContentValue(item), where, whereArgs);
            }
        }
        return code;
    }

    public final int deleteItem(Object item){
        int code=-1;
        if(null!=item){
            Pair<String, String[]> where = getWhere(item);
            code=deleteItem(item,where.first,where.second);
        }
        return code;
    }

    public final int deleteItem(Object item,String where,String... whereArgs){
        int code=-1;
        if(null!=item){
            ensureTable(item.getClass());
            Uri uri = DbTable.getUri(item.getClass());
            Context context = getContext();
            if (null != context && null != uri) {
                ContentResolver resolver = context.getContentResolver();
                code = resolver.delete(uri, where, whereArgs);
            }
        }
        return code;
    }

    public final Pair<String,String[]> getWhere(Object item){
        Pair<String,String[]> wherePair=new Pair<>(null,null);
        if(null!=item){
            Field[] fields = item.getClass().getDeclaredFields();
            HashMap<String,String> fieldItems=new HashMap<>();
            for(int i=0;i<fields.length;i++){
                FieldFilter fieldFilter = fields[i].getAnnotation(FieldFilter.class);
                if(null==fieldFilter||!fieldFilter.value()){
                    fields[i].setAccessible(true);
                    try {
                        Object value = fields[i].get(item);
                        if(null!=value){
                            fieldItems.put(fields[i].getName(),value.toString());
                        }
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                }
            }
            int index=0;
            if(!fieldItems.isEmpty()){
                String where=new String();
                String[] whereArgs=new String[fieldItems.size()];
                for(Map.Entry<String,String> entry:fieldItems.entrySet()){
                    where+=(entry.getKey()+"=? "+(index!=fieldItems.size()-1?"and ":""));
                    whereArgs[index++]=entry.getValue();
                }
                wherePair=new Pair<>(where,whereArgs);
            }
        }
        return wherePair;
    }

    /**
     * 根据包名,以及类,查询所有数据
     *
     * @param clazz
     * @param <E>
     * @return
     */
    public final <E> E queryItem(Class<E> clazz,String where,String[] whereArgs,String order) {
        ensureTable(clazz);
        E item = null;
        Uri uri = DbTable.getUri(clazz);
        Context context = getContext();
        if (null != context && null != uri) {
            ContentResolver resolver = context.getContentResolver();
            //这里查询限制,本次/今天/本周
            Cursor cursor = null;
            try {
                String[] selection = DbTable.getSelection(clazz);
                cursor = resolver.query(uri, selection, where, whereArgs, order);
                if (null != cursor) {
                    item = getItemByCursor(clazz, cursor);
                }
            }catch(Exception e){
                e.printStackTrace();
            } finally {
                if(null!=cursor){
                    cursor.close();
                }
            }
        }
        return item;
    }

    public final <E> ArrayList<E> queryItems(Class<E> clazz) {
        return queryItems(clazz, null, null, null);
    }

    /**
     * 根据包名,以及类,查询所有数据
     *
     * @param clazz
     * @param <E>
     * @return
     */
    public final <E> ArrayList<E> queryItems(Class<E> clazz, String where, String[] whereArgs, String order) {
        ensureTable(clazz);
        ArrayList<E> items = new ArrayList<>();
        Uri uri = DbTable.getUri(clazz);
        Context context = getContext();
        if (null != context && null != uri) {
            ContentResolver resolver = context.getContentResolver();
            //这里查询限制,本次/今天/本周
            Cursor cursor = null;
            try {
                String[] selection = DbTable.getSelection(clazz);
                cursor = resolver.query(uri, selection, where, whereArgs, order);
                if (null != cursor) {
                    while (cursor.moveToNext()) {
                        items.add(getItemByCursor(clazz, cursor));
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                if (null != cursor) {
                    cursor.close();
                }
            }
        }
        return items;
    }

    private <E> E getItemByCursor(Class<E> clazz, Cursor cursor) throws InstantiationException, IllegalAccessException {
        E item = clazz.newInstance();
        Field[] fields = clazz.getDeclaredFields();
        for (int i = 0; i < fields.length; i++) {
            Field field = fields[i];
            field.setAccessible(true);
            String name;
            TableField tableField = field.getAnnotation(TableField.class);
            if (null != tableField && !TextUtils.isEmpty(tableField.value())) {
                name = tableField.value();
            } else {
                name = field.getName();
            }
            Class<?> type = field.getType();
            FieldFilter fieldFilter = fields[i].getAnnotation(FieldFilter.class);
            if (null == fieldFilter || !fieldFilter.value()) {
                int columnIndex = cursor.getColumnIndex(name);
                if (0 <= columnIndex) {
                    if (int.class == type || Integer.class == type) {
                        field.set(item, cursor.getInt(columnIndex));
                    } else if (short.class == type || Short.class == type) {
                        field.set(item, cursor.getShort(columnIndex));
                    } else if (float.class == type || Float.class == type) {
                        field.set(item, cursor.getFloat(columnIndex));
                    } else if (double.class == type || Double.class == type) {
                        field.set(item, cursor.getDouble(columnIndex));
                    } else if (boolean.class == type || Boolean.class == type) {
                        field.set(item, 1 == cursor.getInt(columnIndex));
                    } else if (long.class == type || Long.class == type) {
                        field.set(item, cursor.getLong(columnIndex));
                    } else {
                        field.set(item, cursor.getString(columnIndex));
                    }
                }
            }
        }
        return item;
    }

    /**
     * 清空指定表
     *
     * @param clazz
     */
    public void truncateTable(Class<?> clazz) {
        ensureTable(clazz);
        Uri uri = DbTable.getUri(clazz);
        Context context = getContext();
        ContentResolver contentResolver = context.getContentResolver();
        contentResolver.delete(uri, null, null);
    }

    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion){
        if(null!=upgradeListener){
            upgradeListener.onUpgrade(db,oldVersion,newVersion);
        }
    }

    public void setOnDbUpgradeListener(OnDbUpgradeListener listener){
        this.upgradeListener=listener;
    }


    public interface OnDbUpgradeListener{
        void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion);
    }

}
