package com.css.sdk.cservice.request;

import android.text.TextUtils;
import android.text.format.DateUtils;

import com.css.sdk.BuildConfig;
import com.css.sdk.cservice.base.http.HttpClient;
import com.css.sdk.cservice.base.http.HttpsClient;
import com.css.sdk.cservice.base.utils.LogUtils;
import com.css.sdk.cservice.base.utils.Utils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by t.wang on 2018/11/26.
 * <p>
 * Copyright © 2018 Adsforce. All rights reserved.
 */
public class HttpDnsManager {
    private static final Pattern IP_REG = Pattern.compile("(\\d*)\\.(\\d*)\\.(\\d*)\\.(\\d*)");
    private static final long DNS_IP_CACHE_MILLS = 10 * DateUtils.MINUTE_IN_MILLIS;

    private static String sDnsDomain;
    private static long sLastCallDnsMills;
    private static ArrayList<String> sDnsServerList = new ArrayList<>();
    private static HashMap<String, HashSet<String>> sDnsMap = new HashMap<>();

    public static HashMap<String, HashSet<String>> getDnsMap() {
        return sDnsMap;
    }

    public static String getUrlDomain() {
        return sDnsDomain;
    }

    public static void addDnsMappingServers(String domain, String dns) {
        sDnsDomain = domain;
        sDnsServerList = new ArrayList<>();
        sDnsServerList.add(dns);
    }

    public static void addDnsMappingServers(String domain, ArrayList<String> dnsServerList) {
        sDnsDomain = domain;
        sDnsServerList = Utils.getShuffleList(dnsServerList);
    }

    public static void callForClientDnsIp(final HttpCallback<String> callback) {
        if (sDnsMap.containsKey(sDnsDomain) && System.currentTimeMillis() - sLastCallDnsMills <= DNS_IP_CACHE_MILLS) {
            callback.onResponseSuccess("");
        } else {
            if (sDnsMap != null && sDnsMap.containsKey(sDnsDomain)) {
                sDnsMap.remove(sDnsDomain);
            }
            RunWrapper runnable = new RunWrapper(new Runnable() {
                @Override
                public void run() {
                    callForClientDnsIp(sDnsDomain, 0, callback);
                }
            });
            HttpThread.execute(runnable);
        }
    }

    private static void callForClientDnsIp(String domain, int currentIndex, final HttpCallback<String> callback) {
        boolean ipFound = false;
        String dnsUrl = sDnsServerList.get(currentIndex);
        LogUtils.i("HttpDnsManager callForClientDnsIp dnsUrl is " + dnsUrl);
        try {
            String body="";
            if (dnsUrl.startsWith("http://")){
                 body = HttpClient.builder().setUrl(dnsUrl).request().getBody();
            }else {
                 body = HttpsClient.builder().setmHostName(BuildConfig.CSERVICE_DOMAIN).setUrl(dnsUrl).request().getBody();
            }

            if (!TextUtils.isEmpty(body)) {
                JSONObject json = new JSONObject(body);
                if (json.has("answer")) {
                    JSONArray array = json.getJSONArray("answer");
                    if (array != null && array.length() > 0) {
                        for (int i = 0; i < array.length(); i++) {
                            JSONObject item = array.optJSONObject(i);
                            String type = item.optString("type");
                            if (!"A".equals(type)) {
                                continue;
                            }

                            String ip = item.optString("rdata");
                            Matcher matcher = IP_REG.matcher(ip);
                            if (matcher.matches()) {
                                addToDnsMap(domain, ip);
                                ipFound = true;
                            }
                        }
                    }
                }
            }

            if (!ipFound) {
                callNextDnsServer(domain, currentIndex, callback);
            } else {
                sLastCallDnsMills = System.currentTimeMillis();
                callback.onResponseSuccess("");
            }
        } catch (Throwable e) {
            LogUtils.w("HttpDnsManager callForClientDnsIp error: ", e);

            if (!ipFound) {
                callNextDnsServer(domain, currentIndex, callback);
            } else {
                sLastCallDnsMills = System.currentTimeMillis();
                callback.onResponseSuccess("");
            }
        }
    }

    private static void callNextDnsServer(String domain, int index, HttpCallback<String> callback) {
        index = index + 1;
        if (index < sDnsServerList.size()) {
            callForClientDnsIp(domain, index, callback);
        } else {
            callback.onResponedFail(new Exception("HttpDnsManager callForClientDnsIp failed "), -1);
        }
    }

    private static void addToDnsMap(String domain, String ip) {
        if (sDnsMap == null) {
            sDnsMap = new HashMap<>();
        }

        LogUtils.i("HttpDnsManager addToDnsMap " + domain + " mapping " + ip);

        HashSet<String> ipSet = new HashSet();
        if (sDnsMap.containsKey(domain)) {
            ipSet = sDnsMap.get(domain);
        } else {
            sDnsMap.put(domain, ipSet);
        }
        ipSet.add(ip);
    }

}
