package com.css.sdk.cservice.imageloader;

import android.content.Context;
import android.graphics.Bitmap;
import android.support.v4.util.LruCache;

import com.css.sdk.cservice.imageloader.core.LIFOLinkedBlockingDeque;
import com.css.sdk.cservice.imageloader.core.Type;
import com.css.sdk.cservice.imageloader.disk.LruDiskCache;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class ImageLoaderConfig {
    public int mErrorResId;
    public int mDefaultResId;
    public int mMaxMemCacheSize;
    public boolean mDiskCacheEnable;
    public LruCache<String, Bitmap> mLruCache;

    public LruDiskCache mDiskCache;
    public Executor taskExecutor;
    public Executor taskExecutorForCachedImages;

    private ImageLoaderConfig(Builder builder) {
        this.mDefaultResId = builder.mDefaultResId;
        this.mErrorResId = builder.mErrorResId;
        this.mMaxMemCacheSize = builder.mMaxMemCacheSize;
        this.mDiskCacheEnable = builder.mDiskCacheEnable;
        this.mLruCache = builder.mLruCache;
        this.mDiskCache = builder.diskCache;
        this.taskExecutor = builder.taskExecutor;
        this.taskExecutorForCachedImages = builder.taskExecutorForCachedImages;
    }

    public static ImageLoaderConfig createDefault(Context context) {
        return new Builder(context).build();
    }

    public void stop() {
        if (this.taskExecutorForCachedImages != null) {
            ((ExecutorService) taskExecutorForCachedImages).shutdownNow();
        }

        if (this.taskExecutor != null) {
            ((ExecutorService) taskExecutor).shutdownNow();
        }
    }

    public static class Builder {
        private Context context;
        /**
         * 内存缓存的大小，默认为app最大内存的1/8
         */
        private int mMaxMemCacheSize;
        /**
         * 发生错误时显示的图片
         */
        private int mErrorResId;
        /**
         * 默认显示的图片
         */
        private int mDefaultResId;
        /**
         * 是否开启硬盘缓存
         */
        private boolean mDiskCacheEnable = true;
        private Type mType = Type.FIFO;
        private static final int DEFAULT_THREAD_POOL_SIZE = 3;

        /**
         * 图片缓存的核心对象
         */
        private LruCache<String, Bitmap> mLruCache;

        private LruDiskCache diskCache;
        private Executor taskExecutor;
        private Executor taskExecutorForCachedImages;

        public Builder(Context context) {
            this.context = context.getApplicationContext();
        }

        public ImageLoaderConfig build() {
            initEmptyFieldsWithDefaultValues();
            return new ImageLoaderConfig(this);
        }

        public Builder memCacheSize(int memCacheSize) {
            this.mMaxMemCacheSize = memCacheSize;
            return this;
        }

        public Builder errorResId(int errorResId) {
            this.mErrorResId = errorResId;
            return this;
        }

        public Builder defaultResId(int defaultResId) {
            this.mDefaultResId = defaultResId;
            return this;
        }

        public Builder diskCacheEnbled(boolean cacheEnable) {
            this.mDiskCacheEnable = cacheEnable;
            return this;
        }

        public Builder loadType(Type type) {
            this.mType = type;
            return this;
        }

        private void initEmptyFieldsWithDefaultValues() {
            if (mLruCache == null) {
                mLruCache = createDefaultLruCache();
            }
            if (mDiskCacheEnable && diskCache == null) {
                try {
                    diskCache = new LruDiskCache(context);
                } catch (Exception e) {
                }
            }
            BlockingQueue<Runnable> taskQueue;

            if (mType == Type.FIFO) {
                taskQueue = new LinkedBlockingQueue<Runnable>();
            } else {
                taskQueue = new LIFOLinkedBlockingDeque<Runnable>();
            }

            taskExecutor = new ThreadPoolExecutor(DEFAULT_THREAD_POOL_SIZE, DEFAULT_THREAD_POOL_SIZE, 0L, TimeUnit.MILLISECONDS, taskQueue);

            taskExecutorForCachedImages = new ThreadPoolExecutor(DEFAULT_THREAD_POOL_SIZE, DEFAULT_THREAD_POOL_SIZE, 0L,
                    TimeUnit.MILLISECONDS, taskQueue);
        }

        private LruCache createDefaultLruCache() {
            int maxMemory = (int) Runtime.getRuntime().maxMemory() / 1024;
            int memCacheSize = maxMemory / 8;
            return new LruCache<String, Bitmap>(memCacheSize) {
                @Override
                protected int sizeOf(String key, Bitmap value) {
                    return value.getRowBytes() * value.getHeight() / 1024;
                }
            };
        }
    }
}
