package com.css.sdk.cservice.imageloader;

import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.widget.ImageView;

import com.css.sdk.cservice.imageloader.core.CancelableRequestDelegate;
import com.css.sdk.cservice.imageloader.core.ImageRequest;
import com.css.sdk.cservice.imageloader.core.LoadImageTask;
import com.css.sdk.cservice.userdata.GlobalData;

import java.io.File;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 图片加载类
 */
public class ImageLoader {
    public static final int MSG_LOAD_SUCCESS = 0x110;
    public static final int MSG_LOAD_FAIL = MSG_LOAD_SUCCESS + 1;

    private Executor mTaskDistribute = Executors.newCachedThreadPool();
    private final Map<String, ReentrantLock> uriLocks = new WeakHashMap<String, ReentrantLock>();

    private ImageLoaderConfig mLoaderConfig;

    public void init(ImageLoaderConfig loaderConfig) {
        this.mLoaderConfig = loaderConfig;
    }

    private CancelableRequestDelegate mCancelableRequestDelegate = new CancelableRequestDelegate();

    private Handler HANDLER = new Handler() {
        @Override
        public void handleMessage(Message msg) {
            ImageRequest req = (ImageRequest) msg.obj;
            switch (msg.what) {
                case MSG_LOAD_FAIL:
                    req.setErrorImageRes();
                    break;
                case MSG_LOAD_SUCCESS:
                    if (req.setResBitmap()) {
                        mLoaderConfig.mLruCache.put(req.getCacheKey(), req.getBitmap());
                    }
                    break;
            }
        }
    };

    private static ImageLoader sInstance;

    public static ImageLoader getInstance() {
        if (sInstance == null) {
            synchronized (ImageLoader.class) {
                if (sInstance == null) {
                    sInstance = new ImageLoader();
                }
            }
        }
        return sInstance;
    }

    public CancelableRequestDelegate getCancelableRequestDelegate() {
        return mCancelableRequestDelegate;
    }

    /**
     * 根据path为imageview设置图片
     *
     * @param path
     * @param imageView
     */
    public void load(final String path, final ImageView imageView) {
        if (TextUtils.isEmpty(path)) {
            return;
        }

        if (mLoaderConfig == null) {
            mLoaderConfig = ImageLoaderConfig.createDefault(GlobalData.getContext());
        }

        setDefaultImageRes(imageView);
        final ImageRequest req = buildImageRequest(path, imageView);
        final String cacheKey = req.getCacheKey();
        //记录最新的imageview -> cacheKey
        mCancelableRequestDelegate.putRequest(imageView.hashCode(), cacheKey);
        //如果内存中有则取内存中的缓存
        Bitmap bitmap = mLoaderConfig.mLruCache.get(cacheKey);
        if (bitmap != null) {
            imageView.setImageBitmap(bitmap);
            return;
        }

        mTaskDistribute.execute(new Runnable() {
            @Override
            public void run() {
                String uri = req.getUrl();
                LoadImageTask task = new LoadImageTask(GlobalData.getContext(), req, HANDLER, mLoaderConfig.mDiskCache);
                File cache = mLoaderConfig.mDiskCache.get(uri);
                if (cache != null && cache.exists()) {
                    mLoaderConfig.taskExecutorForCachedImages.execute(task);
                } else {
                    mLoaderConfig.taskExecutor.execute(task);
                }
            }
        });
    }

    public void stop() {
        if (mLoaderConfig != null) {
            mLoaderConfig.stop();
        }
        mCancelableRequestDelegate.clear();
        uriLocks.clear();
    }

    private ReentrantLock getLockForUri(String uri) {
        ReentrantLock lock = uriLocks.get(uri);
        if (lock == null) {
            lock = new ReentrantLock();
            uriLocks.put(uri, lock);
        }
        return lock;
    }

    private void setDefaultImageRes(ImageView imageView) {
        if (mLoaderConfig.mDefaultResId != 0) {
            imageView.setImageResource(mLoaderConfig.mDefaultResId);
        }
    }

    private ImageRequest buildImageRequest(String path, ImageView imageView) {
        ImageRequest req = new ImageRequest(this, path, imageView, mLoaderConfig, getLockForUri(path));
        return req;
    }


}
