package com.css.sdk.cservice.base.http;

import android.text.TextUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

/**
 * Created by echo on 16/6/6.
 */
public class HttpsClient {

    public final static String METHOD_GET = "GET";
    public final static String METHOD_HEAD = "HEAD";
    public final static String METHOD_POST = "POST";

    private URL url;
    private int connectTimeout = 10 * 1000;
    private int readTimeout = 15 * 1000;
    private boolean useCaches = false;
    private boolean redirected = false;
    private String requestMethod = METHOD_GET;
    private String requestBody = null;
    private String userAgent = null;
    private String enctyType = null;
    private Map<String, String> headerMap = new ConcurrentHashMap<>();
    private HttpsURLConnection conn;
    private String mHostName; //自定义host
    private String webUrl;

    public static HttpsClient builder() {
        return new HttpsClient();
    }

    private HttpsClient() {
    }

    public HttpsClient setUrl(String url) throws MalformedURLException {
        this.url = new URL(url);
        this.webUrl=url;
        return this;
    }

    public HttpsClient setConnectTimeout(int timeout) {
        connectTimeout = timeout;
        return this;
    }

    public HttpsClient setReadTimeout(int timeout) {
        readTimeout = timeout;
        return this;
    }

    public HttpsClient setUseCaches(boolean use) {
        useCaches = use;
        return this;
    }

    public HttpsClient setRequestMethod(String method) {
        requestMethod = method;
        return this;
    }

    public HttpsClient setRequestHeader(String key, String value) {
        if (!TextUtils.isEmpty(key) && value != null) {
            headerMap.put(key, value);
        }
        return this;
    }

    public HttpsClient setRequestBody(String body) {
        requestBody = body;
        return this;
    }

    public HttpsClient setUserAgent(String userAgent) {
        this.userAgent = userAgent;
        return this;
    }

    public HttpsClient setEnctyType(String enctyType) {
        this.enctyType = enctyType;
        return this;
    }

    public HttpResponse request() throws Exception {
        if (url == null)
            throw new IOException("URL is empty");

        makeConnection();

        return new HttpResponse(conn);
    }



    public HttpsClient setmHostName(String mHostName) {
        this.mHostName = mHostName;
        return this;
    }

    private void makeConnection() throws Exception {
        try {
            conn = (HttpsURLConnection) url.openConnection();
            conn.setConnectTimeout(connectTimeout);
            conn.setReadTimeout(readTimeout);
            conn.setUseCaches(useCaches);
            conn.setRequestMethod(requestMethod);
            conn.setInstanceFollowRedirects(true);

            if (userAgent != null) {
                conn.setRequestProperty("User-Agent", userAgent);
            }

            if (headerMap != null && !headerMap.isEmpty()) {
                for (Map.Entry<String,String> entry : headerMap.entrySet()) {
                    conn.setRequestProperty(entry.getKey(), headerMap.get(entry.getKey()));
                }
            }

            if (METHOD_GET.equals(requestMethod) || METHOD_HEAD.equals(requestMethod)) {
                conn.setDoOutput(false);
            } else if (METHOD_POST.equals(requestMethod)) {
                conn.setDoOutput(true);
            }

            if (mHostName != null) {
                conn.setRequestProperty("Host", mHostName);
            }
             //进行https的验证
            trustAllCertificate(conn);

            if (requestBody != null) {
                if (enctyType != null) {
                    conn.setRequestProperty("Content-type", enctyType);
                } else {
                    conn.setRequestProperty("Content-type", "application/json;charset=UTF-8");
                }

                OutputStream os = conn.getOutputStream();
                byte[] bytes = requestBody.getBytes("UTF-8");
                os.write(bytes);
                os.close();
            }

            if (conn.getResponseCode() >= 300 && conn.getResponseCode() < 400) {
                String location = conn.getHeaderField("Location");
                if (TextUtils.isEmpty(location)) {
                    location = conn.getHeaderField("location");
                }

                if (!redirected && !TextUtils.isEmpty(location)) {
                    redirected = true;
                    setUrl(location);
                    makeConnection();
                }
            }
        } catch (Throwable e) {
            throw new Exception(e);
        }
    }


    public void trustAllCertificate(HttpsURLConnection http) throws Exception {
        try {
            TrustManager[] tm = {new MyX509TrustManager()};
            SSLContext sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, tm, new java.security.SecureRandom());
            SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();
            http.setHostnameVerifier(new TrustAnyHostnameVerifier());
            http.setSSLSocketFactory(sslSocketFactory);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 不进行主机名确认
     */
    private static class TrustAnyHostnameVerifier implements HostnameVerifier {
        @Override
        public boolean verify(String hostname, SSLSession session) {
            return true;
        }
    }
    /**
     * 信任所有主机 对于任何证书都不做SSL检测
     * 安全验证机制，而Android采用的是X509验证
     */
    private static class MyX509TrustManager implements X509TrustManager {

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return null;
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
        }

    }

  //via  https://blog.csdn.net/qq_25920753/article/details/78527725
}
