package com.css.sdk.cservice.base.http;


import android.text.TextUtils;

import com.css.sdk.cservice.base.utils.LogUtils;
import com.css.sdk.cservice.request.URLConstant;
import com.css.sdk.cservice.userdata.GlobalData;
import com.css.sdk.cservice.utils.ImageUtils;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by echo on 16/6/6.
 */
public class HttpClient {

    public static String METHOD_GET = "GET";
    public static String METHOD_HEAD = "HEAD";
    public static String METHOD_POST = "POST";

    private final static String BOUNDARY = UUID.randomUUID().toString()
            .toLowerCase().replaceAll("-", "");// 边界标识
    private final static String PREFIX = "--";// 必须存在
    private final static String LINE_END = "\r\n";

    private final static String PIC_REQUESTKEY = "picFile";
    private final static String VIDEO_REQUESTKEY = "video_file";

    private URL url;
    private int connectTimeout = 2 * 60 * 1000;
    private int readTimeout = 2 * 60 * 1000;
    private boolean useCaches = false;
    private boolean redirected = false;
    private String requestMethod = METHOD_GET;
    private String requestBody = null;
    private String userAgent = null;
    private String enctyType = null;
    private Map<String, String> headerMap = new ConcurrentHashMap<>();
    private HttpURLConnection conn;
    private String mHostName; //自定义host

    public static HttpClient builder() {
        return new HttpClient();
    }

    private HttpClient() {
    }

    public HttpClient setUrl(String url) throws MalformedURLException {
        this.url = new URL(url);
        return this;
    }

    public HttpClient setConnectTimeout(int timeout) {
        connectTimeout = timeout;
        return this;
    }

    public HttpClient setReadTimeout(int timeout) {
        readTimeout = timeout;
        return this;
    }

    public HttpClient setUseCaches(boolean use) {
        useCaches = use;
        return this;
    }

    public HttpClient setRequestMethod(String method) {
        requestMethod = method;
        return this;
    }

    public HttpClient setRequestHeader(String key, String value) {
        if (!TextUtils.isEmpty(key) && value != null) {
            headerMap.put(key, value);
        }
        return this;
    }

    public HttpClient setRequestBody(String body) {
        requestBody = body;
        return this;
    }

    public HttpClient setUserAgent(String userAgent) {
        this.userAgent = userAgent;
        return this;
    }

    public HttpClient setEnctyType(String enctyType) {
        this.enctyType = enctyType;
        return this;
    }

    public HttpResponse request() throws Exception {
        if (url == null)
            throw new IOException("URL is empty");

        makeConnection();

        return new HttpResponse(conn);
    }

    public HttpResponse requestUpload(Map<String, String> params, List<String> files) throws Exception {
        if (url == null)
            throw new IOException("URL is empty");

        makeUploadConnection(params, files);

        return new HttpResponse(conn);
    }

    private void makeUploadConnection(Map<String, String> params, List<String> files) throws Exception {
        conn = (HttpURLConnection) url.openConnection();
        conn.setConnectTimeout(connectTimeout);
        conn.setReadTimeout(readTimeout);
        conn.setDoOutput(true);
        conn.setDoInput(true);
        conn.setUseCaches(useCaches);
        conn.setChunkedStreamingMode(1024 * 1024);
        conn.setRequestProperty("Accept", "*/*");
        conn.setRequestProperty("Connection", "keep-alive");
        conn.setRequestProperty("Charset", "UTF-8");
        conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);

        OutputStream os = conn.getOutputStream();
        writeParams(params, os);
        writeFile(files, os);
        // 请求结束标志
        String endTarget = PREFIX + BOUNDARY + PREFIX + LINE_END;
        os.write(endTarget.getBytes());
        if (os != null) {
            os.close();
        }
    }


    public HttpClient setmHostName(String mHostName) {
        this.mHostName = mHostName;
        return this;
    }

    private void makeConnection() throws Exception {
        try {
            conn = (HttpURLConnection) url.openConnection();
            conn.setConnectTimeout(connectTimeout);
            conn.setReadTimeout(readTimeout);
            conn.setUseCaches(useCaches);
            conn.setRequestMethod(requestMethod);
            conn.setInstanceFollowRedirects(true);

            if (userAgent != null) {
                conn.setRequestProperty("User-Agent", userAgent);
            }

            if (headerMap != null && !headerMap.isEmpty()) {
                for (String key : headerMap.keySet()) {
                    conn.setRequestProperty(key, headerMap.get(key));
                }
            }

            if (METHOD_GET.equals(requestMethod) || METHOD_HEAD.equals(requestMethod)) {
                conn.setDoOutput(false);
            } else if (METHOD_POST.equals(requestMethod)) {
                conn.setDoOutput(true);
            }

            if (mHostName != null) {
                conn.setRequestProperty("Host", mHostName);
            }

            if (requestBody != null) {
                if (enctyType != null) {
                    conn.setRequestProperty("Content-type", enctyType);
                } else {
                    conn.setRequestProperty("Content-type", "application/json;charset=UTF-8");
                }

                OutputStream os = conn.getOutputStream();
                byte[] bytes = requestBody.getBytes("UTF-8");
                String res = new String(bytes,"UTF-8");
                LogUtils.i("request map result"+res);
                os.write(bytes);
                os.close();
            }

            if (conn.getResponseCode() >= 300 && conn.getResponseCode() < 400) {
                String location = conn.getHeaderField("Location");
                if (TextUtils.isEmpty(location)) {
                    location = conn.getHeaderField("location");

                }
                if (!redirected && !TextUtils.isEmpty(location)) {
                    redirected = true;
                    setUrl(location);
                    makeConnection();
                }
            }

        } catch (Throwable e) {
            throw new Exception(e);
        }
    }

    /**
     * 对post参数进行编码处理并写入数据流中
     *
     * @throws IOException
     */
    private static void writeParams(Map<String, String> requestText,
                                    OutputStream os) throws Exception {
        try {
            if (requestText == null || requestText.isEmpty()) {
                return;
            } else {
                StringBuilder requestParams = new StringBuilder();
                Set<Map.Entry<String, String>> set = requestText.entrySet();
                Iterator<Entry<String, String>> it = set.iterator();
                while (it.hasNext()) {
                    Entry<String, String> entry = it.next();
                    requestParams.append(PREFIX).append(BOUNDARY).append(LINE_END);
                    requestParams.append("Content-Disposition: form-data; name=\"")
                            .append(entry.getKey()).append("\"").append(LINE_END);
                    requestParams.append("Content-Type: text/plain; charset=utf-8")
                            .append(LINE_END);
                    requestParams.append("Content-Transfer-Encoding: 8bit").append(
                            LINE_END);
                    requestParams.append(LINE_END);// 参数头设置完以后需要两个换行，然后才是参数内容
                    requestParams.append(entry.getValue());
                    requestParams.append(LINE_END);
                }
                os.write(requestParams.toString().getBytes());
                os.flush();
            }
        } catch (Exception e) {
            throw new Exception(e);
        }
    }

    /**
     * 对post上传的文件进行编码处理并写入数据流中
     *
     * @throws IOException
     */

    /**
     * 对post上传的文件进行编码处理并写入数据流中
     *
     */

    private static void writeFile(List<String> files, OutputStream os) {
        InputStream is = null;
        try {
            if (files == null || files.isEmpty()) {
                return;
            } else {
                // 将file拆分成 图片和视频
                List<String> videoFiles = new ArrayList<>();
                List<String> imgFiles = new ArrayList<>();
                for (String f : files) {
                    if (f.endsWith(".mp4")) {
                        videoFiles.add(f);
                    } else {
                        imgFiles.add(f);
                    }
                }
                uploadFileInCode(videoFiles, VIDEO_REQUESTKEY, os);
                uploadFileInCode(imgFiles, PIC_REQUESTKEY, os);
                // 上传格式
                /*requestParams 2:
                POST /api/v1/question/uploadPic HTTP/1.1
                Host: test-ul.haloapps.com
                Cache-Control: no-cache
                Postman-Token: 7014efd2-1bd9-8bbb-58ff-fefce199f773
                Content-Type: multipart/form-data; boundary=----WebKitFormBoundary7MA4YWxkTrZu0gW

                ------WebKitFormBoundary7MA4YWxkTrZu0gW
                Content-Disposition: form-data; name="gameGuestId"

                77524102784483902
                ------WebKitFormBoundary7MA4YWxkTrZu0gW
                Content-Disposition: form-data; name="token"

                8ab5bdab06f54d9bbf050868d31edad8
                ------WebKitFormBoundary7MA4YWxkTrZu0gW
                Content-Disposition: form-data; name="video_file"; filename="Recorder_20180730195918.mp4"
                Content-Type: video/mp4
                  file 1
                  file 2
                  file 3

                ------WebKitFormBoundary7MA4YWxkTrZu0gW
                Content-Disposition: form-data; name="picFile"; filename="panda.jpg"
                Content-Type: image/jpeg

                  file 1
                  file 2
                  file 3
                ------WebKitFormBoundary7MA4YWxkTrZu0gW--
                */


            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeStream(is);
        }
    }

    private static void uploadFileInCode(List<String> files, String requestKey, OutputStream os) {
        boolean first = false;
        InputStream is = null;
        try {
            if (files.size() > 0) {
                StringBuilder requestParams = new StringBuilder();
                for (String f : files) {
                    File file = new File(f);
                    if (!first) {
                        requestParams.append(PREFIX).append(BOUNDARY).append(LINE_END);
                        requestParams.append("Content-Disposition: form-data; name=\"").append(requestKey).append("\"; filename=\"")
                                .append(file.getName()).append("\"").append(LINE_END);
                        requestParams.append("Content-Type: application/octet-stream; charset=" + "UTF-8" + LINE_END);
                        requestParams.append(LINE_END);
                    }
                    first = true;
                    os.write(requestParams.toString().getBytes());
                    is = new BufferedInputStream(new FileInputStream(file));
                    byte[] bytes = new byte[1024];
                    int len = -1;
                    while ((len = is.read(bytes)) != -1) {
                        os.write(bytes, 0, len);
                    }
                    os.write(LINE_END.getBytes());
                    os.flush();
                }
                LogUtils.i("file upload string :" + requestParams.toString());
            }
        } catch (Throwable e) {
            LogUtils.i("file upload hase error :" + e.getMessage());
        } finally {
            try {
                is.close();
            } catch (Exception e) {

            }
        }

    }

    private static void closeStream(Closeable stream) {

        if (stream != null) {
            try {
                stream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

}
