//
// MessageSocket.java
//
// Copyright (c) 2018 Couchbase, Inc.  All rights reserved.
//
// Licensed under the Couchbase License Agreement (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// https://info.couchbase.com/rs/302-GJY-034/images/2017-10-30_License_Agreement.pdf
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package com.couchbase.lite;

import com.couchbase.litecore.C4Socket;

import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;

import static com.couchbase.litecore.C4Constants.C4ErrorDomain.WebSocketDomain;
import static com.couchbase.litecore.C4WebSocketCloseCode.kWebSocketCloseUserTransient;
import static com.couchbase.litecore.C4WebSocketCloseCode.kWebSocketCloseUserPermanent;
import static com.couchbase.litecore.C4WebSocketCloseCode.kWebSocketCloseNormal;

/* Internal MessageSocket for MessageEndpoint replication. */
class MessageSocket extends C4Socket implements ReplicatorConnection {
    // ---------------------------------------------------------------------------------------------
    // Variables
    // ---------------------------------------------------------------------------------------------

    private ProtocolType protocolType;
    private MessageEndpointConnection connection;
    private boolean sendResponseStatus;
    private boolean closed;
    private ScheduledExecutorService executor;

    // ---------------------------------------------------------------------------------------------
    // constructor
    // ---------------------------------------------------------------------------------------------

    public MessageSocket(long handle, MessageEndpoint endpoint, Map<String, Object> options) {
        super();
        init(handle, endpoint.getDelegate().createConnection(endpoint), endpoint.getProtocolType());
    }

    public MessageSocket(MessageEndpointConnection connection, ProtocolType protocolType) {
        super();

        C4Socket.socketFactory.put(this, MessageSocket.class);
        String path = String.format(Locale.ENGLISH, "/%s", Integer.toHexString(connection.hashCode()));
        int framing = C4Socket.kC4NoFraming;
        if (protocolType == ProtocolType.BYTE_STREAM)
            framing = C4Socket.kC4WebSocketClientFraming;
        long handle = C4Socket.fromNative(this, "x-msg-conn", "", 0, path, framing);
        init(handle, connection, protocolType);
    }

    private void init(long handle, MessageEndpointConnection connection, ProtocolType protocolType) {
        this.handle = handle;
        this.connection = connection;
        this.protocolType = protocolType;
        this.nativeHandle = this;
        this.closed = false;
        this.sendResponseStatus = true;
        executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable target) {
                return new Thread(target, "MessageSocket");
            }
        });
        C4Socket.reverseLookupTable.put(this.handle, this);
    }

    // ---------------------------------------------------------------------------------------------
    // Abstract method implementation of C4Socket
    // ---------------------------------------------------------------------------------------------

    @Override // socket_write
    protected void send(byte[] allocatedData) {
        final int length = allocatedData.length;
        connection.send(Message.fromData(allocatedData), new MessagingCompletion() {
            @Override
            public void complete(boolean success, MessagingError error) {
                if (success)
                    messageSent(length);
                else
                    close(error);

            }
        });
    }

    @Override // socket_completedReceive
    protected void completedReceive(long byteCount) { /* Not Implemented */ }

    @Override // socket_close
    protected void close() {
        connection.close(null, new MessagingCloseCompletion() {
            @Override
            public void complete() {
                connectionClosed(null);
            }
        });
    }

    @Override // socket_requestClose
    protected void requestClose(final int status, String message) {
        final Exception error = status != kWebSocketCloseNormal ?
                CBLStatus.convertException(WebSocketDomain, status, message, null) : null;
        connection.close(error, new MessagingCloseCompletion() {
            @Override
            public void complete() {
                MessagingError messagingError = null;
                if (error != null) {
                    messagingError = new MessagingError(error, status == kWebSocketCloseUserTransient);
                }
                connectionClosed(messagingError);
            }
        });
    }

    // ---------------------------------------------------------------------------------------------
    // ReplicatorConnection
    // ---------------------------------------------------------------------------------------------

    @Override
    public void close(final MessagingError error) {
        synchronized (this) {
            if (handle == 0L || closed)
                return;

            if (protocolType == ProtocolType.MESSAGE_STREAM) {
                int status = getStatusCode(error);
                String message = error != null ? error.getError().getMessage() : "";
                closeRequested(handle, status, message);
            }
        }

        if (protocolType == ProtocolType.BYTE_STREAM) {
            connection.close(error != null ? error.getError() : null, new MessagingCloseCompletion() {
                @Override
                public void complete() {
                    connectionClosed(error);
                }
            });
        }
    }

    @Override
    public void receive(Message message) {
        synchronized (this) {
            if (message == null)
                throw new IllegalArgumentException("message cannot be null.");

            if (handle == 0L || closed)
                return;
            received(handle, message.toData());
        }
    }


    // ---------------------------------------------------------------------------------------------
    // Socket Factory Callbacks
    // ---------------------------------------------------------------------------------------------

    public static void socket_open(
            long socket,
            Object socketFactoryContext,
            String scheme,
            String hostname,
            int port,
            String path,
            byte[] optionsFleece)  {
        final MessageSocket messageSocket = (MessageSocket) C4Socket.reverseLookupTable.get(socket);
        messageSocket.connection.open(messageSocket, new MessagingCompletion() {
            @Override
            public void complete(boolean success, MessagingError error) {
                if (success)
                    messageSocket.connectionOpened();
                else
                    messageSocket.connectionClosed(error);
            }
        });
    }

    // ---------------------------------------------------------------------------------------------
    // Package Level
    // ---------------------------------------------------------------------------------------------

    MessageEndpointConnection getConnection() {
        return connection;
    }

    //-------------------------------------------------------------------------
    // Acknowledgement
    //-------------------------------------------------------------------------

    void connectionOpened() {
        synchronized (this) {
            if (handle == 0L)
                return;

            if (sendResponseStatus)
                connectionGotResponse(200, null);
            opened(handle);
        }
    }

    void connectionClosed(MessagingError error) {
        synchronized (this) {
            if (handle == 0L || closed)
                return;

            closed = true;

            final int domain = error != null ? WebSocketDomain : 0;
            final int code = error != null ? getStatusCode(error) : 0;
            final String message = error != null ? error.getError().getMessage() : "";

            executor.execute(new Runnable() {
                @Override
                public void run() {
                    closed(handle, domain, code, message);
                }
            });
        }
    }

    void messageSent(int byteCount) {
        synchronized (this) {
            if (handle == 0L || closed)
                return;
            completedWrite(byteCount);
        }
    }

    // ---------------------------------------------------------------------------------------------
    // Private methods
    // ---------------------------------------------------------------------------------------------

    private void connectionGotResponse(int httpStatus, Map headers) {
        if (handle == 0L)
            return;
        gotHTTPResponse(handle, httpStatus, null);
        sendResponseStatus = false;
    }

    private int getStatusCode(MessagingError error) {
        if (error == null)
            return kWebSocketCloseNormal;
        return error.isRecoverable() ? kWebSocketCloseUserTransient : kWebSocketCloseUserPermanent;
    }
}
