/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.util;

import com.couchbase.client.core.annotation.Stability;
import com.couchbase.client.core.cnc.Event;
import com.couchbase.client.core.cnc.EventBus;
import com.couchbase.client.core.cnc.events.config.TlsRequiredButNotEnabledEvent;
import com.couchbase.client.core.cnc.events.core.DnsSrvLookupDisabledEvent;
import com.couchbase.client.core.cnc.events.core.DnsSrvLookupFailedEvent;
import com.couchbase.client.core.cnc.events.core.DnsSrvRecordsLoadedEvent;
import com.couchbase.client.core.env.CoreEnvironment;
import com.couchbase.client.core.env.SeedNode;
import com.couchbase.client.core.error.InvalidArgumentException;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.util.ConnectionString;
import com.couchbase.client.core.util.DnsSrv;
import com.couchbase.client.core.util.HostAndPort;
import com.couchbase.client.core.util.NanoTimestamp;
import java.net.SocketTimeoutException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

@Stability.Internal
public class ConnectionStringUtil {
    public static final String INCOMPATIBLE_CONNECTION_STRING_SCHEME = "Connection string scheme indicates a secure connection, but the pre-built ClusterEnvironment was not configured for TLS.";
    public static final String INCOMPATIBLE_CONNECTION_STRING_PARAMS = "Can't use a pre-built ClusterEnvironment with a connection string that has parameters.";

    private ConnectionStringUtil() {
    }

    public static Set<SeedNode> seedNodesFromConnectionString(ConnectionString connectionString, boolean dnsSrvEnabled, boolean tlsEnabled, EventBus eventBus) {
        if (dnsSrvEnabled && connectionString.isValidDnsSrv()) {
            String srvHostname = connectionString.hosts().get(0).host();
            NanoTimestamp start = NanoTimestamp.now();
            try {
                if (srvHostname.startsWith("_couchbase._tcp.")) {
                    srvHostname = srvHostname.replace("_couchbase._tcp.", "");
                } else if (srvHostname.startsWith("_couchbases._tcp.")) {
                    srvHostname = srvHostname.replace("_couchbases._tcp.", "");
                }
                List<String> foundNodes = ConnectionStringUtil.fromDnsSrvOrThrowIfTlsRequired(srvHostname, tlsEnabled);
                if (foundNodes.isEmpty()) {
                    throw new IllegalStateException("The loaded DNS SRV list from " + srvHostname + " is empty!");
                }
                Duration took = start.elapsed();
                eventBus.publish(new DnsSrvRecordsLoadedEvent(took, foundNodes));
                return foundNodes.stream().map(SeedNode::create).collect(Collectors.toSet());
            }
            catch (InvalidArgumentException t) {
                throw t;
            }
            catch (Throwable t) {
                Duration took = start.elapsed();
                if (t instanceof NameNotFoundException) {
                    eventBus.publish(new DnsSrvLookupFailedEvent(Event.Severity.INFO, took, null, DnsSrvLookupFailedEvent.Reason.NAME_NOT_FOUND));
                } else if (t.getCause() instanceof SocketTimeoutException) {
                    eventBus.publish(new DnsSrvLookupFailedEvent(Event.Severity.INFO, took, null, DnsSrvLookupFailedEvent.Reason.TIMED_OUT));
                } else {
                    eventBus.publish(new DnsSrvLookupFailedEvent(Event.Severity.WARN, took, t, DnsSrvLookupFailedEvent.Reason.OTHER));
                }
                return ConnectionStringUtil.populateSeedsFromConnectionString(connectionString);
            }
        }
        eventBus.publish(new DnsSrvLookupDisabledEvent(dnsSrvEnabled, connectionString.isValidDnsSrv()));
        return ConnectionStringUtil.populateSeedsFromConnectionString(connectionString);
    }

    static Set<SeedNode> populateSeedsFromConnectionString(ConnectionString connectionString) {
        Map<String, List<ConnectionString.UnresolvedSocket>> groupedByHost = connectionString.hosts().stream().collect(Collectors.groupingBy(ConnectionString.UnresolvedSocket::host));
        HashSet<SeedNode> seedNodes = new HashSet<SeedNode>();
        groupedByHost.forEach((host, addresses) -> {
            EnumMap ports = new EnumMap(ConnectionString.PortType.class);
            ConnectionString.PortType assumedPortType = connectionString.scheme() == ConnectionString.Scheme.COUCHBASE2 ? ConnectionString.PortType.PROTOSTELLAR : ConnectionString.PortType.KV;
            addresses.stream().filter(it -> it.port() != 0).forEach(it -> {
                if (connectionString.scheme() == ConnectionString.Scheme.COUCHBASE2 && it.portType().isPresent() && it.portType().get() != ConnectionString.PortType.PROTOSTELLAR) {
                    throw InvalidArgumentException.fromMessage("Invalid connection string. Port type " + (Object)((Object)it.portType().get()) + " is not compatible with scheme " + (Object)((Object)connectionString.scheme()));
                }
                ports.put(it.portType().orElse(assumedPortType), it.port());
            });
            seedNodes.add(SeedNode.create(host).withKvPort((Integer)ports.get((Object)ConnectionString.PortType.KV)).withManagerPort((Integer)ports.get((Object)ConnectionString.PortType.MANAGER)).withProtostellarPort((Integer)ports.get((Object)ConnectionString.PortType.PROTOSTELLAR)));
        });
        ConnectionStringUtil.sanityCheckSeedNodes(connectionString.original(), seedNodes);
        return seedNodes;
    }

    private static void sanityCheckSeedNodes(String connectionString, Set<SeedNode> seedNodes) {
        for (SeedNode seedNode : seedNodes) {
            if (!seedNode.kvPort().isPresent() || seedNode.kvPort().get() != 8091 && seedNode.kvPort().get() != 18091) continue;
            String recommended = connectionString.replace(":8091", "").replace(":18091", "");
            throw new InvalidArgumentException("Specifying 8091 or 18091 in the connection string \"" + connectionString + "\" is likely not what you want (it would connect to key/value via the management port which does not work). Please omit the port and use \"" + recommended + "\" instead.", null, null);
        }
    }

    public static boolean isCapella(ConnectionString connectionString) {
        return connectionString.hosts().stream().allMatch(it -> it.host().endsWith(".cloud.couchbase.com"));
    }

    public static ConnectionString asConnectionString(Collection<SeedNode> nodes) {
        boolean hasProtostellarPort = nodes.stream().anyMatch(it -> it.protostellarPort().isPresent());
        boolean hasClassicPort = nodes.stream().anyMatch(it -> !it.protostellarPort().isPresent());
        if (hasClassicPort && hasProtostellarPort) {
            throw InvalidArgumentException.fromMessage("The seed nodes have an invalid combination of port types. Must be all Protostellar or all KV/Manager.");
        }
        ArrayList<String> addresses = new ArrayList<String>();
        for (SeedNode node : nodes) {
            if (node.protostellarPort().isPresent()) {
                addresses.add(new HostAndPort(node.address(), node.protostellarPort().get()).format());
                continue;
            }
            if (!node.kvPort().isPresent() && !node.clusterManagerPort().isPresent()) {
                addresses.add(new HostAndPort(node.address(), 0).format());
                continue;
            }
            node.kvPort().ifPresent(port -> addresses.add(new HostAndPort(node.address(), (int)port).format() + "=kv"));
            node.clusterManagerPort().ifPresent(port -> addresses.add(new HostAndPort(node.address(), (int)port).format() + "=manager"));
        }
        return ConnectionString.create(String.join((CharSequence)",", addresses)).withScheme(hasProtostellarPort ? ConnectionString.Scheme.COUCHBASE2 : ConnectionString.Scheme.COUCHBASE);
    }

    public static void checkConnectionString(CoreEnvironment env, boolean ownsEnvironment, ConnectionString connStr) {
        boolean capella;
        boolean tls = env.securityConfig().tlsEnabled();
        if (!ownsEnvironment) {
            if (!tls && connStr.scheme() == ConnectionString.Scheme.COUCHBASES) {
                throw new IllegalArgumentException(INCOMPATIBLE_CONNECTION_STRING_SCHEME);
            }
            if (!tls && connStr.scheme() == ConnectionString.Scheme.COUCHBASE2) {
                throw new IllegalArgumentException(INCOMPATIBLE_CONNECTION_STRING_SCHEME);
            }
            if (!connStr.params().isEmpty()) {
                throw new IllegalArgumentException(INCOMPATIBLE_CONNECTION_STRING_PARAMS);
            }
        }
        if ((capella = ConnectionStringUtil.isCapella(connStr)) && !tls) {
            env.eventBus().publish(ownsEnvironment ? TlsRequiredButNotEnabledEvent.forOwnedEnvironment() : TlsRequiredButNotEnabledEvent.forSharedEnvironment());
        }
    }

    @Stability.Internal
    public static List<String> fromDnsSrvOrThrowIfTlsRequired(String serviceName, boolean secure) throws NamingException {
        boolean full = false;
        if (secure) {
            return DnsSrv.fromDnsSrv(serviceName, false, true);
        }
        try {
            return DnsSrv.fromDnsSrv(serviceName, false, false);
        }
        catch (NameNotFoundException errorFromFirstLookup) {
            try {
                if (!DnsSrv.fromDnsSrv(serviceName, false, true).isEmpty()) {
                    throw InvalidArgumentException.fromMessage("The DNS SRV record for '" + RedactableArgument.redactSystem(serviceName) + "' indicates TLS must be used when connecting to this cluster. Please enable TLS by setting the 'security.enableTls' client setting to true. If the Cluster does not use a shared ClusterEnvironment, an alternative way to enable TLS is to prefix the connection string with \"couchbases://\" (note the final 's')");
                }
                throw errorFromFirstLookup;
            }
            catch (InvalidArgumentException propagateMe) {
                throw propagateMe;
            }
            catch (Exception e) {
                if (e != errorFromFirstLookup) {
                    errorFromFirstLookup.addSuppressed(e);
                }
                throw errorFromFirstLookup;
            }
        }
    }
}

