/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.diagnostics;

import com.couchbase.client.core.Core;
import com.couchbase.client.core.Reactor;
import com.couchbase.client.core.annotation.Stability;
import com.couchbase.client.core.config.BucketConfig;
import com.couchbase.client.core.config.ClusterConfig;
import com.couchbase.client.core.config.NodeInfo;
import com.couchbase.client.core.config.PortInfo;
import com.couchbase.client.core.diagnostics.EndpointPingReport;
import com.couchbase.client.core.diagnostics.PingResult;
import com.couchbase.client.core.diagnostics.PingState;
import com.couchbase.client.core.error.TimeoutException;
import com.couchbase.client.core.io.CollectionIdentifier;
import com.couchbase.client.core.msg.RequestContext;
import com.couchbase.client.core.msg.analytics.AnalyticsPingRequest;
import com.couchbase.client.core.msg.kv.KvPingRequest;
import com.couchbase.client.core.msg.kv.KvPingResponse;
import com.couchbase.client.core.msg.query.QueryPingRequest;
import com.couchbase.client.core.msg.search.SearchPingRequest;
import com.couchbase.client.core.msg.view.ViewPingRequest;
import com.couchbase.client.core.node.NodeIdentifier;
import com.couchbase.client.core.retry.RetryStrategy;
import com.couchbase.client.core.service.ServiceType;
import java.time.Duration;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

@Stability.Internal
public class HealthPinger {
    @Stability.Internal
    public static Mono<PingResult> ping(Core core, Optional<Duration> timeout, RetryStrategy retryStrategy, Set<ServiceType> serviceTypes, Optional<String> reportId, boolean clusterLevel) {
        return Mono.defer(() -> {
            Set<PingTarget> targets = HealthPinger.extractPingTargets(core.clusterConfig(), clusterLevel);
            if (serviceTypes != null && !serviceTypes.isEmpty()) {
                targets = targets.stream().filter(t -> serviceTypes.contains((Object)((PingTarget)t).serviceType)).collect(Collectors.toSet());
            }
            return HealthPinger.pingTargets(core, targets, timeout, retryStrategy).collectList().map(reports -> new PingResult(reports.stream().collect(Collectors.groupingBy(EndpointPingReport::type)), core.context().environment().userAgent().formattedShort(), reportId.orElse(UUID.randomUUID().toString())));
        });
    }

    @Stability.Internal
    static Set<PingTarget> extractPingTargets(ClusterConfig clusterConfig, boolean clusterLevel) {
        HashSet<PingTarget> targets = new HashSet<PingTarget>();
        if (clusterLevel) {
            if (clusterConfig.globalConfig() != null) {
                for (PortInfo portInfo : clusterConfig.globalConfig().portInfos()) {
                    for (ServiceType serviceType : portInfo.ports().keySet()) {
                        if (serviceType == ServiceType.KV || serviceType == ServiceType.VIEWS) continue;
                        targets.add(new PingTarget(serviceType, portInfo.identifier(), null));
                    }
                }
            }
            for (Map.Entry entry : clusterConfig.bucketConfigs().entrySet()) {
                for (NodeInfo nodeInfo : ((BucketConfig)entry.getValue()).nodes()) {
                    for (ServiceType serviceType : nodeInfo.services().keySet()) {
                        if (serviceType == ServiceType.KV || serviceType == ServiceType.VIEWS) continue;
                        targets.add(new PingTarget(serviceType, nodeInfo.identifier(), null));
                    }
                }
            }
        } else {
            for (Map.Entry<String, BucketConfig> entry : clusterConfig.bucketConfigs().entrySet()) {
                for (NodeInfo nodeInfo : entry.getValue().nodes()) {
                    for (ServiceType serviceType : nodeInfo.services().keySet()) {
                        if (serviceType != ServiceType.VIEWS && serviceType != ServiceType.KV) {
                            targets.add(new PingTarget(serviceType, nodeInfo.identifier(), null));
                            continue;
                        }
                        targets.add(new PingTarget(serviceType, nodeInfo.identifier(), entry.getKey()));
                    }
                }
            }
        }
        return targets;
    }

    private static Flux<EndpointPingReport> pingTargets(Core core, Set<PingTarget> targets, Optional<Duration> timeout, RetryStrategy retryStrategy) {
        return Flux.fromIterable(targets).flatMap(target -> HealthPinger.pingTarget(core, target, timeout, retryStrategy));
    }

    private static Mono<EndpointPingReport> pingTarget(Core core, PingTarget target, Optional<Duration> timeout, RetryStrategy retryStrategy) {
        RetryStrategy retry = retryStrategy == null ? core.context().environment().retryStrategy() : retryStrategy;
        switch (target.serviceType) {
            case QUERY: {
                return HealthPinger.pingQuery(core, target, timeout, retry);
            }
            case KV: {
                return HealthPinger.pingKv(core, target, timeout, retry);
            }
            case VIEWS: {
                return HealthPinger.pingViews(core, target, timeout, retry);
            }
            case SEARCH: {
                return HealthPinger.pingSearch(core, target, timeout, retry);
            }
            case MANAGER: {
                return Mono.empty();
            }
            case ANALYTICS: {
                return HealthPinger.pingAnalytics(core, target, timeout, retry);
            }
        }
        return Mono.error((Throwable)new IllegalStateException("Unknown service to ping, this is a bug!"));
    }

    private static EndpointPingReport assembleSuccessReport(RequestContext context, String channelId, Optional<String> namespace) {
        String dispatchTo = null;
        String dispatchFrom = null;
        if (context.lastDispatchedTo() != null) {
            dispatchTo = context.lastDispatchedTo().toString();
        }
        if (context.lastDispatchedFrom() != null) {
            dispatchFrom = context.lastDispatchedFrom().toString();
        }
        return new EndpointPingReport(context.request().serviceType(), "0x" + channelId, dispatchFrom, dispatchTo, PingState.OK, namespace, Duration.ofNanos(context.logicalRequestLatency()), Optional.empty());
    }

    private static EndpointPingReport assembleFailureReport(Throwable throwable, RequestContext context, Optional<String> namespace) {
        String dispatchTo = null;
        String dispatchFrom = null;
        if (context.lastDispatchedTo() != null) {
            dispatchTo = context.lastDispatchedTo().toString();
        }
        if (context.lastDispatchedFrom() != null) {
            dispatchFrom = context.lastDispatchedFrom().toString();
        }
        PingState state = throwable instanceof TimeoutException ? PingState.TIMEOUT : PingState.ERROR;
        return new EndpointPingReport(context.request().serviceType(), null, dispatchFrom, dispatchTo, state, namespace, state == PingState.TIMEOUT ? context.request().timeout() : Duration.ofNanos(context.logicalRequestLatency()), Optional.empty());
    }

    private static Mono<EndpointPingReport> pingKv(Core core, PingTarget target, Optional<Duration> userTimeout, RetryStrategy retryStrategy) {
        return Mono.defer(() -> {
            Duration timeout = userTimeout.orElse(core.context().environment().timeoutConfig().kvTimeout());
            CollectionIdentifier collectionIdentifier = CollectionIdentifier.fromDefault(target.bucketName);
            KvPingRequest request = new KvPingRequest(timeout, core.context(), retryStrategy, collectionIdentifier, target.nodeIdentifier);
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                return HealthPinger.assembleSuccessReport(request.context(), ((KvPingResponse)response).channelId(), Optional.ofNullable(target.bucketName));
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete();
                return Mono.just((Object)HealthPinger.assembleFailureReport(throwable, request.context(), Optional.ofNullable(target.bucketName)));
            });
        });
    }

    private static Mono<EndpointPingReport> pingQuery(Core core, PingTarget target, Optional<Duration> userTimeout, RetryStrategy retryStrategy) {
        return Mono.defer(() -> {
            Duration timeout = userTimeout.orElse(core.context().environment().timeoutConfig().queryTimeout());
            QueryPingRequest request = new QueryPingRequest(timeout, core.context(), retryStrategy, target.nodeIdentifier);
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                return HealthPinger.assembleSuccessReport(request.context(), response.channelId(), Optional.empty());
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete();
                return Mono.just((Object)HealthPinger.assembleFailureReport(throwable, request.context(), Optional.empty()));
            });
        });
    }

    private static Mono<EndpointPingReport> pingAnalytics(Core core, PingTarget target, Optional<Duration> userTimeout, RetryStrategy retryStrategy) {
        return Mono.defer(() -> {
            Duration timeout = userTimeout.orElse(core.context().environment().timeoutConfig().analyticsTimeout());
            AnalyticsPingRequest request = new AnalyticsPingRequest(timeout, core.context(), retryStrategy, target.nodeIdentifier);
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                return HealthPinger.assembleSuccessReport(request.context(), response.channelId(), Optional.empty());
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete();
                return Mono.just((Object)HealthPinger.assembleFailureReport(throwable, request.context(), Optional.empty()));
            });
        });
    }

    private static Mono<EndpointPingReport> pingViews(Core core, PingTarget target, Optional<Duration> userTimeout, RetryStrategy retryStrategy) {
        return Mono.defer(() -> {
            Duration timeout = userTimeout.orElse(core.context().environment().timeoutConfig().viewTimeout());
            ViewPingRequest request = new ViewPingRequest(timeout, core.context(), retryStrategy, target.bucketName, target.nodeIdentifier);
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                return HealthPinger.assembleSuccessReport(request.context(), response.channelId(), Optional.ofNullable(target.bucketName));
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete();
                return Mono.just((Object)HealthPinger.assembleFailureReport(throwable, request.context(), Optional.ofNullable(target.bucketName)));
            });
        });
    }

    private static Mono<EndpointPingReport> pingSearch(Core core, PingTarget target, Optional<Duration> userTimeout, RetryStrategy retryStrategy) {
        return Mono.defer(() -> {
            Duration timeout = userTimeout.orElse(core.context().environment().timeoutConfig().searchTimeout());
            SearchPingRequest request = new SearchPingRequest(timeout, core.context(), retryStrategy, target.nodeIdentifier);
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                return HealthPinger.assembleSuccessReport(request.context(), response.channelId(), Optional.empty());
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete();
                return Mono.just((Object)HealthPinger.assembleFailureReport(throwable, request.context(), Optional.empty()));
            });
        });
    }

    @Stability.Internal
    public static class PingTarget {
        private final ServiceType serviceType;
        private final NodeIdentifier nodeIdentifier;
        private final String bucketName;

        PingTarget(ServiceType serviceType, NodeIdentifier nodeIdentifier, String bucketName) {
            this.serviceType = serviceType;
            this.nodeIdentifier = nodeIdentifier;
            this.bucketName = bucketName;
        }

        public ServiceType serviceType() {
            return this.serviceType;
        }

        public NodeIdentifier nodeIdentifier() {
            return this.nodeIdentifier;
        }

        public String bucketName() {
            return this.bucketName;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PingTarget that = (PingTarget)o;
            return this.serviceType == that.serviceType && Objects.equals(this.nodeIdentifier, that.nodeIdentifier) && Objects.equals(this.bucketName, that.bucketName);
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.serviceType, this.nodeIdentifier, this.bucketName});
        }

        public String toString() {
            return "PingTarget{serviceType=" + (Object)((Object)this.serviceType) + ", nodeIdentifier=" + this.nodeIdentifier + ", bucketName='" + this.bucketName + '\'' + '}';
        }
    }
}

