/*
 * Decompiled with CFR 0.152.
 */
package com.cosium.openid_connect.mock.server;

import com.cosium.openid_connect.mock.server.AuthorizationCode;
import com.cosium.openid_connect.mock.server.Client;
import com.cosium.openid_connect.mock.server.ClientBasicAuthentication;
import com.cosium.openid_connect.mock.server.User;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.http.HttpServer;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.auth.PubSecKeyOptions;
import io.vertx.ext.auth.jwt.JWTAuth;
import io.vertx.ext.auth.jwt.JWTAuthOptions;
import io.vertx.ext.jwt.JWTOptions;
import io.vertx.ext.web.Router;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.handler.BodyHandler;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;

public class OpenIdConnectServer {
    private static final String ADDRESS_TEMPLATE = "http://127.0.0.1:%s";
    private static final char[] KEYSTORE_PASSWORD = "changeit".toCharArray();
    private static final String KEY_ALIAS = "oidcserver";
    private static final char[] KEY_PASSWORD = "changeit".toCharArray();
    private static final String KTY = "RSA";
    private static final String ALGORITHM = "RS256";
    private static final String BEARER_TYPE = "Bearer";
    private static final String REFRESH_TYPE = "Offline";
    private static final String ID_TYPE = "ID";
    private static final String AUTHORIZATION_CODE_GRANT_TYPE = "authorization_code";
    private final RSAPrivateKey rsaPrivateKey;
    private final RSAPublicKey rsaPublicKey;
    private final Vertx vertx;
    private final HttpServer server;
    private final URI uri;
    private final Map<String, Client> clientById = new HashMap<String, Client>();
    private User currentUser;

    private OpenIdConnectServer() throws KeyStoreException, IOException, CertificateException, NoSuchAlgorithmException, UnrecoverableKeyException {
        String keystoreName = "keystore.pkcs12";
        KeyStore keyStore = KeyStore.getInstance("pkcs12");
        try (InputStream keystoreStream = this.getClass().getResourceAsStream(keystoreName);){
            keyStore.load(keystoreStream, KEYSTORE_PASSWORD);
        }
        if (!keyStore.containsAlias(KEY_ALIAS)) {
            throw new IllegalStateException("Could not find alias oidcserver in keystore " + keystoreName);
        }
        Key key = keyStore.getKey(KEY_ALIAS, KEY_PASSWORD);
        if (!(key instanceof RSAPrivateKey)) {
            throw new IllegalStateException("Alias oidcserver of keystore " + keystoreName + " is not bound to an RSA private key");
        }
        this.rsaPrivateKey = (RSAPrivateKey)key;
        Certificate certificate = keyStore.getCertificate(KEY_ALIAS);
        PublicKey publicKey = certificate.getPublicKey();
        if (!(publicKey instanceof RSAPublicKey)) {
            throw new IllegalStateException("Alias oidcserver of keystore " + keystoreName + " is not bound to an RSA public key");
        }
        this.rsaPublicKey = (RSAPublicKey)publicKey;
        this.vertx = Vertx.vertx();
        Router router = Router.router((Vertx)this.vertx);
        router.route().handler((Handler)BodyHandler.create());
        this.server = this.vertx.createHttpServer().requestHandler((Handler)router);
        CountDownLatch serverStartedLatch = new CountDownLatch(1);
        this.server.listen(0, event -> serverStartedLatch.countDown());
        try {
            serverStartedLatch.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        this.uri = URI.create(String.format(ADDRESS_TEMPLATE, this.server.actualPort()));
        router.get("/auth").handler(this::authenticate);
        router.post("/token").handler(this::createToken);
        router.get("/certs").handler(this::getJWKSet);
    }

    public URI uri() {
        return this.uri;
    }

    public static OpenIdConnectServer start() {
        try {
            return new OpenIdConnectServer();
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
            throw new RuntimeException(e);
        }
    }

    public void registerClient(String id, String secret) {
        this.clientById.put(id, new Client(id, secret));
    }

    public void setCurrentUser(User user) {
        this.currentUser = user;
    }

    public void stop() {
        this.server.close();
    }

    public void reset() {
        this.clientById.clear();
        this.currentUser = null;
    }

    private void authenticate(RoutingContext routingContext) {
        if (this.currentUser == null) {
            routingContext.response().setStatusCode(501).end("No current user set in " + this);
            return;
        }
        HttpServerRequest request = routingContext.request();
        String clientId = request.getParam("client_id");
        Client client = this.clientById.get(clientId);
        if (client == null) {
            routingContext.response().setStatusCode(404).end("No client found for client id '" + clientId + "'");
            return;
        }
        String state = request.getParam("state");
        String redirectUri = request.getParam("redirect_uri");
        String nonce = request.getParam("nonce");
        AuthorizationCode authorizationCode = new AuthorizationCode(client, redirectUri, nonce);
        client.authorizationCodesByValue.put(authorizationCode.value, authorizationCode);
        routingContext.response().setStatusCode(302).putHeader("Location", redirectUri + "?code=" + authorizationCode.value + "&state=" + state).end();
    }

    private void createToken(RoutingContext routingContext) {
        ClientBasicAuthentication clientAuthentication = new ClientBasicAuthentication(routingContext.request());
        if (!clientAuthentication.isComplete()) {
            routingContext.response().setStatusCode(401).end("Missing client id or client secret in the provided basic authentication");
            return;
        }
        Client client = this.clientById.get(clientAuthentication.clientId);
        if (client == null) {
            routingContext.response().setStatusCode(401).end("No client found for client id '" + clientAuthentication.clientId + "'");
            return;
        }
        if (!client.matchSecret(clientAuthentication.clientSecret)) {
            routingContext.response().setStatusCode(401).end("Provided secret '" + clientAuthentication.clientSecret + "' does not match client '" + client.id + "' secret");
            return;
        }
        HttpServerRequest request = routingContext.request();
        String grantType = request.getFormAttribute("grant_type");
        if (!AUTHORIZATION_CODE_GRANT_TYPE.equals(grantType)) {
            routingContext.response().setStatusCode(400).end("Received grant type '" + grantType + "'. '" + AUTHORIZATION_CODE_GRANT_TYPE + "' is the only supported grant type.");
            return;
        }
        String authorizationCodeValue = routingContext.request().getFormAttribute("code");
        AuthorizationCode authorizationCode = client.authorizationCodesByValue.get(authorizationCodeValue);
        if (authorizationCode == null) {
            routingContext.response().setStatusCode(401).end("Unrecognized authorization code '" + authorizationCodeValue + "'");
            return;
        }
        String redirectUri = request.getFormAttribute("redirect_uri");
        if (!authorizationCode.redirectUri.equals(redirectUri)) {
            routingContext.response().setStatusCode(401).end("The provided redirect uri '" + redirectUri + "' does not match the authorization redirect uri '" + authorizationCode.redirectUri + "'");
            return;
        }
        PubSecKeyOptions pubSecKeyOptions = new PubSecKeyOptions().setAlgorithm(ALGORITHM).setPublicKey(Base64.getEncoder().encodeToString(this.rsaPublicKey.getEncoded())).setSecretKey(Base64.getEncoder().encodeToString(this.rsaPrivateKey.getEncoded()));
        JWTAuthOptions jwtAuthOptions = new JWTAuthOptions().addPubSecKey(pubSecKeyOptions);
        JWTAuth jwtProvider = JWTAuth.create((Vertx)this.vertx, (JWTAuthOptions)jwtAuthOptions);
        JWTOptions jwtOptions = new JWTOptions().setAlgorithm(ALGORITHM);
        String accessToken = jwtProvider.generateToken(this.createJwtClaims(authorizationCode, BEARER_TYPE), jwtOptions);
        String refreshToken = jwtProvider.generateToken(this.createJwtClaims(authorizationCode, REFRESH_TYPE), jwtOptions);
        String idToken = jwtProvider.generateToken(this.createJwtClaims(authorizationCode, ID_TYPE), jwtOptions);
        JsonObject fullToken = new JsonObject().put("access_token", accessToken).put("expires_in", Integer.valueOf(60)).put("refresh_expires_in", Integer.valueOf(0)).put("refresh_token", refreshToken).put("token_type", "bearer").put("id_token", idToken).put("not-before-policy", Integer.valueOf(0)).put("scope", "openid");
        routingContext.response().putHeader("content-type", "application/json").end(fullToken.encodePrettily());
    }

    private void getJWKSet(RoutingContext routingContext) {
        JsonObject jwk = new JsonObject().put("kty", KTY).put("alg", ALGORITHM).put("use", "sig").put("e", Base64.getEncoder().encodeToString(this.rsaPublicKey.getPublicExponent().toByteArray())).put("n", Base64.getEncoder().encodeToString(this.rsaPublicKey.getModulus().toByteArray()));
        routingContext.response().end(new JsonObject().put("keys", new JsonArray().add(jwk)).encodePrettily());
    }

    private JsonObject createJwtClaims(AuthorizationCode authorizationCode, String type) {
        long exp = ZonedDateTime.now().plus(1L, ChronoUnit.DAYS).toEpochSecond();
        long iat = ZonedDateTime.now().toEpochSecond();
        JsonObject claims = new JsonObject().put("jti", UUID.randomUUID().toString()).put("exp", Long.valueOf(exp)).put("nbf", Integer.valueOf(0)).put("iat", Long.valueOf(iat)).put("iss", this.uri.toString()).put("sub", this.currentUser.subject).put("typ", type).put("azp", authorizationCode.client.id).put("nonce", authorizationCode.nonce);
        if (BEARER_TYPE.equals(type) || ID_TYPE.equals(type)) {
            claims.put("name", this.currentUser.name).put("given_name", this.currentUser.givenName).put("family_name", this.currentUser.familyName).put("auth_time", Long.valueOf(authorizationCode.authenticationTime.toEpochSecond()));
        } else {
            claims.put("auth_time", Integer.valueOf(0));
        }
        if (BEARER_TYPE.equals(type) || ID_TYPE.equals(type)) {
            claims.put("acr", Integer.valueOf(0));
        }
        if (BEARER_TYPE.equals(type) || REFRESH_TYPE.equals(type)) {
            claims.put("scope", "openid");
        }
        if (REFRESH_TYPE.equals(type)) {
            claims.put("aud", this.uri.toString());
        } else if (ID_TYPE.equals(type)) {
            claims.put("aud", authorizationCode.client.id);
        }
        return claims;
    }
}

