package com.configureit.citpermissionlib;

import android.app.Activity;
import android.app.Dialog;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.provider.Settings;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;

import java.util.ArrayList;
import java.util.Locale;

/**
 *
 */
public class PermissionUtils {

    private static final int REQ_PERMISSION = 38;
    private static final int REQUEST_PERMISSION_SETTING = 39;

    private String requestCode;
    private Activity activity;
    private String[] permissions;
    private PermissionCallback callback;

    private PermissionUtils(Activity activity){
        this.activity =activity;
    }

    private void executeRequest(){
        ArrayList<String> requestedPermissions = new ArrayList<String>();
        for (String permission: permissions) {
            if (ContextCompat.checkSelfPermission(activity, permission) != PackageManager.PERMISSION_GRANTED) {
                requestedPermissions.add(permission);
            }
        }

        if(requestedPermissions.size() > 0){
            String[] arr = new String[requestedPermissions.size()];
            arr = requestedPermissions.toArray(arr);

            ActivityCompat.requestPermissions(activity,
                    arr,
                    REQ_PERMISSION);
        }else{
            //all permission are already accessed
            if(callback != null){
                callback.onResult(requestCode, true, permissions, null, null);
            }
        }
    }

    public void retry(){
        executeRequest();
    }

    public static void showExplicitPermissionSetting(Activity activity){
        Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS);
        Uri uri = Uri.fromParts("package", activity.getPackageName(), null);
        intent.setData(uri);
        activity.startActivityForResult(intent, REQUEST_PERMISSION_SETTING);
    }

    public static class Builder{
        private PermissionUtils pu;

        public Builder(Activity activity, String requestCode){
            pu = new PermissionUtils(activity);
            pu.requestCode = requestCode;
        }

        public Builder requestPermission(String... permissions){
            if(permissions == null || permissions.length < 1)
                throw new IllegalArgumentException("You must specify atleast one permission for request");
            pu.permissions = permissions;
            return this;
        }

        public Builder setCallback(PermissionCallback callback){
            pu.callback = callback;
            return this;
        }

        public PermissionUtils build(){
            return pu;
        }

    }

    public void execute(){
        executeRequest();
    }

    public void onRequestPermissionsResult(int code,
                                           String permissions[], int[] grantResults) {
        switch (code) {
            case REQ_PERMISSION: {
                if(grantResults == null || grantResults.length < 1){
                    if(callback != null){
                        callback.onResult(requestCode, false, null, permissions, null);
                    }
                }else if (permissions.length > 0) {
                    ArrayList<String> allows = new ArrayList<String>();
                    ArrayList<String> denys = new ArrayList<String>();
                    ArrayList<String> neverAskAgains = new ArrayList<String>();
                    int i = 0;
                    for (String permission:permissions) {
                        if(grantResults[i++] == PackageManager.PERMISSION_DENIED){
                            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
                                boolean showRationale = ActivityCompat.shouldShowRequestPermissionRationale(activity, permission );
                                if(!showRationale){
                                    neverAskAgains.add(permission);
                                    continue;
                                }
                            }
                            denys.add(permission);
                        }else
                            allows.add(permission);
                    }

                    String[] arr1 = new String[allows.size()];
                    String[] arr2 = new String[denys.size()];
                    String[] arr3 = new String[neverAskAgains.size()];

                    arr1 = allows.toArray(arr1);
                    arr2 = allows.toArray(arr2);
                    arr3 = allows.toArray(arr3);

                    if(callback != null){
                        boolean isSuccess = denys.size() == 0 && neverAskAgains.size() == 0;
                        callback.onResult(requestCode, isSuccess, arr1, arr2, arr3);
                    }
                }
            }
        }
    }

    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if(requestCode == REQUEST_PERMISSION_SETTING){
            retry();
        }
    }

    public interface PermissionCallback{

        void onResult(String requestCode, boolean isGrantedAll, String[] allowPermissions, String[] denyPermissions, String[] neverAskPermissions);
    }

    /**
     * This method used to show information dialog for permission deny or deny with never ask again
     * If user select never ask again then after dialog application setting activity will open or
     * ask for permission
     *
     */
    public void showPermissionErrorDialog(PermissionDialogClickListener dialogClickListener){
        showPermissionErrorDialog(null, null, dialogClickListener );
    }

    /**
     * This method used to show information dialog for permission deny or deny with never ask again
     * If user select never ask again then after dialog application setting activity will open or
     * ask for permission
     *
     * @param message : String : Message for user
     */
    public void showPermissionErrorDialog(String title, String message, PermissionDialogClickListener dialogClickListener){
        showPermissionErrorDialog(title, message, null, null, dialogClickListener );
    }
    /**
     * This method used to show information dialog for permission deny or deny with never ask again
     * If user select never ask again then after dialog application setting activity will open or
     * ask for permission
     *
     * @param message : String : Message for user
     */
    public void showPermissionErrorDialog(String title, String message, String positiveBtnText, String negativeBtnText, final PermissionDialogClickListener dialogClickListener){

        new PermissionDeniedDialog(activity).showPermissionErrorDialog(title, message, positiveBtnText, negativeBtnText, new PermissionDialogClickListener() {
            @Override
            public void positiveListener(Activity context, Dialog dialog) {
                showExplicitPermissionSetting(activity);
                if(dialogClickListener != null){
                    dialogClickListener.positiveListener(activity, dialog);
                }
            }

            @Override
            public void negativeListener(Activity context, Dialog dialog) {
                if(dialogClickListener != null){
                    dialogClickListener.negativeListener(activity, dialog);
                }
            }
        });

    }

    public static String[] getPermission(String permissions){
        if(TextUtils.isEmpty(permissions))
            return null;

        String[] s = permissions.split(",", 0);
        if(s != null && s.length > 0){
            String permission[] = new String[s.length];
            for (int i = 0; i < s.length; i++) {
                String per = s[i];
                String newPermission = "android.permission."+per.toUpperCase(Locale.US);
                permission[i] =  newPermission;
            }
            return permission;
        }

        return null;
    }



}
