package com.configureit.utils.alert;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.support.annotation.DrawableRes;
import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.design.widget.TextInputLayout;
import android.support.v7.app.AlertDialog;
import android.text.TextUtils;
import android.util.Log;
import android.util.Patterns;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;


/**
 * @author ConfigureIT Created on 12/4/16.
 */
public class CITAlert {

    private WeakReference<Context> mWeakReference;
    private String message;
    private String title;
    private String buttonPositive;
    private String buttonNegative;
    private String buttonNeutral;
    private String reQuestCode;

    private static AlertDialog alertDialog;

    private CITAlertListener listener;
    private View customView;
    private int alertType;
    private static Dialog dialog;

    private boolean isValidation = true;
    private boolean isCancelable = false;
    private boolean singleInstance = true;

    private String emptyEmailMessage = "Email should not be empty.";
    private String wrongEmailMessage = "Enter proper Email Address.";
    private String emptyPasswordMessage = "Password should not be empty.";

    @DrawableRes
    private int drawableIcon = 0;

    public static final int ALERT_TYPE_SIMPLE = 0;
    public static final int ALERT_TYPE_INPUT_PASSWORD = 1;
    public static final int ALERT_TYPE_INPUT_EMAIL = 2;
    public static final int ALERT_TYPE_INPUT_EMAIL_PASSWORD = 3;
    public static final int ALERT_TYPE_CUSTOM = 4;

    public static final int BUTTON_DISMISS = 0;
    public static final String BUTTON_DISMISS_TEXT = "cit_dismiss";

    @IntDef({ALERT_TYPE_SIMPLE, ALERT_TYPE_INPUT_EMAIL, ALERT_TYPE_INPUT_PASSWORD, ALERT_TYPE_INPUT_EMAIL_PASSWORD, ALERT_TYPE_CUSTOM})
    @Retention(RetentionPolicy.SOURCE)
    public @interface AlertType {
    }

    //Protected to prevent out side class Object creation
    protected CITAlert(@NonNull Context context) {
        mWeakReference = new WeakReference<>(context);
        //noinspection ConstantConditions
        if (context != null) {
            emptyEmailMessage = context.getString(R.string.cit_empty_email_dialog_message);
            wrongEmailMessage = context.getString(R.string.cit_valid_email_dialog_message);
            emptyPasswordMessage = context.getString(R.string.cit_empty_password_dialog_message);
            emailPlaceHolder = context.getString(R.string.cit_enter_email_dialog_hint);
            passwordPlaceHolder = context.getString(R.string.cit_enter_password_dialog_hint);
        }
    }

    /**
     * @param context The context to use.  Usually your {@link android.app.Application}
     *                or {@link android.app.Activity} object.
     * @return Class Object
     */
    public static CITAlert on(@NonNull Context context) {
        return new CITAlert(context);
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @param buttonNegative
     * @param buttonNeutral
     * @return Class instance
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive,
                          @NonNull String buttonNegative, @NonNull String buttonNeutral) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        this.buttonNegative = buttonNegative;
        this.buttonNeutral = buttonNeutral;
        return this;
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @param buttonNegative
     * @return Class instance
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive,
                          @NonNull String buttonNegative) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        this.buttonNegative = buttonNegative;
        return this;
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @return Class instance
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        return this;
    }

    /**
     * @param title
     * @return Class instance
     */
    public CITAlert title(@NonNull String title) {
        this.title = title;
        return this;
    }

    /**
     * @param message
     * @return Class instance
     */
    public CITAlert message(@NonNull String message) {
        this.message = message;
        return this;
    }

    /**
     * @param buttonPositive
     * @return Class instance
     */
    public CITAlert buttonPositive(@NonNull String buttonPositive) {
        this.buttonPositive = buttonPositive;
        return this;
    }

    /**
     * @param buttonNegative
     * @return Class instance
     */
    public CITAlert buttonNegative(@NonNull String buttonNegative) {
        this.buttonNegative = buttonNegative;
        return this;
    }

    /**
     * @param buttonNeutral
     * @return Class instance
     */
    public CITAlert buttonNeutral(@NonNull String buttonNeutral) {
        this.buttonNeutral = buttonNeutral;
        return this;
    }

    /**
     * @param reQuestCode String Request Code for Differentiate callbacks
     * @param listener    CITAlertListener
     * @return Class Instance
     */
    public CITAlert getCallBack(@NonNull String reQuestCode, @NonNull CITAlertListener listener) {
        this.reQuestCode = reQuestCode;
        this.listener = listener;
        return this;
    }

    /**
     * @param listener CITAlertListener
     * @return Class Instance
     */
    public CITAlert getCallBack(@NonNull CITAlertListener listener) {
        this.listener = listener;
        return this;
    }


    /**
     * @param alertType set CIT Alert type
     * @return Class Instance
     */
    public CITAlert alertType(@AlertType int alertType) {
        this.alertType = alertType;
        return this;
    }

    /**
     * @param view set Custom dialog inflated view object
     * @return Class Instance
     */
    public CITAlert setCustomView(@NonNull View view) {
        this.customView = view;
        return this;
    }

    /**
     * @param emptyInputWarning String message required for set empty input message
     * @return Class Instance
     */
    public CITAlert setEmptyInputWarning(@NonNull String emptyInputWarning) {
        if (TextUtils.isEmpty(emptyInputWarning)) return this;
        this.emptyEmailMessage = emptyInputWarning;
        return this;
    }

    /**
     * @param emptyPasswordWarning String message required for set empty input message
     * @return Class Instance
     */
    public CITAlert setEmptyPasswordWarning(@NonNull String emptyPasswordWarning) {
        if (TextUtils.isEmpty(emptyPasswordWarning)) return this;
        this.emptyPasswordMessage = emptyPasswordWarning;
        return this;
    }

    /**
     * @param wrongInputWarning String message required for set wrong input message
     * @return Class Instance
     */
    public CITAlert setWrongInputWarning(@NonNull String wrongInputWarning) {
        if (TextUtils.isEmpty(wrongInputWarning)) return this;
        this.wrongEmailMessage = wrongInputWarning;
        return this;
    }

    /**
     * It will set the input field Alert validation
     *
     * @param isValidation
     * @return Class Instance
     */
    public CITAlert isInputValidationRequire(boolean isValidation) {
        this.isValidation = isValidation;
        return this;
    }

    /**
     * @param drawableIcon
     * @return Class Instance
     */
    public CITAlert setIcon(@DrawableRes int drawableIcon) {
        this.drawableIcon = drawableIcon;
        return this;
    }

    /**
     * @param isCancelable
     * @return Class Instance
     */
    public CITAlert setCancelable(boolean isCancelable) {
        this.isCancelable = isCancelable;
        return this;
    }

    /**
     * @param isSingleInstance
     * @return Class Instance
     */
    public CITAlert setSingleInstance(boolean isSingleInstance) {
        this.singleInstance = isSingleInstance;
        return this;
    }

    private String emailPlaceHolder = "Enter Email";
    private String passwordPlaceHolder = "Enter password";
    private String emailText;
    private String passwordText;

    /**
     * @param emailPlaceHolder
     * @return Class Instance
     */
    public CITAlert setEmailPlaceHolder(@NonNull String emailPlaceHolder) {
        this.emailPlaceHolder = !TextUtils.isEmpty(emailPlaceHolder)
                ? emailPlaceHolder : this.emailPlaceHolder;
        return this;
    }


    /**
     * @param passwordPlaceHolder
     * @return Class Instance
     */
    public CITAlert setPasswordPlaceHolder(@NonNull String passwordPlaceHolder) {
        this.passwordPlaceHolder = !TextUtils.isEmpty(passwordPlaceHolder)
                ? passwordPlaceHolder : this.passwordPlaceHolder;
        return this;
    }


    /**
     * @param emailText
     * @return Class Instance
     */
    public CITAlert setEmailText(@NonNull String emailText) {
        this.emailText = emailText;
        return this;
    }


    /**
     * @param passwordText
     * @return Class Instance
     */
    public CITAlert setPasswordText(@NonNull String passwordText) {
        this.passwordText = passwordText;
        return this;
    }

    /**
     * create CIT Alert dialog according to the alert type
     *
     * @return Dialog
     */

    public Dialog show() {
        if(mWeakReference.get() != null) {
            if (singleInstance && alertDialog != null && alertDialog.isShowing()) return null;
            if (alertType == ALERT_TYPE_INPUT_EMAIL || alertType == ALERT_TYPE_INPUT_PASSWORD) {
                dialog = createAlertWithInput();
            } else if (alertType == ALERT_TYPE_INPUT_EMAIL_PASSWORD)
                dialog = createAlertWithTwoInput();
            else if (alertType == ALERT_TYPE_CUSTOM) {
                if (customView == null)
                    throw new NullPointerException("CIT CustomDialog view should not be empty.");
                dialog = showDialogWithView();
            } else {
                dialog = createDialog();
            }
        }
        return dialog;
    }


    public static void dismiss() {
        if (dialog != null && dialog.isShowing() && dialog.getContext() != null) {
            dialog.dismiss();
            dialog = null;
        }
    }

    /**
     * This method will create Simple alert dialog.
     *
     * @return
     */
    private AlertDialog createDialog() {
        try {
            if (mWeakReference.get() != null) {
                AlertDialog.Builder builder = new AlertDialog.Builder(mWeakReference.get());
                if (!TextUtils.isEmpty(title))
                    builder.setTitle(title);
                if (!TextUtils.isEmpty(message))
                    builder.setMessage(message);
                if (drawableIcon != 0) {
                    builder.setIcon(drawableIcon);
                }
                if (!TextUtils.isEmpty(buttonPositive)) {
                    builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int id) {
                            if(mWeakReference.get() != null) {
                                alertDialog.dismiss();
                                if (listener != null)
                                    listener.onPerformClick(reQuestCode, buttonPositive, id, "", "");
                            }

                        }
                    });
                }

                if (!TextUtils.isEmpty(buttonNegative)) {
                    builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int id) {
                            if(mWeakReference.get() != null) {
                                alertDialog.dismiss();
                                if (listener != null)
                                    listener.onPerformClick(reQuestCode, buttonNegative, id, "", "");
                            }

                        }
                    });
                }

                if (!TextUtils.isEmpty(buttonNeutral)) {
                    builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int id) {
                            if(mWeakReference.get() != null) {
                                alertDialog.dismiss();
                                if (listener != null)
                                    listener.onPerformClick(reQuestCode, buttonNeutral, id, "", "");
                            }

                        }
                    });
                }

                alertDialog = builder.create();
                alertDialog.setCancelable(isCancelable);
                alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                    @Override
                    public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                        if (keyCode == KeyEvent.KEYCODE_BACK && mWeakReference.get() != null) {
                            dialog.dismiss();
                            if (listener != null) {
//                            listener.onCancel(reQuestCode);
                                listener.onPerformClick(reQuestCode, BUTTON_DISMISS_TEXT, BUTTON_DISMISS, "", "");
                            }

                        }
                        return false;
                    }
                });
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return alertDialog;
    }

    /**
     * This method will create Alert Dialog with email or password Input Box view
     *
     * @return AlertDialog
     */
    private EditText inputBox;
    private TextInputLayout inputLayout;

    private AlertDialog createAlertWithInput() {
        if (mWeakReference.get() != null) {
            try {
                View inputDialogView;
                if (alertType == ALERT_TYPE_INPUT_EMAIL) {
                    inputDialogView = LayoutInflater.from(mWeakReference.get()).inflate(R.layout.dialog_input, null);
                    inputBox = (EditText) inputDialogView.findViewById(R.id.inputBoxEmail);
                    inputLayout = (TextInputLayout) inputDialogView.findViewById(R.id.inputLayoutEmail);
                    inputBox.setHint(emailPlaceHolder);
                    inputLayout.setHint(emailPlaceHolder);
                    if (!TextUtils.isEmpty(emailText))
                        inputBox.setText(emailText);
                } else {
                    inputDialogView = LayoutInflater.from(mWeakReference.get()).inflate(R.layout.dialog_input_secure, null);
                    inputBox = inputDialogView.findViewById(R.id.inputBoxPassword);
                    inputLayout = inputDialogView.findViewById(R.id.inputLayoutPassword);
                    inputBox.setHint(passwordPlaceHolder);
                    inputLayout.setHint(passwordPlaceHolder);
                    if (!TextUtils.isEmpty(passwordText))
                        inputBox.setText(passwordText);
                }

                TextView dialogTitle = inputDialogView.findViewById(R.id.dialogTitle);

                if (TextUtils.isEmpty(title)) {
                    dialogTitle.setVisibility(View.GONE);
                }
                else {
                    dialogTitle.setVisibility(View.VISIBLE);
                    dialogTitle.setText(title);
                }

                TextView dialogDescription = inputDialogView.findViewById(R.id.dialogMessage);
                if(dialogDescription != null) {
                    if (TextUtils.isEmpty(message))
                        dialogDescription.setVisibility(View.GONE);
                    else {
                        dialogDescription.setVisibility(View.VISIBLE);
                        dialogDescription.setText(message);
                    }
                }

                AlertDialog.Builder builder = new AlertDialog.Builder(mWeakReference.get(), Window.FEATURE_NO_TITLE);
                builder.setView(inputDialogView);
                if (drawableIcon != 0) {
                    builder.setIcon(drawableIcon);
                }
                if (!TextUtils.isEmpty(buttonPositive)) {
                    builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                        }
                    });
                }
                if (!TextUtils.isEmpty(buttonNegative)) {
                    builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            if(mWeakReference.get() != null) {
                                hideSoftKeyboard(mWeakReference.get(), inputBox);
                                alertDialog.dismiss();
                                if (listener != null) {
                                    listener.onPerformClick(reQuestCode, buttonNegative, AlertDialog.BUTTON_NEGATIVE, inputBox != null ? inputBox.getText().toString() : "", "");
//                            listener.onPerformClick(reQuestCode, buttonNegative, which, "", "");
                                }
                            }

                        }
                    });
                }
                if (!TextUtils.isEmpty(buttonNeutral)) {
                    builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            hideSoftKeyboard(mWeakReference.get(), inputBox);
                        }
                    });
                }

                alertDialog = builder.create();
                alertDialog.setCancelable(isCancelable);
                alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                    @Override
                    public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                        if (keyCode == KeyEvent.KEYCODE_BACK && mWeakReference.get() != null) {
                            hideSoftKeyboard(mWeakReference.get(), inputBox);
                            dialog.dismiss();
                            if (listener != null) {
//                            listener.onCancel(reQuestCode);
                                listener.onPerformClick(reQuestCode, BUTTON_DISMISS_TEXT, BUTTON_DISMISS, "", "");
                            }

                        }
                        return false;
                    }
                });

                alertDialog.setOnShowListener(new DialogInterface.OnShowListener() {
                    @Override
                    public void onShow(final DialogInterface dialog) {
                        Button n = alertDialog.getButton(AlertDialog.BUTTON_NEUTRAL);
                        n.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if(mWeakReference.get() != null) {
                                    hideSoftKeyboard(mWeakReference.get(), inputBox);
                                    inputBox.setText("");
                                    inputLayout.setError(null);
                                    if (listener != null) {
                                        listener.onPerformClick(reQuestCode, buttonNeutral, AlertDialog.BUTTON_NEUTRAL, inputBox != null ? inputBox.getText().toString() : "", "");
                                        //listener.onPerformClick(reQuestCode, buttonNeutral, AlertDialog.BUTTON_NEUTRAL, "", "");
                                    }
                                }
                            }
                        });
                        n = alertDialog.getButton(AlertDialog.BUTTON_NEGATIVE);
                        n.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if(mWeakReference.get() != null) {
                                    hideSoftKeyboard(mWeakReference.get(), inputBox);
                                    dialog.dismiss();
                                    inputBox.setText("");
                                    inputLayout.setError(null);
                                    if (listener != null) {
                                        listener.onPerformClick(reQuestCode, buttonNegative, AlertDialog.BUTTON_NEGATIVE, inputBox != null ? inputBox.getText().toString() : "", "");
                                        //listener.onPerformClick(reQuestCode, buttonNeutral, AlertDialog.BUTTON_NEUTRAL, "", "");
                                    }
                                }

                            }
                        });
                        Button p = alertDialog.getButton(AlertDialog.BUTTON_POSITIVE);
                        p.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if(mWeakReference.get() != null) {
                                    if (inputBox != null) {
                                        hideSoftKeyboard(mWeakReference.get(), inputBox);
                                        if (!isValidation) {
                                            dialog.dismiss();
                                            sendPositiveData(AlertDialog.BUTTON_POSITIVE);
                                        } else if (alertType == ALERT_TYPE_INPUT_PASSWORD) {
                                            if (TextUtils.isEmpty(inputBox.getText())) {
                                                inputLayout.setError(emptyPasswordMessage);
                                            } else {
                                                dialog.dismiss();
                                                sendPositiveData(AlertDialog.BUTTON_POSITIVE);
                                            }
                                        } else if (alertType == ALERT_TYPE_INPUT_EMAIL) {
                                            if (TextUtils.isEmpty(inputBox.getText())) {
                                                inputLayout.setError(emptyEmailMessage);
                                            } else if (!Patterns.EMAIL_ADDRESS.matcher(inputBox.getText().toString().trim()).matches()) {
                                                inputLayout.setError(wrongEmailMessage);
                                            } else {
//                                        Log.e("Input-->", inputBox.getText().toString());
                                                dialog.dismiss();
                                                sendPositiveData(AlertDialog.BUTTON_POSITIVE);

                                            }
                                        }
                                    } else dialog.dismiss();
                                }
                            }
                        });
                    }
                });
                alertDialog.show();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        return alertDialog;
    }

    /**
     * This method will create Alert Dialog with email and password Input Box view
     *
     * @return AlertDialog
     */
    private AlertDialog createAlertWithTwoInput() {
        if (mWeakReference.get() != null) {
            try {
                View inputDialogView = LayoutInflater.from(mWeakReference.get()).inflate(R.layout.dialog_two_input, null);
                final EditText inputBoxEmail =  inputDialogView.findViewById(R.id.inputBoxEmail);
                final EditText inputBoxPassword = inputDialogView.findViewById(R.id.inputBoxPassword);
                final TextInputLayout emailLayout = inputDialogView.findViewById(R.id.inputLayoutEmail);
                final TextInputLayout passwordLayout = inputDialogView.findViewById(R.id.inputLayoutPassword);

                TextView dialogTitle = (TextView) inputDialogView.findViewById(R.id.dialogTitle);

                inputBoxEmail.setHint(emailPlaceHolder);
                inputBoxPassword.setHint(passwordPlaceHolder);
                emailLayout.setHint(emailPlaceHolder);
                passwordLayout.setHint(passwordPlaceHolder);

                if (!TextUtils.isEmpty(emailText))
                    inputBoxEmail.setText(emailText);

                if (!TextUtils.isEmpty(passwordText))
                    inputBoxPassword.setText(passwordText);

                if (TextUtils.isEmpty(title))
                    dialogTitle.setVisibility(View.GONE);
                else {
                    dialogTitle.setVisibility(View.VISIBLE);
                    dialogTitle.setText(title);
                }

                TextView dialogDescription = inputDialogView.findViewById(R.id.dialogMessage);
                if(dialogDescription != null) {
                    if (TextUtils.isEmpty(message))
                        dialogDescription.setVisibility(View.GONE);
                    else {
                        dialogDescription.setVisibility(View.VISIBLE);
                        dialogDescription.setText(message);
                    }
                }

                AlertDialog.Builder builder = new AlertDialog.Builder(mWeakReference.get(), Window.FEATURE_NO_TITLE);
                builder.setView(inputDialogView);
                if (drawableIcon != 0) {
                    builder.setIcon(drawableIcon);
                }
                if (!TextUtils.isEmpty(buttonPositive)) {
                    builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                        }
                    });
                }
                if (!TextUtils.isEmpty(buttonNegative)) {
                    builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            if (mWeakReference.get() != null) {
                                hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                                if (listener != null)
                                    listener.onPerformClick(reQuestCode, buttonNegative, which, "", "");
                                alertDialog.dismiss();
                            }
                        }
                    });
                }
                if (!TextUtils.isEmpty(buttonNeutral)) {
                    builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                        }
                    });
                }

                alertDialog = builder.create();
                alertDialog.setCancelable(isCancelable);
                alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                    @Override
                    public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                        if (keyCode == KeyEvent.KEYCODE_BACK && mWeakReference.get() != null) {
                            hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                            if (listener != null) {
                                //listener.onCancel(reQuestCode);
                                listener.onPerformClick(reQuestCode, BUTTON_DISMISS_TEXT, BUTTON_DISMISS, "", "");
                            }
                            dialog.dismiss();
                        }
                        return false;
                    }
                });

                alertDialog.setOnShowListener(new DialogInterface.OnShowListener() {
                    @Override
                    public void onShow(final DialogInterface dialog) {
                        Button n = alertDialog.getButton(AlertDialog.BUTTON_NEUTRAL);
                        n.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if (mWeakReference.get() != null) {
                                    hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                                    inputBoxEmail.setText("");
                                    inputBoxPassword.setText("");
                                    emailLayout.setError(null);
                                    passwordLayout.setError(null);
                                    if (listener != null) {
                                        listener.onPerformClick(reQuestCode, buttonNeutral, AlertDialog.BUTTON_NEUTRAL,
                                                inputBoxEmail != null ? inputBoxEmail.getText().toString() : "",
                                                inputBoxPassword != null ? inputBoxPassword.getText().toString() : "");
                                    }
                                }
                            }
                        });
                        n = alertDialog.getButton(AlertDialog.BUTTON_NEGATIVE);
                        n.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if (mWeakReference.get() != null) {
                                    dialog.dismiss();
                                    if (listener != null) {
                                        listener.onPerformClick(reQuestCode, buttonNegative, AlertDialog.BUTTON_NEGATIVE,
                                                inputBoxEmail != null ? inputBoxEmail.getText().toString() : "",
                                                inputBoxPassword != null ? inputBoxPassword.getText().toString() : "");
                                    }
                                }

                            }
                        });

                        Button p = alertDialog.getButton(AlertDialog.BUTTON_POSITIVE);
                        p.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                if (mWeakReference.get() != null) {
                                    hideSoftKeyboard(mWeakReference.get(), inputBoxEmail);
                                    if (!isValidation) {
                                        if (listener != null)
                                            listener.onPerformClick(reQuestCode, buttonPositive, view.getId()
                                                    , inputBoxEmail.getText().toString()
                                                    , inputBoxPassword.getText().toString());
                                        dialog.dismiss();
                                    } else {
                                        if (TextUtils.isEmpty(inputBoxEmail.getText())) {
                                            emailLayout.setError(emptyEmailMessage);
                                        } else if (!Patterns.EMAIL_ADDRESS.matcher(inputBoxEmail.getText().toString().trim()).matches()) {
                                            emailLayout.setError(wrongEmailMessage);
                                        } else if (TextUtils.isEmpty(inputBoxPassword.getText())) {
                                            emailLayout.setError(emptyPasswordMessage);
                                        } else {
                                            Log.e("Input-->", inputBoxEmail.getText().toString());
                                            if (listener != null)
                                                listener.onPerformClick(reQuestCode, buttonPositive, view.getId()
                                                        , inputBoxEmail.getText().toString()
                                                        , inputBoxPassword.getText().toString());
                                            dialog.dismiss();
                                        }
                                    }
                                }
                            }
                        });
                    }
                });
                alertDialog.show();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        return alertDialog;
    }

    private void sendPositiveData(int id) {
        if (listener != null) {
            if (alertType == ALERT_TYPE_INPUT_EMAIL)
                listener.onPerformClick(reQuestCode, buttonPositive, id, inputBox.getText().toString(), "");
            else if (alertType == ALERT_TYPE_INPUT_PASSWORD)
                listener.onPerformClick(reQuestCode, buttonPositive, id, "", inputBox.getText().toString());
        }
    }

    /**
     * This method create dialog with inflated View object.
     *
     * @return Dialog
     */
    private Dialog showDialogWithView() {
        if (mWeakReference.get() != null) {
            try {
                dialog = new Dialog(mWeakReference.get());
                dialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
                dialog.getWindow().setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
                dialog.setContentView(customView);
                dialog.setCancelable(isCancelable);
                dialog.setCanceledOnTouchOutside(isCancelable);
                WindowManager.LayoutParams windowLayout = dialog.getWindow().getAttributes();
                windowLayout.gravity = Gravity.CENTER;
                dialog.show();
                dialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                    @Override
                    public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                        if (keyCode == KeyEvent.KEYCODE_BACK && mWeakReference.get() != null) {
                            if (listener != null) {
//                            listener.onCancel(reQuestCode);
                                listener.onPerformClick(reQuestCode, BUTTON_DISMISS_TEXT, BUTTON_DISMISS, "", "");
                            }
                            dialog.dismiss();
                        }
                        return false;
                    }
                });
                dialog.show();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        return dialog == null && mWeakReference.get() != null ? new Dialog(mWeakReference.get()) : dialog;
    }

    /**
     * @param textInputLayout
     * @param color
     */
    private void setErrorTextColor(TextInputLayout textInputLayout, int color) {
        try {
            Field fErrorView = TextInputLayout.class.getDeclaredField("mErrorView");
            fErrorView.setAccessible(true);
            TextView mErrorView = (TextView) fErrorView.get(textInputLayout);
            Field fCurTextColor = TextView.class.getDeclaredField("mCurTextColor");
            fCurTextColor.setAccessible(true);
            fCurTextColor.set(mErrorView, color);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static void hideSoftKeyboard(Context context, View v) {
        if (context != null) {
            if (v != null) {
                InputMethodManager imm = (InputMethodManager) context.getSystemService(Context.INPUT_METHOD_SERVICE);
                imm.hideSoftInputFromWindow(v.getWindowToken(), 0);
            } else {
                EditText editText = new EditText(context);
                InputMethodManager imm = (InputMethodManager) context.getSystemService(Context.INPUT_METHOD_SERVICE);
                imm.hideSoftInputFromWindow(editText.getWindowToken(), 0);
            }
        }
    }

    public static void showSoftKeyboard(Context context, View v) {
        if (context != null) {
            if (v != null) {
                InputMethodManager imm = (InputMethodManager) context.getSystemService(Context.INPUT_METHOD_SERVICE);
                imm.showSoftInput(v, InputMethodManager.SHOW_IMPLICIT);
            }
        }
    }
}