package com.configureit.utils.alert;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.GradientDrawable;
import android.support.annotation.ColorInt;
import android.support.annotation.ColorRes;
import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.support.v4.app.ActivityCompat;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.LinearLayout.LayoutParams;
import android.widget.TextView;
import android.widget.Toast;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * @author ConfigureIT Created on 10/5/16.
 */
public class CITToast {

    private Context mContext;

    private String message = "";

    private int bgColor = Color.DKGRAY;
    private int messageColor = Color.WHITE;

    @ToastType
    private int toastType = SIMPLE_TOAST;

    @Duration
    private int toastDuration = LENGTH_SHORT;

    public static final int SIMPLE_TOAST = 0;
    public static final int CUSTOM_TOAST = 1;

    @IntDef({SIMPLE_TOAST, CUSTOM_TOAST})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ToastType {
    }

    public static final int LENGTH_SHORT = 0;
    public static final int LENGTH_LONG = 1;

    @IntDef({LENGTH_SHORT, LENGTH_LONG})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Duration {
    }

    protected CITToast(@NonNull Context mContext) {
        this.mContext = mContext;
    }

    /**
     * @param mContext The context to use.  Usually your {@link android.app.Application}
     *                 or {@link android.app.Activity} object.
     * @return Class Instance
     */
    public static CITToast on(@NonNull Context mContext) {
        return new CITToast(mContext);
    }

    /**
     * @param message String for toast message
     * @return Class Instance
     */
    public CITToast message(@NonNull String message) {
        if (TextUtils.isEmpty(message)) return this;
        this.message = message;
        return this;
    }

    /**
     * @param message
     * @return Class Instance
     */
    public CITToast message(@StringRes int message) {
        if (TextUtils.isEmpty(mContext.getString(message))) return this;
        this.message = mContext.getString(message);
        return this;
    }

    /**
     * @param bgColour
     * @return Class Instance
     */
    public CITToast setbgColour(@NonNull String bgColour) {
        toastType = CUSTOM_TOAST;
        this.bgColor = validateColour(bgColour, this.bgColor);
        return this;
    }

    /**
     * @param bgColour
     * @return Class Instance
     */
    public CITToast setbgColourResource(@ColorRes int bgColour) {
        toastType = CUSTOM_TOAST;
        this.bgColor = ActivityCompat.getColor(mContext, bgColour);
        return this;
    }

    /**
     * @param bgColour
     * @return Class Instance
     */
    public CITToast setbgColour(@ColorInt int bgColour) {
        toastType = CUSTOM_TOAST;
        this.bgColor = bgColour;
        return this;
    }

    /**
     * @param messageColor
     * @return Class Instance
     */
    public CITToast setMessageColour(@NonNull String messageColor) {
        toastType = CUSTOM_TOAST;
        this.messageColor = validateColour(messageColor, this.messageColor);
        return this;
    }

    /**
     * @param messageColor
     * @return Class Instance
     */
    public CITToast setMessageColourResource(@ColorRes int messageColor) {
        toastType = CUSTOM_TOAST;
        this.messageColor = ActivityCompat.getColor(mContext, messageColor);
        return this;
    }

    /**
     * @param messageColor
     * @return Class Instance
     */
    public CITToast setMessageColour(@ColorInt int messageColor) {
        toastType = CUSTOM_TOAST;
        this.messageColor = messageColor;
        return this;
    }

    /**
     * @param duration Toast display time duration
     * @return Class Instance
     */
    public CITToast setDuration(@Duration int duration) {
        this.toastDuration = duration;
        return this;
    }

    /**
     * Create Toast view
     */
    public void show() {
        if (toastType == CUSTOM_TOAST) {
            Toast toast = new Toast(mContext);
            View view = getToastLayout();
            toast.setView(view);
            toast.setDuration(toastDuration == LENGTH_SHORT ? Toast.LENGTH_SHORT : Toast.LENGTH_LONG);
            toast.show();
        } else
            Toast.makeText(mContext, message, toastDuration == LENGTH_SHORT
                    ? Toast.LENGTH_SHORT : Toast.LENGTH_LONG).show();
    }

    @SuppressWarnings("deprecation")
    @SuppressLint("NewApi")
    private View getToastLayout() {
        LinearLayout linearLayout = new LinearLayout(mContext);
        linearLayout.setOrientation(LinearLayout.HORIZONTAL);
        LayoutParams imgParam = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        imgParam.gravity = Gravity.CENTER;
        TextView text = new TextView(mContext);
        text.setPadding(45, 20, 45, 20);
        text.setText(message);
        text.setTextSize(15);
        text.setTextColor(messageColor);
        linearLayout.addView(text, imgParam);
        GradientDrawable shape = new GradientDrawable();
        shape.setCornerRadius(50);
        shape.setColor(bgColor);
        try {
            linearLayout.setBackground(shape);
        } catch (Exception e) {
            linearLayout.setBackgroundDrawable(shape);
        } catch (Error e) {
            linearLayout.setBackgroundDrawable(shape);
        }
        return linearLayout;
    }

    /**
     * @param colourCode
     * @param defaultColour
     * @return
     */
    @ColorInt
    private int validateColour(String colourCode, @ColorInt int defaultColour) {
        if (!colourCode.startsWith("#"))
            colourCode = "#" + colourCode;
        int clrLength = colourCode.length();
        return (clrLength == 4 || clrLength == 5 || clrLength == 7 || clrLength == 9) ?
                Color.parseColor(colourCode) : defaultColour;
    }
}
