package com.configureit.utils.alert;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.support.annotation.DrawableRes;
import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.design.widget.TextInputLayout;
import android.support.v7.app.AlertDialog;
import android.text.TextUtils;
import android.util.Log;
import android.util.Patterns;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;


/**
 * @author ConfigureIT Created on 12/4/16.
 */
public class CITAlert {

    private Context mContext;
    private String message;
    private String title;
    private String buttonPositive;
    private String buttonNegative;
    private String buttonNeutral;
    private String reQuestCode;

    private static AlertDialog alertDialog;
    private CITAlertListener listener;
    private int alertType;
    private Dialog dialog;
    private View customView;

    private boolean isValidation = true;
    private boolean isCancelable = false;
    private boolean singleInstance = true;

    private String emptyEmailMessage = "Email should not be empty.";
    private String wrongEmailMessage = "Enter proper Email Address.";
    private String emptyPasswordMessage = "Password should not be empty.";

    @DrawableRes
    private int drawableIcon = 0;

    public static final int ALERT_TYPE_SIMPLE = 0;
    public static final int ALERT_TYPE_INPUT_EMAIL = 1;
    public static final int ALERT_TYPE_INPUT_PASSWORD = 2;
    public static final int ALERT_TYPE_INPUT_EMAIL_PASSWORD = 3;
    public static final int ALERT_TYPE_CUSTOM = 4;

    @IntDef({ALERT_TYPE_SIMPLE, ALERT_TYPE_INPUT_EMAIL, ALERT_TYPE_INPUT_PASSWORD, ALERT_TYPE_INPUT_EMAIL_PASSWORD, ALERT_TYPE_CUSTOM})
    @Retention(RetentionPolicy.SOURCE)
    public @interface AlertType {
    }

    //Protected to prevent out side class Object creation
    protected CITAlert(Context mContext) {
        this.mContext = mContext;
    }

    /**
     * @param context
     * @return
     */
    public static CITAlert on(@NonNull Context context) {
        return new CITAlert(context);
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @param buttonNegative
     * @param buttonNeutral
     * @return
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive,
                          @NonNull String buttonNegative, @NonNull String buttonNeutral) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        this.buttonNegative = buttonNegative;
        this.buttonNeutral = buttonNeutral;
        return this;
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @param buttonNegative
     * @return
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive,
                          @NonNull String buttonNegative) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        this.buttonNegative = buttonNegative;
        return this;
    }

    /**
     * @param title
     * @param message
     * @param buttonPositive
     * @return
     */
    public CITAlert setUp(@NonNull String title, @NonNull String message, @NonNull String buttonPositive) {
        this.title = title;
        this.message = message;
        this.buttonPositive = buttonPositive;
        return this;
    }

    /**
     * @param title
     * @return
     */
    public CITAlert title(@NonNull String title) {
        this.title = title;
        return this;
    }

    /**
     * @param message
     * @return
     */
    public CITAlert message(@NonNull String message) {
        this.message = message;
        return this;
    }

    /**
     * @param buttonPositive
     * @return
     */
    public CITAlert buttonPositive(@NonNull String buttonPositive) {
        this.buttonPositive = buttonPositive;
        return this;
    }

    /**
     * @param buttonNegative
     * @return
     */
    public CITAlert buttonNegative(@NonNull String buttonNegative) {
        this.buttonNegative = buttonNegative;
        return this;
    }

    /**
     * @param buttonNeutral
     * @return
     */
    public CITAlert buttonNeutral(@NonNull String buttonNeutral) {
        this.buttonNeutral = buttonNeutral;
        return this;
    }

    /**
     * @param reQuestCode String Request Code for Differentiate callbacks
     * @param listener    CITAlertListener
     * @return Class Instance
     */
    public CITAlert getCallBack(@NonNull String reQuestCode, @NonNull CITAlertListener listener) {
        this.reQuestCode = reQuestCode;
        this.listener = listener;
        return this;
    }

    /**
     * @param listener CITAlertListener
     * @return Class Instance
     */
    public CITAlert getCallBack(@NonNull CITAlertListener listener) {
        this.listener = listener;
        return this;
    }


    /**
     * @param alertType set CIT Alert type
     * @return Class Instance
     */
    public CITAlert alertType(@AlertType int alertType) {
        this.alertType = alertType;
        return this;
    }

    /**
     * @param view set Custom dialog inflated view object
     * @return Class Instance
     */
    public CITAlert setCustomView(@NonNull View view) {
        this.customView = view;
        return this;
    }

    /**
     * @param emptyInputWarning String message required for set empty input message
     * @return Class Instance
     */
    public CITAlert setEmptyInputWarning(@NonNull String emptyInputWarning) {
        if (TextUtils.isEmpty(emptyInputWarning)) return this;
        this.emptyEmailMessage = emptyInputWarning;
        return this;
    }

    /**
     * @param emptyPasswordWarning String message required for set empty input message
     * @return Class Instance
     */
    public CITAlert setEmptyPasswordWarning(@NonNull String emptyPasswordWarning) {
        if (TextUtils.isEmpty(emptyPasswordWarning)) return this;
        this.emptyPasswordMessage = emptyPasswordWarning;
        return this;
    }

    /**
     * @param wrongInputWarning String message required for set wrong input message
     * @return Class Instance
     */
    public CITAlert setWrongInputWarning(@NonNull String wrongInputWarning) {
        if (TextUtils.isEmpty(wrongInputWarning)) return this;
        this.wrongEmailMessage = wrongInputWarning;
        return this;
    }

    /**
     * It will set the input field Alert validation
     *
     * @param isValidation
     * @return Class Instance
     */
    public CITAlert isInputValidationRequire(boolean isValidation) {
        this.isValidation = isValidation;
        return this;
    }

    /**
     * @param drawableIcon
     * @return
     */
    public CITAlert setIcon(@DrawableRes int drawableIcon) {
        this.drawableIcon = drawableIcon;
        return this;
    }

    /**
     * @param isCancelable
     * @return
     */
    public CITAlert setCancelable(boolean isCancelable) {
        this.isCancelable = isCancelable;
        return this;
    }

    /**
     * @param isSingleInstance
     * @return
     */
    public CITAlert setSingleInstance(boolean isSingleInstance) {
        this.singleInstance = isSingleInstance;
        return this;
    }

    /**
     * create CIT Alert dialog according to the alert type
     *
     * @return Dialog
     */
    public Dialog show() {
        if (singleInstance && alertDialog != null && alertDialog.isShowing()) return null;
        if (alertType == ALERT_TYPE_INPUT_EMAIL || alertType == ALERT_TYPE_INPUT_PASSWORD)
            return createAlertWithInput();
        else if (alertType == ALERT_TYPE_INPUT_EMAIL_PASSWORD)
            return createAlertWithTwoInput();
        else if (alertType == ALERT_TYPE_CUSTOM) {
            if (customView == null)
                throw new NullPointerException("CIT CustomDialog view should not be empty.");
            return showDialogWithView();
        } else
            return createDialog();
    }

    /**
     * This method will create Simple alert dialog.
     *
     * @return
     */
    private AlertDialog createDialog() {
        try {
            AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
            if (!TextUtils.isEmpty(title))
                builder.setTitle(title);
            if (!TextUtils.isEmpty(message))
                builder.setMessage(message);
            if (drawableIcon != 0) {
                builder.setIcon(drawableIcon);
            }
            if (!TextUtils.isEmpty(buttonPositive)) {
                builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        if (listener != null)
                            listener.onPerformClick(reQuestCode, buttonPositive, id, "", "");
                        alertDialog.dismiss();
                    }
                });
            }

            if (!TextUtils.isEmpty(buttonNegative)) {
                builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        if (listener != null)
                            listener.onPerformClick(reQuestCode, buttonNegative, id, "", "");
                        alertDialog.dismiss();
                    }
                });
            }

            if (!TextUtils.isEmpty(buttonNeutral)) {
                builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        if (listener != null)
                            listener.onPerformClick(reQuestCode, buttonNeutral, id, "", "");
                        alertDialog.dismiss();
                    }
                });
            }

            alertDialog = builder.create();
            alertDialog.setCancelable(isCancelable);
            alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                @Override
                public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        if (listener != null)
                            listener.onCancel(reQuestCode);
                        dialog.dismiss();
                    }
                    return false;
                }
            });
            alertDialog.show();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return alertDialog;
    }

    /**
     * This method will create Alert Dialog with email or password Input Box view
     *
     * @return AlertDialog
     */
    private EditText inputBox;
    private TextInputLayout inputLayout;

    private AlertDialog createAlertWithInput() {
        try {
            View inputDialogView;
            if (alertType == ALERT_TYPE_INPUT_EMAIL) {
                inputDialogView = LayoutInflater.from(mContext).inflate(R.layout.dialog_input, null);
                inputBox = (EditText) inputDialogView.findViewById(R.id.inputBoxEmail);
                inputLayout = (TextInputLayout) inputDialogView.findViewById(R.id.inputLayoutEmail);
            } else {
                inputDialogView = LayoutInflater.from(mContext).inflate(R.layout.dialog_input_secure, null);
                inputBox = (EditText) inputDialogView.findViewById(R.id.inputBoxPassword);
                inputLayout = (TextInputLayout) inputDialogView.findViewById(R.id.inputLayoutPassword);
            }

            TextView dialogTitle = (TextView) inputDialogView.findViewById(R.id.dialogTitle);

            if (TextUtils.isEmpty(title))
                dialogTitle.setVisibility(View.GONE);
            else {
                dialogTitle.setVisibility(View.VISIBLE);
                dialogTitle.setText(title);
            }

            AlertDialog.Builder builder = new AlertDialog.Builder(mContext, Window.FEATURE_NO_TITLE);
            builder.setView(inputDialogView);
            if (drawableIcon != 0) {
                builder.setIcon(drawableIcon);
            }
            if (!TextUtils.isEmpty(buttonPositive)) {
                builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                    }
                });
            }
            if (!TextUtils.isEmpty(buttonNegative)) {
                builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        if (listener != null)
                            listener.onPerformClick(reQuestCode, buttonNegative, which, "", "");
                        alertDialog.dismiss();
                    }
                });
            }
            if (!TextUtils.isEmpty(buttonNeutral)) {
                builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                    }
                });
            }

            alertDialog = builder.create();
            alertDialog.setCancelable(isCancelable);
            alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                @Override
                public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        if (listener != null)
                            listener.onCancel(reQuestCode);
                        dialog.dismiss();
                    }
                    return false;
                }
            });

            alertDialog.setOnShowListener(new DialogInterface.OnShowListener() {
                @Override
                public void onShow(final DialogInterface dialog) {
                    Button n = alertDialog.getButton(AlertDialog.BUTTON_NEUTRAL);
                    n.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View view) {
                            inputBox.setText("");
                            inputLayout.setError(null);
                            if (listener != null)
                                listener.onPerformClick(reQuestCode, buttonNeutral, view.getId(), "", "");
                        }
                    });
                    Button p = alertDialog.getButton(AlertDialog.BUTTON_POSITIVE);
                    p.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View view) {
                            if (inputBox != null) {
                                if (!isValidation) {
                                    sendPositiveData(view.getId());
                                    dialog.dismiss();
                                } else if (alertType == ALERT_TYPE_INPUT_PASSWORD) {
                                    if (TextUtils.isEmpty(inputBox.getText())) {
                                        inputLayout.setError(emptyPasswordMessage);
                                    } else {
                                        sendPositiveData(view.getId());
                                        dialog.dismiss();
                                    }
                                } else if (alertType == ALERT_TYPE_INPUT_EMAIL) {
                                    if (TextUtils.isEmpty(inputBox.getText())) {
                                        inputLayout.setError(emptyEmailMessage);
                                    } else if (!Patterns.EMAIL_ADDRESS.matcher(inputBox.getText().toString().trim()).matches()) {
                                        inputLayout.setError(wrongEmailMessage);
                                    } else {
                                        Log.e("Input-->", inputBox.getText().toString());
                                        sendPositiveData(view.getId());
                                        dialog.dismiss();
                                    }
                                }
                            } else dialog.dismiss();
                        }
                    });
                }
            });
            alertDialog.show();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return alertDialog;
    }

    /**
     * This method will create Alert Dialog with email and password Input Box view
     *
     * @return AlertDialog
     */
    private AlertDialog createAlertWithTwoInput() {
        try {
            View inputDialogView = LayoutInflater.from(mContext).inflate(R.layout.dialog_two_input, null);
            final EditText inputBoxEmail = (EditText) inputDialogView.findViewById(R.id.inputBoxEmail);
            final EditText inputBoxPassword = (EditText) inputDialogView.findViewById(R.id.inputBoxPassword);
            final TextInputLayout emailLayout = (TextInputLayout) inputDialogView.findViewById(R.id.inputLayoutEmail);
            final TextInputLayout passwordLayout = (TextInputLayout) inputDialogView.findViewById(R.id.inputLayoutPassword);

            TextView dialogTitle = (TextView) inputDialogView.findViewById(R.id.dialogTitle);

            if (TextUtils.isEmpty(title))
                dialogTitle.setVisibility(View.GONE);
            else {
                dialogTitle.setVisibility(View.VISIBLE);
                dialogTitle.setText(title);
            }

            AlertDialog.Builder builder = new AlertDialog.Builder(mContext, Window.FEATURE_NO_TITLE);
            builder.setView(inputDialogView);
            if (drawableIcon != 0) {
                builder.setIcon(drawableIcon);
            }
            if (!TextUtils.isEmpty(buttonPositive)) {
                builder.setPositiveButton(buttonPositive, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                    }
                });
            }
            if (!TextUtils.isEmpty(buttonNegative)) {
                builder.setNegativeButton(buttonNegative, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        if (listener != null)
                            listener.onPerformClick(reQuestCode, buttonNegative, which, "", "");
                        alertDialog.dismiss();
                    }
                });
            }
            if (!TextUtils.isEmpty(buttonNeutral)) {
                builder.setNeutralButton(buttonNeutral, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                    }
                });
            }

            alertDialog = builder.create();
            alertDialog.setCancelable(isCancelable);
            alertDialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                @Override
                public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        if (listener != null)
                            listener.onCancel(reQuestCode);
                        dialog.dismiss();
                    }
                    return false;
                }
            });

            alertDialog.setOnShowListener(new DialogInterface.OnShowListener() {
                @Override
                public void onShow(final DialogInterface dialog) {
                    Button n = alertDialog.getButton(AlertDialog.BUTTON_NEUTRAL);
                    n.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View view) {
                            inputBoxEmail.setText("");
                            inputBoxPassword.setText("");
                            emailLayout.setError(null);
                            passwordLayout.setError(null);
                            if (listener != null)
                                listener.onPerformClick(reQuestCode, buttonNeutral, view.getId(), "", "");
                        }
                    });

                    Button p = alertDialog.getButton(AlertDialog.BUTTON_POSITIVE);
                    p.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View view) {
                            if (inputBoxEmail != null && inputBoxPassword != null) {
                                if (!isValidation) {
                                    if (listener != null)
                                        listener.onPerformClick(reQuestCode, buttonPositive, view.getId()
                                                , inputBoxEmail.getText().toString()
                                                , inputBoxPassword.getText().toString());
                                    dialog.dismiss();
                                } else {
                                    if (TextUtils.isEmpty(inputBoxEmail.getText())) {
                                        emailLayout.setError(emptyEmailMessage);
                                    } else if (!Patterns.EMAIL_ADDRESS.matcher(inputBoxEmail.getText().toString().trim()).matches()) {
                                        emailLayout.setError(wrongEmailMessage);
                                    } else if (TextUtils.isEmpty(inputBoxPassword.getText())) {
                                        emailLayout.setError(emptyPasswordMessage);
                                    } else {
                                        Log.e("Input-->", inputBoxEmail.getText().toString());
                                        if (listener != null)
                                            listener.onPerformClick(reQuestCode, buttonPositive, view.getId()
                                                    , inputBoxEmail.getText().toString()
                                                    , inputBoxPassword.getText().toString());
                                        dialog.dismiss();
                                    }
                                }
                            } else dialog.dismiss();
                        }
                    });
                }
            });
            alertDialog.show();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return alertDialog;
    }

    private void sendPositiveData(int id) {
        if (listener != null) {
            if (alertType == ALERT_TYPE_INPUT_EMAIL)
                listener.onPerformClick(reQuestCode, buttonPositive, id, inputBox.getText().toString(), "");
            else if (alertType == ALERT_TYPE_INPUT_PASSWORD)
                listener.onPerformClick(reQuestCode, buttonPositive, id, "", inputBox.getText().toString());
        }
    }

    /**
     * This method create dialog with inflated View object.
     *
     * @return Dialog
     */
    public Dialog showDialogWithView() {
        try {
            dialog = new Dialog(mContext);
            dialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
            dialog.getWindow().setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
            dialog.setContentView(customView);
            dialog.setCancelable(isCancelable);
            dialog.setCanceledOnTouchOutside(isCancelable);
            WindowManager.LayoutParams windowLayout = dialog.getWindow().getAttributes();
            windowLayout.gravity = Gravity.CENTER;
            dialog.show();
            dialog.setOnKeyListener(new DialogInterface.OnKeyListener() {
                @Override
                public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        if (listener != null)
                            listener.onCancel(reQuestCode);
                        dialog.dismiss();
                    }
                    return false;
                }
            });
            dialog.show();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return dialog == null ? new Dialog(mContext) : dialog;
    }
}