package com.cogknit.fovea.utils;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;

import com.cogknit.fovea.services.FusedIntentService;
import com.cogknit.fovea.utils.LocationConstants.Intervals;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.GoogleApiClient.ConnectionCallbacks;
import com.google.android.gms.common.api.GoogleApiClient.OnConnectionFailedListener;
import com.google.android.gms.location.ActivityRecognition;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

/**
 * A manager class for abstracting interactions with Google Play Services
 * 
 * @author Chirag
 * 
 */
public class FusedLocationManager implements ConnectionCallbacks,
		OnConnectionFailedListener {

	private static final int FIX_PLAY_SERVICES = 1;
	private boolean connected, connecting;
	private GoogleApiClient mGoogleApiClient;
	private Context context;
	private PendingIntent pIntent;
	private boolean activeLocationRequest, activeActivityRequest;
	private long locationInterval, activityInterval;
	private static final String TAG = "LocationController";

	public FusedLocationManager(Context context) {

		this.context = context;
		mGoogleApiClient = new GoogleApiClient.Builder(context, this, this)
				.addApi(ActivityRecognition.API).addApi(LocationServices.API)
				.addConnectionCallbacks(this)
				.addOnConnectionFailedListener(this).build();

		Intent intent = new Intent(context, FusedIntentService.class);
		pIntent = PendingIntent.getService(context, 0, intent,
				PendingIntent.FLAG_CANCEL_CURRENT);

		connected = false;
		activeLocationRequest = false;
		activeActivityRequest = false;
	}

	/**
	 * Helper method to check if Play Services are available and establish a
	 * connection through the API client
	 */
	private void connect() {

		if (servicesAvailable()) { // Check for Google Play services
			// If a request is not already underway
			if (!connecting) {

				// Indicate that a request is in progress
				connecting = true;

				// Request a connection to Location Services
				mGoogleApiClient.connect();
			}
		}
	}

	/**
	 * Helper method to check if the Google Play Service are available
	 * 
	 * @return true if Google Play Service are available.
	 */
	public boolean servicesAvailable() {

		// Check that Google Play services is available
		int resultCode = GooglePlayServicesUtil
				.isGooglePlayServicesAvailable(context);
//		if( resultCode != ConnectionResult.SUCCESS)
//		{
//		}
		// If Google Play services is available
		if (resultCode == ConnectionResult.SUCCESS) {

			// Continue
			return true;

		} else { // Google Play services is not available!

			FoveaLog.v(TAG,"Google play services are not available, please do the needful!");
			return false;
		}
	}

	/**
	 * Helper method to disconnect from Google Play Services
	 */
	private void disconnect() {

		connecting = false;
		if (connected) {

			mGoogleApiClient.disconnect();
		}
		connected = false;
	}

	/**
	 * Helper method to remove location/activity updates and disconnect from
	 * Google API client
	 */
	public void shutDown() {

		ActivityRecognition.ActivityRecognitionApi.removeActivityUpdates(
				mGoogleApiClient, pIntent);

		LocationServices.FusedLocationApi.removeLocationUpdates(
				mGoogleApiClient, pIntent);

		disconnect();
	}

	@Override
	public void onConnectionFailed(ConnectionResult arg0) {
		connected = false;

	}

	@Override
	public void onConnected(Bundle arg0) {
		connecting = false;
		connected = true;

		if (activeLocationRequest) {
			requestLocationUpdate(locationInterval);
		}

		if (activeActivityRequest) {

			requestActivityUpdate(activityInterval);
		}

	}

	@Override
	public void onConnectionSuspended(int arg0) {
		connected = false;

	}

	/**
	 * Helper method to build a LocationRequest object for a given interval
	 * 
	 * @param interval
	 *            : The desired update interval
	 * @return A LocationRequest object with the given time interval and
	 *         PRIORITY_BALANCED_POWER_ACCURACY
	 */
	private LocationRequest getLocationRequest(long interval) {

		LocationRequest request = LocationRequest.create();
		request.setInterval(interval);
		request.setPriority(LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY);
		return request;
	}

	/**
	 * Helper method to request activity updates at a given interval
	 * 
	 * @param interval
	 *            : Desired interval for activity updates
	 */
	public void requestActivityUpdate(long interval) {
		activityInterval = interval;
		if (connected) {
			if (interval == Intervals.NONE) {
				ActivityRecognition.ActivityRecognitionApi
						.removeActivityUpdates(mGoogleApiClient, pIntent);
			} else {
				ActivityRecognition.ActivityRecognitionApi
						.requestActivityUpdates(mGoogleApiClient,
								activityInterval, pIntent);
			}
			activeActivityRequest = false;
			disconnect();
		} else {
			activeActivityRequest = true;
			connect();

		}
	}

	/**
	 * Helper method to request location updates at a given interval
	 * 
	 * @param interval
	 *            : Desired interval for location updates
	 */
	public void requestLocationUpdate(long interval) {

		locationInterval = interval;
		if (connected) {
			if (interval == Intervals.NONE) {
				LocationServices.FusedLocationApi.removeLocationUpdates(
						mGoogleApiClient, pIntent);
			} else {
				LocationServices.FusedLocationApi
						.requestLocationUpdates(mGoogleApiClient,
								getLocationRequest(interval), pIntent);
			}
			activeLocationRequest = false;
			disconnect();
		} else {
			activeLocationRequest = true;
			connect();

		}
	}

}
